/*
 * Decompiled with CFR 0.152.
 */
package io.deephaven.util.files;

import io.deephaven.hash.KeyedObjectHashMap;
import io.deephaven.hash.KeyedObjectKey;
import java.io.File;
import java.io.IOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import org.apache.commons.io.monitor.FileAlterationListener;
import org.apache.commons.io.monitor.FileAlterationListenerAdaptor;
import org.apache.commons.io.monitor.FileAlterationMonitor;
import org.apache.commons.io.monitor.FileAlterationObserver;
import org.jetbrains.annotations.NotNull;

public class DirWatchService {
    private static final KeyedObjectKey<String, ExactMatchFileWatcher> EXACT_MATCH_KEY = new KeyedObjectKey.Basic<String, ExactMatchFileWatcher>(){

        public String getKey(@NotNull ExactMatchFileWatcher value) {
            return value.tokenToMatch;
        }
    };
    private final List<SeparatorToExactMatchWatchersPair> separatorToExactMatchWatchers = new ArrayList<SeparatorToExactMatchWatchersPair>();
    private final Deque<FilteringFileWatcher> filteringFileWatchers = new ArrayDeque<FilteringFileWatcher>();
    private final Consumer<ExceptionConsumerParameter> exceptionConsumer;
    private final Path keyDir;
    private final WatcherInterface watcherImpl;

    public DirWatchService(@NotNull String dirToWatch, @NotNull Consumer<ExceptionConsumerParameter> exceptionConsumer, @NotNull WatchServiceType watchServiceType, long pollIntervalMillis, WatchEvent.Kind ... kinds) {
        this.exceptionConsumer = exceptionConsumer;
        this.keyDir = Paths.get(dirToWatch, new String[0]);
        switch (watchServiceType) {
            case POLLWATCHSERVICE: {
                this.watcherImpl = new WatcherInterfacePollImpl(pollIntervalMillis, dirToWatch, kinds);
                break;
            }
            case JAVAWATCHSERVICE: {
                this.watcherImpl = new WatcherInterfaceJavaImpl(dirToWatch, kinds);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unknown watch service type: " + watchServiceType);
            }
        }
    }

    private void callExceptionConsumer(Exception e, Boolean watchServiceTerminated) {
        try {
            this.exceptionConsumer.accept(new ExceptionConsumerParameter(e, watchServiceTerminated, this.keyDir));
        }
        catch (Exception e2) {
            throw new RuntimeException(e);
        }
    }

    public synchronized void addExactFileWatch(@NotNull String separator, @NotNull String filePattern, @NotNull BiConsumer<Path, WatchEvent.Kind> consumer) {
        SeparatorToExactMatchWatchersPair pair = this.separatorToExactMatchWatchers.stream().filter(p -> p.fileSeparator.equals(separator)).findFirst().orElseGet(() -> {
            SeparatorToExactMatchWatchersPair p = new SeparatorToExactMatchWatchersPair(separator);
            this.separatorToExactMatchWatchers.add(p);
            return p;
        });
        pair.addWatcher(filePattern, consumer);
    }

    public synchronized void addFileWatchAtEnd(@NotNull Predicate<String> matcher, @NotNull BiConsumer<Path, WatchEvent.Kind> consumer) {
        this.filteringFileWatchers.addLast(new FilteringFileWatcher(consumer, matcher));
    }

    public synchronized void addFileWatchAtStart(@NotNull Predicate<String> matcher, @NotNull BiConsumer<Path, WatchEvent.Kind> consumer) {
        this.filteringFileWatchers.addFirst(new FilteringFileWatcher(consumer, matcher));
    }

    public void stop() throws Exception {
        this.watcherImpl.stop();
    }

    public void start() throws Exception {
        this.watcherImpl.start();
    }

    private synchronized FileWatcher checkExactMatches(String fileName) {
        for (SeparatorToExactMatchWatchersPair pair : this.separatorToExactMatchWatchers) {
            String fileNameBeforeSeparator;
            FileWatcher fileWatcher;
            String fileSeparator = pair.fileSeparator;
            KeyedObjectHashMap<String, ExactMatchFileWatcher> exactMatchWatchers = pair.exactMatchWatchers;
            int fileSeparatorIndex = fileName.indexOf(fileSeparator);
            if (fileSeparatorIndex <= 0 || (fileWatcher = (FileWatcher)exactMatchWatchers.get(fileNameBeforeSeparator = fileName.substring(0, fileSeparatorIndex))) == null) continue;
            return fileWatcher;
        }
        return null;
    }

    private synchronized FileWatcher checkPatternMatches(String fileName) {
        for (FilteringFileWatcher fileWatcher : this.filteringFileWatchers) {
            if (!fileWatcher.matches(fileName)) continue;
            return fileWatcher;
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleFileEvent(String fileName, WatchEvent.Kind kind) {
        FileWatcher foundFileWatcher;
        DirWatchService dirWatchService = this;
        synchronized (dirWatchService) {
            foundFileWatcher = this.checkExactMatches(fileName);
            if (foundFileWatcher == null) {
                foundFileWatcher = this.checkPatternMatches(fileName);
            }
        }
        if (foundFileWatcher != null) {
            Path fullPath = this.keyDir.resolve(fileName);
            foundFileWatcher.consume(fullPath, kind);
        }
    }

    public static Predicate<String> makeRegexMatcher(@NotNull String regex) {
        Pattern pattern = Pattern.compile(regex);
        return fileName -> pattern.matcher((CharSequence)fileName).matches();
    }

    public static Predicate<String> makeStartsWithMatcher(@NotNull String prefix) {
        return fileName -> fileName.startsWith(prefix);
    }

    public static Predicate<String> makeEndsWithMatcher(@NotNull String suffix) {
        return fileName -> fileName.endsWith(suffix);
    }

    private class WatcherInterfacePollImpl
    implements WatcherInterface {
        private final FileAlterationMonitor fileAlterationMonitor;
        private boolean fileAlterationMonitorStarted;

        private WatcherInterfacePollImpl(long pollIntervalMillis, String dirToWatch, WatchEvent.Kind ... kinds) {
            FileAlterationObserver fileAlterationObserver = new FileAlterationObserver(new File(dirToWatch));
            for (WatchEvent.Kind kind : kinds) {
                if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
                    fileAlterationObserver.addListener((FileAlterationListener)new FileAlterationListenerAdaptor(){

                        public void onFileCreate(File file) {
                            WatcherInterfacePollImpl.this.handlePollFileEvent(file, StandardWatchEventKinds.ENTRY_CREATE);
                        }
                    });
                    continue;
                }
                if (kind == StandardWatchEventKinds.ENTRY_MODIFY) {
                    fileAlterationObserver.addListener((FileAlterationListener)new FileAlterationListenerAdaptor(){

                        public void onFileChange(File file) {
                            WatcherInterfacePollImpl.this.handlePollFileEvent(file, StandardWatchEventKinds.ENTRY_MODIFY);
                        }
                    });
                    continue;
                }
                if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
                    fileAlterationObserver.addListener((FileAlterationListener)new FileAlterationListenerAdaptor(){

                        public void onFileDelete(File file) {
                            WatcherInterfacePollImpl.this.handlePollFileEvent(file, StandardWatchEventKinds.ENTRY_DELETE);
                        }
                    });
                    continue;
                }
                throw new IllegalArgumentException("DirWatchService passed unsupported watch event kind: " + kind.name());
            }
            this.fileAlterationMonitor = new FileAlterationMonitor(pollIntervalMillis, new FileAlterationObserver[]{fileAlterationObserver});
            this.fileAlterationMonitorStarted = false;
        }

        @Override
        public synchronized void start() throws Exception {
            if (this.fileAlterationMonitorStarted) {
                throw new IllegalStateException("Trying to start an already-started DirWatchService!");
            }
            this.fileAlterationMonitor.start();
            this.fileAlterationMonitorStarted = true;
        }

        @Override
        public synchronized void stop() throws Exception {
            if (this.fileAlterationMonitorStarted) {
                this.fileAlterationMonitor.stop();
                this.fileAlterationMonitorStarted = false;
            }
        }

        private void handlePollFileEvent(File file, WatchEvent.Kind kind) {
            String fileName = file.getName();
            DirWatchService.this.handleFileEvent(fileName, kind);
        }
    }

    private class WatcherInterfaceJavaImpl
    implements WatcherInterface {
        private volatile Thread watcherThread;
        private volatile boolean stopRequested;
        private volatile WatchService watcher;
        private final String dirToWatch;
        private final Path dir;
        private final WatchEvent.Kind[] watchEventKinds;

        private WatcherInterfaceJavaImpl(String dirToWatch, WatchEvent.Kind ... kinds) {
            this.dirToWatch = dirToWatch;
            this.dir = Paths.get(dirToWatch, new String[0]);
            this.stopRequested = false;
            this.watchEventKinds = kinds;
        }

        @Override
        public synchronized void start() throws IOException {
            if (this.watcherThread != null) {
                throw new IllegalStateException("Trying to start a running DirWatchService!");
            }
            this.watcher = FileSystems.getDefault().newWatchService();
            this.dir.register(this.watcher, this.watchEventKinds);
            this.watcherThread = new Thread(this::runJavaFileWatcher, "DirWatchService-" + this.dirToWatch);
            this.watcherThread.setDaemon(true);
            this.watcherThread.start();
        }

        @Override
        public synchronized void stop() throws IOException {
            if (this.watcherThread == null) {
                throw new IllegalStateException("Trying to stop an already-stopped DirWatchService!");
            }
            this.stopRequested = true;
            Thread t = this.watcherThread;
            if (t != null) {
                t.interrupt();
                while (t.isAlive()) {
                    try {
                        t.join();
                    }
                    catch (InterruptedException interruptedException) {}
                }
            }
            this.watcher.close();
            this.watcher = null;
            this.watcherThread = null;
            this.stopRequested = false;
        }

        private void runJavaFileWatcher() {
            while (!this.stopRequested) {
                WatchKey watchKey;
                try {
                    watchKey = this.watcher.take();
                }
                catch (InterruptedException e) {
                    if (this.stopRequested) continue;
                    DirWatchService.this.callExceptionConsumer(e, false);
                    continue;
                }
                catch (ClosedWatchServiceException e) {
                    DirWatchService.this.callExceptionConsumer(e, true);
                    return;
                }
                for (WatchEvent<?> event : watchKey.pollEvents()) {
                    WatchEvent.Kind<?> kind = event.kind();
                    if (kind == StandardWatchEventKinds.OVERFLOW) {
                        DirWatchService.this.callExceptionConsumer(new RuntimeException("Overflow event received in DirWatchService, file events possibly lost"), true);
                        return;
                    }
                    WatchEvent<?> watchEvent = event;
                    Path filePath = (Path)watchEvent.context();
                    String fileName = filePath.toString();
                    DirWatchService.this.handleFileEvent(fileName, kind);
                }
                if (watchKey.reset()) continue;
                DirWatchService.this.callExceptionConsumer(new RuntimeException("Error resetting watch key in directory watch service"), true);
                return;
            }
        }
    }

    private static interface WatcherInterface {
        public void start() throws Exception;

        public void stop() throws Exception;
    }

    private class FilteringFileWatcher
    extends FileWatcher {
        private final Predicate<String> matcher;

        private FilteringFileWatcher(@NotNull BiConsumer<Path, WatchEvent.Kind> consumer, Predicate<String> matcher) {
            super(consumer);
            this.matcher = matcher;
        }

        private boolean matches(@NotNull String fileName) {
            return this.matcher.test(fileName);
        }
    }

    private class SeparatorToExactMatchWatchersPair {
        private final String fileSeparator;
        private final KeyedObjectHashMap<String, ExactMatchFileWatcher> exactMatchWatchers;

        private SeparatorToExactMatchWatchersPair(String fileSeparator) {
            this.fileSeparator = fileSeparator;
            this.exactMatchWatchers = new KeyedObjectHashMap(EXACT_MATCH_KEY);
        }

        private void addWatcher(@NotNull String filePattern, @NotNull BiConsumer<Path, WatchEvent.Kind> consumer) {
            this.exactMatchWatchers.compute((Object)filePattern, (k, oldFileWatcher) -> {
                if (oldFileWatcher != null) {
                    oldFileWatcher.addConsumer(consumer);
                    return oldFileWatcher;
                }
                return new ExactMatchFileWatcher(consumer, filePattern);
            });
        }
    }

    private class ExactMatchFileWatcher
    extends FileWatcher {
        private final String tokenToMatch;

        private ExactMatchFileWatcher(@NotNull BiConsumer<Path, WatchEvent.Kind> consumer, String tokenToMatch) {
            super(consumer);
            this.tokenToMatch = tokenToMatch;
        }
    }

    private class FileWatcher {
        private final List<BiConsumer<Path, WatchEvent.Kind>> consumers = new LinkedList<BiConsumer<Path, WatchEvent.Kind>>();

        FileWatcher(BiConsumer<Path, WatchEvent.Kind> consumer) {
            this.addConsumer(consumer);
        }

        void addConsumer(@NotNull BiConsumer<Path, WatchEvent.Kind> consumer) {
            this.consumers.add(consumer);
        }

        void consume(Path p, WatchEvent.Kind k) {
            for (BiConsumer<Path, WatchEvent.Kind> consumer : this.consumers) {
                try {
                    consumer.accept(p, k);
                }
                catch (Exception e) {
                    DirWatchService.this.callExceptionConsumer(e, false);
                }
            }
        }
    }

    public static class ExceptionConsumerParameter {
        private final Exception exception;
        private final boolean watchServiceTerminated;
        private final Path path;

        private ExceptionConsumerParameter(Exception exception, boolean watchServiceTerminated, Path path) {
            this.exception = exception;
            this.watchServiceTerminated = watchServiceTerminated;
            this.path = path;
        }

        public Exception getException() {
            return this.exception;
        }

        public boolean watchServiceTerminated() {
            return this.watchServiceTerminated;
        }

        public Path getPath() {
            return this.path;
        }
    }

    public static enum WatchServiceType {
        JAVAWATCHSERVICE("JavaWatchService"),
        POLLWATCHSERVICE("PollWatchService");

        private final String name;
        private static final String[] names;

        private WatchServiceType(String name) {
            this.name = name;
        }

        public String getName() {
            return this.name;
        }

        public static String[] getNames() {
            return names;
        }

        static {
            WatchServiceType[] serviceTypes = WatchServiceType.values();
            names = new String[serviceTypes.length];
            int i = 0;
            for (WatchServiceType serviceType : serviceTypes) {
                WatchServiceType.names[i++] = serviceType.getName();
            }
        }
    }
}

