/*
 * Decompiled with CFR 0.152.
 */
package io.deephaven.csv.reading;

import io.deephaven.csv.containers.ByteSlice;
import io.deephaven.csv.containers.GrowableByteBuffer;
import io.deephaven.csv.tokenization.RangeTests;
import io.deephaven.csv.util.CsvReaderException;
import io.deephaven.csv.util.MutableBoolean;
import java.io.IOException;
import java.io.InputStream;

public final class CellGrabber {
    public static final int BUFFER_SIZE = 65536;
    private final InputStream inputStream;
    private final byte quoteChar;
    private final byte fieldDelimiter;
    private final boolean ignoreSurroundingSpaces;
    private final boolean trim;
    private final byte[] buffer;
    private int size;
    private int offset;
    private int startOffset;
    private final GrowableByteBuffer spillBuffer;
    private int physicalRowNum;

    public CellGrabber(InputStream inputStream, byte quoteChar, byte fieldDelimiter, boolean ignoreSurroundingSpaces, boolean trim) {
        this.inputStream = inputStream;
        this.quoteChar = quoteChar;
        this.fieldDelimiter = fieldDelimiter;
        this.ignoreSurroundingSpaces = ignoreSurroundingSpaces;
        this.trim = trim;
        this.buffer = new byte[65536];
        this.size = 0;
        this.offset = 0;
        this.startOffset = 0;
        this.spillBuffer = new GrowableByteBuffer();
        this.physicalRowNum = 0;
    }

    public void grabNext(ByteSlice dest, MutableBoolean lastInRow, MutableBoolean endOfInput) throws CsvReaderException {
        this.spillBuffer.clear();
        this.startOffset = this.offset;
        if (this.ignoreSurroundingSpaces) {
            this.skipWhitespace();
        }
        if (this.tryEnsureMore() && this.buffer[this.offset] == this.quoteChar) {
            ++this.offset;
            this.processQuotedMode(dest, lastInRow, endOfInput);
            if (this.trim) {
                CellGrabber.trimWhitespace(dest);
            }
        } else {
            this.processUnquotedMode(dest, lastInRow, endOfInput);
            if (this.ignoreSurroundingSpaces) {
                CellGrabber.trimWhitespace(dest);
            }
        }
    }

    private void processQuotedMode(ByteSlice dest, MutableBoolean lastInRow, MutableBoolean endOfInput) throws CsvReaderException {
        this.startOffset = this.offset;
        boolean prevCharWasCarriageReturn = false;
        while (true) {
            byte peek;
            byte ch;
            if (this.offset == this.size && !this.tryEnsureMore()) {
                throw new CsvReaderException("Cell did not have closing quote character");
            }
            if ((ch = this.buffer[this.offset++]) == 13) {
                ++this.physicalRowNum;
                prevCharWasCarriageReturn = true;
            } else {
                if (ch == 10 && !prevCharWasCarriageReturn) {
                    ++this.physicalRowNum;
                }
                prevCharWasCarriageReturn = false;
            }
            if (ch != this.quoteChar) continue;
            if (!this.tryEnsureMore() || (peek = this.buffer[this.offset]) != this.quoteChar) break;
            this.spillRange();
            this.startOffset = ++this.offset;
        }
        this.finishField(dest, lastInRow, endOfInput);
        while (dest.begin() != dest.end() && RangeTests.isSpaceOrTab(dest.back())) {
            dest.setEnd(dest.end() - 1);
        }
        if (dest.begin() == dest.end() || dest.back() != this.quoteChar) {
            throw new RuntimeException("Logic error: final non-whitespace in field is not quoteChar");
        }
        dest.setEnd(dest.end() - 1);
    }

    private void processUnquotedMode(ByteSlice dest, MutableBoolean lastInRow, MutableBoolean endOfInput) throws CsvReaderException {
        this.startOffset = this.offset;
        this.finishField(dest, lastInRow, endOfInput);
    }

    private void skipWhitespace() throws CsvReaderException {
        while (this.offset != this.size || this.tryEnsureMore()) {
            byte ch = this.buffer[this.offset];
            if (ch == this.fieldDelimiter || !RangeTests.isSpaceOrTab(ch)) {
                return;
            }
            ++this.offset;
        }
        return;
    }

    private void finishField(ByteSlice dest, MutableBoolean lastInRow, MutableBoolean endOfInput) throws CsvReaderException {
        while (true) {
            if (!this.tryEnsureMore()) {
                this.finish(dest);
                lastInRow.setValue(true);
                endOfInput.setValue(true);
                return;
            }
            byte ch = this.buffer[this.offset];
            if (ch == this.fieldDelimiter) {
                this.finish(dest);
                ++this.offset;
                lastInRow.setValue(false);
                endOfInput.setValue(false);
                return;
            }
            if (ch == 10) {
                this.finish(dest);
                ++this.offset;
                lastInRow.setValue(true);
                endOfInput.setValue(false);
                ++this.physicalRowNum;
                return;
            }
            if (ch == 13) {
                ++this.offset;
                int excess = 1;
                if (this.tryEnsureMore() && this.buffer[this.offset] == 10) {
                    ++this.offset;
                    excess = 2;
                }
                this.finish(dest);
                dest.reset(dest.data(), dest.begin(), dest.end() - excess);
                lastInRow.setValue(true);
                endOfInput.setValue(false);
                ++this.physicalRowNum;
                return;
            }
            ++this.offset;
        }
    }

    private boolean tryEnsureMore() throws CsvReaderException {
        if (this.offset != this.size) {
            return true;
        }
        this.spillRange();
        this.refillBuffer();
        return this.size != 0;
    }

    private void spillRange() {
        this.spillBuffer.append(this.buffer, this.startOffset, this.offset - this.startOffset);
        this.startOffset = this.offset;
    }

    private void refillBuffer() throws CsvReaderException {
        this.offset = 0;
        this.startOffset = 0;
        try {
            int bytesRead = this.inputStream.read(this.buffer, 0, this.buffer.length);
            if (bytesRead < 0) {
                this.size = 0;
                return;
            }
            if (bytesRead > 0) {
                this.size = bytesRead;
                return;
            }
            throw new CsvReaderException("Logic error: zero-length read");
        }
        catch (IOException inner) {
            throw new CsvReaderException("Caught exception", inner);
        }
    }

    private void finish(ByteSlice dest) {
        if (this.spillBuffer.size() == 0) {
            dest.reset(this.buffer, this.startOffset, this.offset);
            return;
        }
        this.spillRange();
        dest.reset(this.spillBuffer.data(), 0, this.spillBuffer.size());
    }

    public int physicalRowNum() {
        return this.physicalRowNum;
    }

    private static void trimWhitespace(ByteSlice cs) {
        int begin;
        byte[] data = cs.data();
        int end = cs.end();
        for (begin = cs.begin(); begin != end && RangeTests.isSpaceOrTab(data[begin]); ++begin) {
        }
        while (begin != end && RangeTests.isSpaceOrTab(data[end - 1])) {
            --end;
        }
        cs.reset(data, begin, end);
    }
}

