/*
 * Decompiled with CFR 0.152.
 */
package io.deephaven.time.calendar;

import io.deephaven.base.verify.Require;
import io.deephaven.time.DateTimeUtils;
import io.deephaven.time.TimeZoneAliases;
import io.deephaven.time.calendar.BusinessCalendar;
import io.deephaven.time.calendar.CalendarDay;
import io.deephaven.time.calendar.Calendars;
import io.deephaven.time.calendar.TimeRange;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneId;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import org.jetbrains.annotations.NotNull;

public final class BusinessCalendarXMLParser {
    public static BusinessCalendar loadBusinessCalendar(@NotNull String file) {
        Require.neqNull((Object)file, (String)"file");
        return BusinessCalendarXMLParser.loadBusinessCalendar(new File(file));
    }

    public static BusinessCalendar loadBusinessCalendar(@NotNull File file) {
        Require.neqNull((Object)file, (String)"file");
        BusinessCalendarInputs in = BusinessCalendarXMLParser.parseBusinessCalendarInputs(file);
        return new BusinessCalendar(in.calendarName, in.description, in.timeZone, in.firstValidDate, in.lastValidDate, in.standardBusinessDay, in.weekendDays, in.holidays);
    }

    public static BusinessCalendar loadBusinessCalendar(@NotNull InputStream inputStream) {
        Require.neqNull((Object)inputStream, (String)"inputStream");
        BusinessCalendarInputs in = BusinessCalendarXMLParser.parseBusinessCalendarInputs(inputStream);
        return new BusinessCalendar(in.calendarName, in.description, in.timeZone, in.firstValidDate, in.lastValidDate, in.standardBusinessDay, in.weekendDays, in.holidays);
    }

    public static BusinessCalendar loadBusinessCalendarFromResource(String resource) throws IOException {
        InputStream in = Calendars.class.getResourceAsStream(resource);
        if (in == null) {
            throw new RuntimeException("Could not open resource " + resource + " from classpath");
        }
        try (BufferedInputStream bin = new BufferedInputStream(in);){
            BusinessCalendar businessCalendar = BusinessCalendarXMLParser.loadBusinessCalendar(bin);
            return businessCalendar;
        }
    }

    private static BusinessCalendarInputs parseBusinessCalendarInputs(@NotNull File file) {
        Require.neqNull((Object)file, (String)"file");
        try {
            return BusinessCalendarXMLParser.fill(BusinessCalendarXMLParser.loadXMLRootElement(file));
        }
        catch (Exception e) {
            throw new RuntimeException("Unable to load calendar file: file=" + file.getPath(), e);
        }
    }

    private static BusinessCalendarInputs parseBusinessCalendarInputs(@NotNull InputStream in) {
        Require.neqNull((Object)in, (String)"in");
        try {
            return BusinessCalendarXMLParser.fill(BusinessCalendarXMLParser.loadXMLRootElement(in));
        }
        catch (Exception e) {
            throw new RuntimeException("Unable to load calendar file: inputStream=" + in, e);
        }
    }

    private static BusinessCalendarInputs fill(Element root) throws Exception {
        BusinessCalendarInputs calendarElements = new BusinessCalendarInputs();
        calendarElements.calendarName = BusinessCalendarXMLParser.getText(BusinessCalendarXMLParser.getRequiredChild(root, "name"));
        calendarElements.timeZone = TimeZoneAliases.zoneId(BusinessCalendarXMLParser.getText(BusinessCalendarXMLParser.getRequiredChild(root, "timeZone")));
        calendarElements.description = BusinessCalendarXMLParser.getText(root.getChild("description"));
        calendarElements.holidays = BusinessCalendarXMLParser.parseHolidays(root, calendarElements.timeZone);
        String firstValidDateStr = BusinessCalendarXMLParser.getText(root.getChild("firstValidDate"));
        calendarElements.firstValidDate = firstValidDateStr == null ? Collections.min(calendarElements.holidays.keySet()) : DateTimeUtils.parseLocalDate(firstValidDateStr);
        String lastValidDateStr = BusinessCalendarXMLParser.getText(root.getChild("lastValidDate"));
        calendarElements.lastValidDate = lastValidDateStr == null ? Collections.max(calendarElements.holidays.keySet()) : DateTimeUtils.parseLocalDate(lastValidDateStr);
        Element defaultElement = BusinessCalendarXMLParser.getRequiredChild(root, "default");
        calendarElements.weekendDays = BusinessCalendarXMLParser.parseWeekendDays(defaultElement);
        calendarElements.standardBusinessDay = BusinessCalendarXMLParser.parseCalendarDaySchedule(defaultElement);
        return calendarElements;
    }

    private static Element loadXMLRootElement(File calendarFile) throws Exception {
        Document doc;
        try {
            SAXBuilder builder = new SAXBuilder();
            doc = builder.build(calendarFile);
        }
        catch (JDOMException e) {
            throw new Exception("Error parsing business calendar: file=" + calendarFile, e);
        }
        catch (IOException e) {
            throw new Exception("Error loading business calendar: file=" + calendarFile, e);
        }
        return doc.getRootElement();
    }

    private static Element loadXMLRootElement(InputStream in) throws Exception {
        Document doc;
        try {
            SAXBuilder builder = new SAXBuilder();
            doc = builder.build(in);
        }
        catch (JDOMException e) {
            throw new Exception("Error parsing business calendar: inputStream=" + in, e);
        }
        catch (IOException e) {
            throw new Exception("Error loading business calendar: inputStream=" + in, e);
        }
        return doc.getRootElement();
    }

    private static Element getRequiredChild(@NotNull Element root, String child) throws Exception {
        Element element = root.getChild(child);
        if (element != null) {
            return element;
        }
        throw new Exception("Missing the '" + child + "' tag in calendar file: text=" + root.getTextTrim());
    }

    private static String getText(Element element) {
        return element == null ? null : element.getTextTrim();
    }

    private static CalendarDay<LocalTime> parseCalendarDaySchedule(Element element) throws Exception {
        List businessTimes = element.getChildren("businessTime");
        List businessPeriods = element.getChildren("businessPeriod");
        if (businessTimes.isEmpty() && businessPeriods.isEmpty()) {
            return CalendarDay.HOLIDAY;
        }
        if (!businessTimes.isEmpty() && businessPeriods.isEmpty()) {
            return new CalendarDay<LocalTime>(BusinessCalendarXMLParser.parseBusinessRanges(businessTimes));
        }
        if (businessTimes.isEmpty() && !businessPeriods.isEmpty()) {
            return new CalendarDay<LocalTime>(BusinessCalendarXMLParser.parseBusinessRangesLegacy(businessPeriods));
        }
        throw new Exception("Cannot have both 'businessTime' and 'businessPeriod' tags in the same element: text=" + element.getTextTrim());
    }

    private static TimeRange<LocalTime>[] parseBusinessRanges(List<Element> businessRanges) throws Exception {
        TimeRange[] rst = new TimeRange[businessRanges.size()];
        for (int i = 0; i < businessRanges.size(); ++i) {
            String openTxt = BusinessCalendarXMLParser.getText(BusinessCalendarXMLParser.getRequiredChild(businessRanges.get(i), "open"));
            String closeTxt = BusinessCalendarXMLParser.getText(BusinessCalendarXMLParser.getRequiredChild(businessRanges.get(i), "close"));
            if (closeTxt.startsWith("24:00")) {
                throw new RuntimeException("Close time (" + closeTxt + ") is on the next day.  '23:59:59.999999999' is the maximum close time.");
            }
            LocalTime open = DateTimeUtils.parseLocalTime(openTxt);
            LocalTime close = DateTimeUtils.parseLocalTime(closeTxt);
            rst[i] = new TimeRange<LocalTime>(open, close, true);
        }
        return rst;
    }

    private static TimeRange<LocalTime>[] parseBusinessRangesLegacy(List<Element> businessRanges) throws Exception {
        TimeRange[] rst = new TimeRange[businessRanges.size()];
        int i = 0;
        for (Element br : businessRanges) {
            String[] openClose = br.getTextTrim().split(",");
            if (openClose.length != 2) {
                throw new IllegalArgumentException("Can not parse business periods; open/close = " + br.getText());
            }
            String openTxt = openClose[0];
            String closeTxt = openClose[1];
            LocalTime open = DateTimeUtils.parseLocalTime(openTxt);
            LocalTime close = DateTimeUtils.parseLocalTime(closeTxt);
            rst[i] = new TimeRange<LocalTime>(open, close, true);
            ++i;
        }
        return rst;
    }

    private static Map<LocalDate, CalendarDay<Instant>> parseHolidays(Element root, ZoneId timeZone) throws Exception {
        ConcurrentHashMap<LocalDate, CalendarDay<Instant>> holidays = new ConcurrentHashMap<LocalDate, CalendarDay<Instant>>();
        List holidayElements = root.getChildren("holiday");
        for (Element holidayElement : holidayElements) {
            Element dateElement = BusinessCalendarXMLParser.getRequiredChild(holidayElement, "date");
            Object dateStr = BusinessCalendarXMLParser.getText(dateElement);
            if (((String)dateStr).length() == 8) {
                dateStr = ((String)dateStr).substring(0, 4) + "-" + ((String)dateStr).substring(4, 6) + "-" + ((String)dateStr).substring(6, 8);
            }
            LocalDate date = DateTimeUtils.parseLocalDate((String)dateStr);
            CalendarDay<LocalTime> schedule = BusinessCalendarXMLParser.parseCalendarDaySchedule(holidayElement);
            holidays.put(date, CalendarDay.toInstant(schedule, date, timeZone));
        }
        return holidays;
    }

    private static Set<DayOfWeek> parseWeekendDays(@NotNull Element defaultElement) throws Exception {
        HashSet<DayOfWeek> weekendDays = new HashSet<DayOfWeek>();
        List weekends = defaultElement.getChildren("weekend");
        if (weekends != null) {
            for (Element weekendElement : weekends) {
                DayOfWeek dow;
                String weekend = BusinessCalendarXMLParser.getText(weekendElement);
                String dows = weekend.trim().toUpperCase();
                try {
                    dow = DayOfWeek.valueOf(dows);
                }
                catch (IllegalArgumentException e) {
                    throw new Exception("Invalid day of week: day=" + dows, e);
                }
                weekendDays.add(dow);
            }
        }
        return weekendDays;
    }

    private static class BusinessCalendarInputs {
        private String calendarName;
        private String description;
        private ZoneId timeZone;
        private LocalDate firstValidDate;
        private LocalDate lastValidDate;
        private CalendarDay<LocalTime> standardBusinessDay;
        private Set<DayOfWeek> weekendDays;
        private Map<LocalDate, CalendarDay<Instant>> holidays;

        private BusinessCalendarInputs() {
        }
    }
}

