/*
 * Decompiled with CFR 0.152.
 */
package io.druid.indexing.kafka.supervisor;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.MoreExecutors;
import com.metamx.common.ISE;
import com.metamx.emitter.EmittingLogger;
import io.druid.concurrent.Execs;
import io.druid.indexing.common.TaskInfoProvider;
import io.druid.indexing.common.TaskLocation;
import io.druid.indexing.common.TaskStatus;
import io.druid.indexing.common.task.Task;
import io.druid.indexing.common.task.TaskResource;
import io.druid.indexing.kafka.KafkaDataSourceMetadata;
import io.druid.indexing.kafka.KafkaIOConfig;
import io.druid.indexing.kafka.KafkaIndexTask;
import io.druid.indexing.kafka.KafkaIndexTaskClient;
import io.druid.indexing.kafka.KafkaIndexTaskClientFactory;
import io.druid.indexing.kafka.KafkaPartitions;
import io.druid.indexing.kafka.supervisor.KafkaSupervisorIOConfig;
import io.druid.indexing.kafka.supervisor.KafkaSupervisorReport;
import io.druid.indexing.kafka.supervisor.KafkaSupervisorSpec;
import io.druid.indexing.overlord.DataSourceMetadata;
import io.druid.indexing.overlord.IndexerMetadataStorageCoordinator;
import io.druid.indexing.overlord.TaskMaster;
import io.druid.indexing.overlord.TaskQueue;
import io.druid.indexing.overlord.TaskRunner;
import io.druid.indexing.overlord.TaskRunnerListener;
import io.druid.indexing.overlord.TaskRunnerWorkItem;
import io.druid.indexing.overlord.TaskStorage;
import io.druid.indexing.overlord.supervisor.Supervisor;
import io.druid.indexing.overlord.supervisor.SupervisorReport;
import io.druid.metadata.EntryExistsException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.serialization.Deserializer;
import org.joda.time.DateTime;
import org.joda.time.ReadableDuration;
import org.joda.time.ReadableInstant;

public class KafkaSupervisor
implements Supervisor {
    private static final EmittingLogger log = new EmittingLogger(KafkaSupervisor.class);
    private static final Random RANDOM = new Random();
    private static final long MAX_RUN_FREQUENCY_MILLIS = 1000L;
    private static final int SHUTDOWN_TIMEOUT_MILLIS = 30000;
    private final HashMap<Integer, TaskGroup> taskGroups = new HashMap();
    private final HashMap<Integer, List<TaskGroup>> pendingCompletionTaskGroups = new HashMap();
    private Map<Integer, Map<Integer, Long>> partitionGroups = new HashMap<Integer, Map<Integer, Long>>();
    private final TaskStorage taskStorage;
    private final TaskMaster taskMaster;
    private final IndexerMetadataStorageCoordinator indexerMetadataStorageCoordinator;
    private final KafkaIndexTaskClient taskClient;
    private final ObjectMapper sortingMapper;
    private final KafkaSupervisorSpec spec;
    private final String dataSource;
    private final KafkaSupervisorIOConfig ioConfig;
    private final String supervisorId;
    private final TaskInfoProvider taskInfoProvider;
    private final ExecutorService exec;
    private final ScheduledExecutorService scheduledExec;
    private final BlockingQueue<Notice> notices = new LinkedBlockingDeque<Notice>();
    private final Object stopLock = new Object();
    private final Object stateChangeLock = new Object();
    private boolean listenerRegistered = false;
    private long lastRunTime;
    private volatile DateTime firstRunTime;
    private volatile KafkaConsumer consumer;
    private volatile boolean started = false;
    private volatile boolean stopped = false;

    public KafkaSupervisor(final TaskStorage taskStorage, final TaskMaster taskMaster, IndexerMetadataStorageCoordinator indexerMetadataStorageCoordinator, KafkaIndexTaskClientFactory taskClientFactory, ObjectMapper mapper, KafkaSupervisorSpec spec) {
        this.taskStorage = taskStorage;
        this.taskMaster = taskMaster;
        this.indexerMetadataStorageCoordinator = indexerMetadataStorageCoordinator;
        this.sortingMapper = mapper.copy().configure(MapperFeature.SORT_PROPERTIES_ALPHABETICALLY, true);
        this.spec = spec;
        this.dataSource = spec.getDataSchema().getDataSource();
        this.ioConfig = spec.getIoConfig();
        this.supervisorId = String.format("KafkaSupervisor-%s", this.dataSource);
        this.exec = Execs.singleThreaded((String)(this.supervisorId + "-%d"));
        this.scheduledExec = Execs.scheduledSingleThreaded((String)(this.supervisorId + "-Scheduler-%d"));
        this.taskInfoProvider = new TaskInfoProvider(){

            public TaskLocation getTaskLocation(final String id) {
                Preconditions.checkNotNull((Object)id, (Object)"id");
                Optional taskRunner = taskMaster.getTaskRunner();
                if (taskRunner.isPresent()) {
                    Optional item = Iterables.tryFind((Iterable)((TaskRunner)taskRunner.get()).getRunningTasks(), (Predicate)new Predicate<TaskRunnerWorkItem>(){

                        public boolean apply(TaskRunnerWorkItem taskRunnerWorkItem) {
                            return id.equals(taskRunnerWorkItem.getTaskId());
                        }
                    });
                    if (item.isPresent()) {
                        return ((TaskRunnerWorkItem)item.get()).getLocation();
                    }
                } else {
                    log.error("Failed to get task runner because I'm not the leader!", new Object[0]);
                }
                return TaskLocation.unknown();
            }

            public Optional<TaskStatus> getTaskStatus(String id) {
                return taskStorage.getStatus(id);
            }
        };
        this.taskClient = taskClientFactory.build(this.taskInfoProvider);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void start() {
        Object object = this.stateChangeLock;
        synchronized (object) {
            Preconditions.checkState((!this.started ? 1 : 0) != 0, (Object)"already started");
            Preconditions.checkState((!this.exec.isShutdown() ? 1 : 0) != 0, (Object)"already stopped");
            try {
                this.consumer = this.getKafkaConsumer();
                this.exec.submit(new Runnable(){

                    @Override
                    public void run() {
                        block4: while (true) {
                            try {
                                while (!Thread.currentThread().isInterrupted()) {
                                    Notice notice = (Notice)KafkaSupervisor.this.notices.take();
                                    try {
                                        notice.handle();
                                        continue block4;
                                    }
                                    catch (Exception e) {
                                        log.makeAlert((Throwable)e, "KafkaSupervisor[%s] failed to handle notice", new Object[]{KafkaSupervisor.this.dataSource}).addData("noticeClass", (Object)notice.getClass().getSimpleName()).emit();
                                    }
                                }
                                break;
                            }
                            catch (InterruptedException e) {
                                log.info("KafkaSupervisor[%s] interrupted, exiting", new Object[]{KafkaSupervisor.this.dataSource});
                                break;
                            }
                        }
                    }
                });
            }
            catch (Exception e) {
                log.makeAlert((Throwable)e, "Exception starting KafkaSupervisor[%s]", new Object[]{this.dataSource}).emit();
                throw Throwables.propagate((Throwable)e);
            }
            this.firstRunTime = DateTime.now().plus((ReadableDuration)this.ioConfig.getStartDelay());
            this.scheduledExec.scheduleAtFixedRate(this.buildRunTask(), this.ioConfig.getStartDelay().getMillis(), Math.max(this.ioConfig.getPeriod().getMillis(), 1000L), TimeUnit.MILLISECONDS);
            this.started = true;
            log.info("Started KafkaSupervisor[%s], first run in [%s]", new Object[]{this.dataSource, this.ioConfig.getStartDelay()});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop(boolean stopGracefully) {
        Object object = this.stateChangeLock;
        synchronized (object) {
            Preconditions.checkState((boolean)this.started, (Object)"not started");
            log.info("Beginning shutdown of KafkaSupervisor[%s]", new Object[]{this.dataSource});
            try {
                this.scheduledExec.shutdownNow();
                Optional taskRunner = this.taskMaster.getTaskRunner();
                if (taskRunner.isPresent()) {
                    ((TaskRunner)taskRunner.get()).unregisterListener(this.supervisorId);
                }
                Object object2 = this.stopLock;
                synchronized (object2) {
                    if (stopGracefully) {
                        log.info("Posting GracefulShutdownNotice, signalling managed tasks to complete and publish", new Object[0]);
                        this.notices.add(new GracefulShutdownNotice());
                    } else {
                        log.info("Posting ShutdownNotice", new Object[0]);
                        this.notices.add(new ShutdownNotice());
                    }
                    long endTime = System.currentTimeMillis() + 30000L;
                    while (!this.stopped) {
                        long sleepTime = endTime - System.currentTimeMillis();
                        if (sleepTime <= 0L) {
                            log.info("Timed out while waiting for shutdown", new Object[0]);
                            this.stopped = true;
                            break;
                        }
                        this.stopLock.wait(sleepTime);
                    }
                }
                log.info("Shutdown notice handled", new Object[0]);
                this.exec.shutdownNow();
                this.started = false;
                log.info("KafkaSupervisor[%s] has stopped", new Object[]{this.dataSource});
            }
            catch (Exception e) {
                log.makeAlert((Throwable)e, "Exception stopping KafkaSupervisor[%s]", new Object[]{this.dataSource}).emit();
            }
        }
    }

    public SupervisorReport getStatus() {
        return this.generateReport(true);
    }

    public void possiblyRegisterListener() {
        if (this.listenerRegistered) {
            return;
        }
        Optional taskRunner = this.taskMaster.getTaskRunner();
        if (taskRunner.isPresent()) {
            ((TaskRunner)taskRunner.get()).registerListener(new TaskRunnerListener(){

                public String getListenerId() {
                    return KafkaSupervisor.this.supervisorId;
                }

                public void locationChanged(String taskId, TaskLocation newLocation) {
                }

                public void statusChanged(String taskId, TaskStatus status) {
                    KafkaSupervisor.this.notices.add(new RunNotice());
                }
            }, (Executor)MoreExecutors.sameThreadExecutor());
            this.listenerRegistered = true;
        }
    }

    @VisibleForTesting
    void runInternal() {
        this.possiblyRegisterListener();
        this.updatePartitionDataFromKafka();
        this.discoverTasks();
        this.updateTaskStatus();
        this.checkTaskDuration();
        this.checkPendingCompletionTasks();
        this.checkCurrentTaskState();
        this.createNewTasks();
        if (log.isDebugEnabled()) {
            log.debug(this.generateReport(true).toString(), new Object[0]);
        } else {
            log.info(this.generateReport(false).toString(), new Object[0]);
        }
    }

    @VisibleForTesting
    String generateSequenceName(int groupId) {
        String tuningConfig;
        String dataSchema;
        StringBuilder sb = new StringBuilder();
        Map<Integer, Long> startPartitions = this.taskGroups.get((Object)Integer.valueOf((int)groupId)).partitionOffsets;
        for (Map.Entry<Integer, Long> entry : startPartitions.entrySet()) {
            sb.append(String.format("+%d(%d)", entry.getKey(), entry.getValue()));
        }
        String partitionOffsetStr = sb.toString().substring(1);
        Optional<DateTime> minimumMessageTime = this.taskGroups.get((Object)Integer.valueOf((int)groupId)).minimumMessageTime;
        String minMsgTimeStr = minimumMessageTime.isPresent() ? String.valueOf(((DateTime)minimumMessageTime.get()).getMillis()) : "";
        try {
            dataSchema = this.sortingMapper.writeValueAsString((Object)this.spec.getDataSchema());
            tuningConfig = this.sortingMapper.writeValueAsString((Object)this.spec.getTuningConfig());
        }
        catch (JsonProcessingException e) {
            throw Throwables.propagate((Throwable)e);
        }
        String hashCode = DigestUtils.sha1Hex((String)(dataSchema + tuningConfig + partitionOffsetStr + minMsgTimeStr)).substring(0, 15);
        return Joiner.on((String)"_").join((Object)"index_kafka", (Object)this.dataSource, new Object[]{hashCode});
    }

    private static String getRandomId() {
        StringBuilder suffix = new StringBuilder(8);
        for (int i = 0; i < 8; ++i) {
            suffix.append((char)(97 + (RANDOM.nextInt() >>> i * 4 & 0xF)));
        }
        return suffix.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private KafkaConsumer<byte[], byte[]> getKafkaConsumer() {
        Properties props = new Properties();
        props.putAll(this.ioConfig.getConsumerProperties());
        props.setProperty("enable.auto.commit", "false");
        props.setProperty("metadata.max.age.ms", "10000");
        props.setProperty("group.id", String.format("kafka-supervisor-%s", KafkaSupervisor.getRandomId()));
        ClassLoader currCtxCl = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
            KafkaConsumer kafkaConsumer = new KafkaConsumer(props, (Deserializer)new ByteArrayDeserializer(), (Deserializer)new ByteArrayDeserializer());
            return kafkaConsumer;
        }
        finally {
            Thread.currentThread().setContextClassLoader(currCtxCl);
        }
    }

    private void updatePartitionDataFromKafka() {
        Map topics;
        try {
            topics = this.consumer.listTopics();
        }
        catch (Exception e) {
            log.warn((Throwable)e, "Unable to get partition data from Kafka for brokers [%s], are the brokers up?", new Object[]{this.ioConfig.getConsumerProperties().get("bootstrap.servers")});
            return;
        }
        List partitions = (List)topics.get(this.ioConfig.getTopic());
        int numPartitions = partitions != null ? partitions.size() : 0;
        log.debug("Found [%d] Kafka partitions for topic [%s]", new Object[]{numPartitions, this.ioConfig.getTopic()});
        for (int partition = 0; partition < numPartitions; ++partition) {
            Map<Integer, Long> partitionMap;
            int taskGroupId = this.getTaskGroupIdForPartition(partition);
            if (this.partitionGroups.get(taskGroupId) == null) {
                this.partitionGroups.put(taskGroupId, new HashMap());
            }
            if ((partitionMap = this.partitionGroups.get(taskGroupId)).containsKey(partition)) continue;
            log.info("New partition [%d] discovered for topic [%s], adding to task group [%d]", new Object[]{partition, this.ioConfig.getTopic(), taskGroupId});
            partitionMap.put(partition, null);
        }
    }

    private void discoverTasks() {
        int taskCount = 0;
        List tasks = this.taskStorage.getActiveTasks();
        for (Task task : tasks) {
            if (!(task instanceof KafkaIndexTask) || !this.dataSource.equals(task.getDataSource())) continue;
            ++taskCount;
            KafkaIndexTask kafkaTask = (KafkaIndexTask)task;
            String taskId = task.getId();
            Iterator<Integer> it = kafkaTask.getIOConfig().getStartPartitions().getPartitionOffsetMap().keySet().iterator();
            Integer taskGroupId = it.hasNext() ? Integer.valueOf(this.getTaskGroupIdForPartition(it.next())) : null;
            if (taskGroupId == null) continue;
            TaskGroup taskGroup = this.taskGroups.get(taskGroupId);
            if (this.isTaskInPendingCompletionGroups(taskId) || taskGroup != null && taskGroup.tasks.containsKey(taskId)) continue;
            Optional<KafkaIndexTask.Status> status = this.getTaskStatus(taskId);
            if (status.isPresent() && status.get() == KafkaIndexTask.Status.PUBLISHING) {
                this.addDiscoveredTaskToPendingCompletionTaskGroups(taskGroupId, taskId, kafkaTask.getIOConfig().getStartPartitions().getPartitionOffsetMap());
                Map<Integer, Long> publishingTaskCurrentOffsets = this.getCurrentOffsets(taskId, true);
                for (Map.Entry<Integer, Long> entry : publishingTaskCurrentOffsets.entrySet()) {
                    Integer partition = entry.getKey();
                    Long offset = entry.getValue();
                    Map<Integer, Long> partitionOffsets = this.partitionGroups.get(this.getTaskGroupIdForPartition(partition));
                    if (partitionOffsets.get(partition) != null && partitionOffsets.get(partition) >= offset) continue;
                    partitionOffsets.put(partition, offset);
                }
                continue;
            }
            for (Integer partition : kafkaTask.getIOConfig().getStartPartitions().getPartitionOffsetMap().keySet()) {
                if (taskGroupId.equals(this.getTaskGroupIdForPartition(partition))) continue;
                log.warn("Stopping task [%s] which does not match the expected partition allocation", new Object[]{taskId});
                this.stopTask(taskId, false);
                taskGroupId = null;
                break;
            }
            if (taskGroupId == null) continue;
            if (!this.taskGroups.containsKey(taskGroupId)) {
                log.debug("Creating new task group [%d]", new Object[]{taskGroupId});
                this.taskGroups.put(taskGroupId, new TaskGroup(kafkaTask.getIOConfig().getStartPartitions().getPartitionOffsetMap(), kafkaTask.getIOConfig().getMinimumMessageTime()));
            }
            if (!this.isTaskCurrent(taskGroupId, taskId)) {
                log.info("Stopping task [%s] which does not match the expected parameters and ingestion spec", new Object[]{taskId});
                this.stopTask(taskId, false);
                continue;
            }
            this.taskGroups.get((Object)taskGroupId).tasks.put(taskId, new TaskData());
        }
        log.debug("Found [%d] Kafka indexing tasks for dataSource [%s]", new Object[]{taskCount, this.dataSource});
    }

    private void addDiscoveredTaskToPendingCompletionTaskGroups(int groupId, String taskId, Map<Integer, Long> startingPartitions) {
        if (!this.pendingCompletionTaskGroups.containsKey(groupId)) {
            this.pendingCompletionTaskGroups.put(groupId, Lists.newArrayList());
        }
        List<TaskGroup> taskGroupList = this.pendingCompletionTaskGroups.get(groupId);
        for (TaskGroup taskGroup : taskGroupList) {
            if (!taskGroup.partitionOffsets.equals(startingPartitions)) continue;
            if (!taskGroup.tasks.containsKey(taskId)) {
                log.info("Adding discovered task [%s] to existing pending task group", new Object[]{taskId});
                taskGroup.tasks.put(taskId, new TaskData());
            }
            return;
        }
        log.info("Creating new pending completion task group for discovered task [%s]", new Object[]{taskId});
        TaskGroup newTaskGroup = new TaskGroup(startingPartitions, (Optional<DateTime>)Optional.absent());
        newTaskGroup.tasks.put(taskId, new TaskData());
        newTaskGroup.completionTimeout = DateTime.now().plus((ReadableDuration)this.ioConfig.getCompletionTimeout());
        taskGroupList.add(newTaskGroup);
    }

    private void updateTaskStatus() {
        for (TaskGroup taskGroup : this.taskGroups.values()) {
            for (Map.Entry<String, TaskData> entry : taskGroup.tasks.entrySet()) {
                String taskId = entry.getKey();
                TaskData taskData = entry.getValue();
                if (taskData.startTime == null) {
                    try {
                        long millisRemaining;
                        taskData.startTime = this.getTaskStartTime(taskId);
                        if (taskData.startTime != null && (millisRemaining = this.ioConfig.getTaskDuration().getMillis() - (System.currentTimeMillis() - taskData.startTime.getMillis())) > 0L) {
                            this.scheduledExec.schedule(this.buildRunTask(), millisRemaining + 1000L, TimeUnit.MILLISECONDS);
                        }
                    }
                    catch (Exception e) {
                        log.warn((Throwable)e, "Task [%s] failed to return start time, killing task", new Object[]{taskId});
                        this.killTask(taskId);
                    }
                }
                taskData.status = (TaskStatus)this.taskStorage.getStatus(taskId).get();
            }
        }
        for (List list : this.pendingCompletionTaskGroups.values()) {
            for (TaskGroup group : list) {
                for (Map.Entry<String, TaskData> entry : group.tasks.entrySet()) {
                    entry.getValue().status = (TaskStatus)this.taskStorage.getStatus(entry.getKey()).get();
                }
            }
        }
    }

    private void checkTaskDuration() {
        Iterator<Map.Entry<Integer, TaskGroup>> i = this.taskGroups.entrySet().iterator();
        while (i.hasNext()) {
            Map.Entry<Integer, TaskGroup> groupEntry = i.next();
            Integer groupId = groupEntry.getKey();
            TaskGroup group = groupEntry.getValue();
            DateTime earliestTaskStart = DateTime.now();
            for (TaskData taskData : group.tasks.values()) {
                if (!earliestTaskStart.isAfter((ReadableInstant)taskData.startTime)) continue;
                earliestTaskStart = taskData.startTime;
            }
            if (!earliestTaskStart.plus((ReadableDuration)this.ioConfig.getTaskDuration()).isBeforeNow()) continue;
            log.info("Task group [%d] has run for [%s]", new Object[]{groupId, this.ioConfig.getTaskDuration()});
            Map<Integer, Long> endOffsets = this.signalTasksToFinish(groupId);
            group.completionTimeout = DateTime.now().plus((ReadableDuration)this.ioConfig.getCompletionTimeout());
            if (!this.pendingCompletionTaskGroups.containsKey(groupId)) {
                this.pendingCompletionTaskGroups.put(groupId, Lists.newArrayList());
            }
            this.pendingCompletionTaskGroups.get(groupId).add(group);
            if (endOffsets != null) {
                for (Map.Entry<Integer, Long> entry : endOffsets.entrySet()) {
                    this.partitionGroups.get(groupId).put(entry.getKey(), entry.getValue());
                }
            }
            i.remove();
        }
    }

    private Map<Integer, Long> signalTasksToFinish(int groupId) {
        TaskData task;
        String taskId;
        Map.Entry<String, TaskData> taskEntry;
        TaskGroup taskGroup = this.taskGroups.get(groupId);
        HashMap<Integer, Long> endOffsets = new HashMap<Integer, Long>();
        Iterator<Map.Entry<String, TaskData>> i = taskGroup.tasks.entrySet().iterator();
        while (i.hasNext()) {
            Map<Integer, Long> currentOffsets;
            taskEntry = i.next();
            taskId = taskEntry.getKey();
            task = taskEntry.getValue();
            if (task.status.isSuccess()) {
                this.stopTasksInGroup(taskGroup);
                return null;
            }
            if (!task.status.isRunnable()) continue;
            if (this.taskInfoProvider.getTaskLocation(taskId).equals((Object)TaskLocation.unknown())) {
                log.info("Killing task [%s] which hasn't been assigned to a worker", new Object[]{taskId});
                this.killTask(taskId);
                i.remove();
                continue;
            }
            try {
                currentOffsets = this.taskClient.pause(taskId);
            }
            catch (Exception e) {
                log.warn((Throwable)e, "Task [%s] failed to respond to [pause] in a timely manner, killing task", new Object[]{taskId});
                this.killTask(taskId);
                i.remove();
                continue;
            }
            for (Map.Entry<Integer, Long> offset : currentOffsets.entrySet()) {
                if (endOffsets.containsKey(offset.getKey()) && ((Long)endOffsets.get(offset.getKey())).compareTo(offset.getValue()) >= 0) continue;
                endOffsets.put(offset.getKey(), offset.getValue());
            }
        }
        log.info("Setting endOffsets for tasks in taskGroup [%d] to %s and resuming", new Object[]{groupId, endOffsets});
        i = taskGroup.tasks.entrySet().iterator();
        while (i.hasNext()) {
            taskEntry = i.next();
            taskId = taskEntry.getKey();
            task = taskEntry.getValue();
            if (!task.status.isRunnable()) continue;
            try {
                this.taskClient.setEndOffsets(taskId, endOffsets, true);
            }
            catch (Exception e) {
                log.warn((Throwable)e, "Task [%s] failed to respond to [set end offsets] in a timely manner, killing task", new Object[]{taskId});
                this.killTask(taskId);
                i.remove();
            }
        }
        return endOffsets;
    }

    private void checkPendingCompletionTasks() {
        for (Map.Entry<Integer, List<TaskGroup>> pendingGroupList : this.pendingCompletionTaskGroups.entrySet()) {
            boolean stopTasksInTaskGroup = false;
            Integer groupId = pendingGroupList.getKey();
            Iterator<TaskGroup> iTaskGroup = pendingGroupList.getValue().iterator();
            while (iTaskGroup.hasNext()) {
                boolean foundSuccess = false;
                boolean entireTaskGroupFailed = false;
                TaskGroup group = iTaskGroup.next();
                if (stopTasksInTaskGroup) {
                    this.stopTasksInGroup(group);
                    iTaskGroup.remove();
                    continue;
                }
                Iterator<Map.Entry<String, TaskData>> iTask = group.tasks.entrySet().iterator();
                while (iTask.hasNext()) {
                    Map.Entry<String, TaskData> task = iTask.next();
                    if (task.getValue().status.isFailure()) {
                        iTask.remove();
                        if (group.tasks.isEmpty()) {
                            entireTaskGroupFailed = true;
                            break;
                        }
                    }
                    if (!task.getValue().status.isSuccess()) continue;
                    log.info("Task [%s] completed successfully, stopping tasks %s", new Object[]{task.getKey(), group.tasks.keySet()});
                    this.stopTasksInGroup(group);
                    foundSuccess = true;
                    iTaskGroup.remove();
                    break;
                }
                if ((foundSuccess || !group.completionTimeout.isBeforeNow()) && !entireTaskGroupFailed) continue;
                if (entireTaskGroupFailed) {
                    log.warn("All tasks in group [%d] failed to publish, killing all tasks for these partitions", new Object[]{groupId});
                } else {
                    log.makeAlert("No task in [%s] succeeded before the completion timeout elapsed [%s]!", new Object[]{group.tasks.keySet(), this.ioConfig.getCompletionTimeout()}).emit();
                }
                this.partitionGroups.remove(groupId);
                this.stopTasksInGroup(group);
                stopTasksInTaskGroup = true;
                this.stopTasksInGroup(this.taskGroups.remove(groupId));
                iTaskGroup.remove();
            }
        }
    }

    private void checkCurrentTaskState() {
        Iterator<Map.Entry<Integer, TaskGroup>> iTaskGroups = this.taskGroups.entrySet().iterator();
        while (iTaskGroups.hasNext()) {
            Map.Entry<Integer, TaskGroup> taskGroupEntry = iTaskGroups.next();
            Integer groupId = taskGroupEntry.getKey();
            TaskGroup taskGroup = taskGroupEntry.getValue();
            log.debug("Task group [%d] pre-pruning: %s", new Object[]{groupId, taskGroup.tasks.keySet()});
            Iterator<Map.Entry<String, TaskData>> iTasks = taskGroup.tasks.entrySet().iterator();
            while (iTasks.hasNext()) {
                Map.Entry<String, TaskData> task = iTasks.next();
                String taskId = task.getKey();
                TaskData taskData = task.getValue();
                if (!this.isTaskCurrent(groupId, taskId)) {
                    log.info("Stopping task [%s] which does not match the expected offset range and ingestion spec", new Object[]{taskId});
                    this.stopTask(taskId, false);
                    iTasks.remove();
                    continue;
                }
                if (taskData.status.isFailure()) {
                    iTasks.remove();
                    continue;
                }
                if (!taskData.status.isSuccess()) continue;
                this.stopTasksInGroup(taskGroup);
                iTaskGroups.remove();
                break;
            }
            log.debug("Task group [%d] post-pruning: %s", new Object[]{groupId, taskGroup.tasks.keySet()});
        }
    }

    void createNewTasks() {
        for (Integer groupId : this.partitionGroups.keySet()) {
            if (this.taskGroups.containsKey(groupId)) continue;
            log.info("Creating new task group [%d] for partitions %s", new Object[]{groupId, this.partitionGroups.get(groupId).keySet()});
            Optional minimumMessageTime = this.ioConfig.getLateMessageRejectionPeriod().isPresent() ? Optional.of((Object)DateTime.now().minus((ReadableDuration)this.ioConfig.getLateMessageRejectionPeriod().get())) : Optional.absent();
            this.taskGroups.put(groupId, new TaskGroup(this.generateStartingOffsetsForPartitionGroup(groupId), (Optional<DateTime>)minimumMessageTime));
        }
        boolean createdTask = false;
        for (Map.Entry<Integer, TaskGroup> entry : this.taskGroups.entrySet()) {
            TaskGroup taskGroup = entry.getValue();
            Integer groupId = entry.getKey();
            if (this.ioConfig.getReplicas() <= taskGroup.tasks.size()) continue;
            log.info("Number of tasks [%d] does not match configured numReplicas [%d] in task group [%d], creating more tasks", new Object[]{taskGroup.tasks.size(), this.ioConfig.getReplicas(), groupId});
            this.createKafkaTasksForGroup(groupId, this.ioConfig.getReplicas() - taskGroup.tasks.size());
            createdTask = true;
        }
        if (createdTask && this.firstRunTime.isBeforeNow()) {
            this.scheduledExec.schedule(this.buildRunTask(), 5000L, TimeUnit.MILLISECONDS);
        }
    }

    private void createKafkaTasksForGroup(int groupId, int replicas) {
        Map<Integer, Long> startPartitions = this.taskGroups.get((Object)Integer.valueOf((int)groupId)).partitionOffsets;
        HashMap<Integer, Long> endPartitions = new HashMap<Integer, Long>();
        for (Integer partition : startPartitions.keySet()) {
            endPartitions.put(partition, Long.MAX_VALUE);
        }
        String sequenceName = this.generateSequenceName(groupId);
        HashMap consumerProperties = Maps.newHashMap(this.ioConfig.getConsumerProperties());
        DateTime minimumMessageTime = (DateTime)this.taskGroups.get((Object)Integer.valueOf((int)groupId)).minimumMessageTime.orNull();
        KafkaIOConfig kafkaIOConfig = new KafkaIOConfig(sequenceName, new KafkaPartitions(this.ioConfig.getTopic(), startPartitions), new KafkaPartitions(this.ioConfig.getTopic(), endPartitions), consumerProperties, true, false, minimumMessageTime);
        for (int i = 0; i < replicas; ++i) {
            String taskId = Joiner.on((String)"_").join((Object)sequenceName, (Object)KafkaSupervisor.getRandomId(), new Object[0]);
            KafkaIndexTask indexTask = new KafkaIndexTask(taskId, new TaskResource(sequenceName, 1), this.spec.getDataSchema(), this.spec.getTuningConfig(), kafkaIOConfig, (Map<String, Object>)ImmutableMap.of(), null);
            Optional taskQueue = this.taskMaster.getTaskQueue();
            if (taskQueue.isPresent()) {
                try {
                    ((TaskQueue)taskQueue.get()).add((Task)indexTask);
                }
                catch (EntryExistsException e) {
                    log.error("Tried to add task [%s] but it already exists", new Object[]{indexTask.getId()});
                }
                continue;
            }
            log.error("Failed to get task queue because I'm not the leader!", new Object[0]);
        }
    }

    private Map<Integer, Long> generateStartingOffsetsForPartitionGroup(int groupId) {
        HashMap<Integer, Long> startingOffsets = new HashMap<Integer, Long>();
        for (Map.Entry<Integer, Long> entry : this.partitionGroups.get(groupId).entrySet()) {
            Integer partition = entry.getKey();
            Long offset = entry.getValue();
            if (offset != null) {
                startingOffsets.put(partition, offset);
                continue;
            }
            startingOffsets.put(partition, this.getOffsetFromStorageForPartition(partition));
        }
        return startingOffsets;
    }

    private long getOffsetFromStorageForPartition(int partition) {
        long offset;
        Map<Integer, Long> metadataOffsets = this.getOffsetsFromMetadataStorage();
        if (metadataOffsets.get(partition) != null) {
            offset = metadataOffsets.get(partition);
            log.debug("Getting offset [%,d] from metadata storage for partition [%d]", new Object[]{offset, partition});
            long latestKafkaOffset = this.getOffsetFromKafkaForPartition(partition, false);
            if (offset > latestKafkaOffset) {
                throw new ISE("Offset in metadata storage [%,d] > latest Kafka offset [%,d] for partition [%d]. If your Kafka offsets have been reset, you will need to remove the entry for [%s] from the dataSource table.", new Object[]{offset, latestKafkaOffset, partition, this.dataSource});
            }
        } else {
            offset = this.getOffsetFromKafkaForPartition(partition, this.ioConfig.isUseEarliestOffset());
            log.debug("Getting offset [%,d] from Kafka for partition [%d]", new Object[]{offset, partition});
        }
        return offset;
    }

    private Map<Integer, Long> getOffsetsFromMetadataStorage() {
        KafkaPartitions partitions;
        DataSourceMetadata dataSourceMetadata = this.indexerMetadataStorageCoordinator.getDataSourceMetadata(this.dataSource);
        if (dataSourceMetadata != null && dataSourceMetadata instanceof KafkaDataSourceMetadata && (partitions = ((KafkaDataSourceMetadata)dataSourceMetadata).getKafkaPartitions()) != null) {
            if (!this.ioConfig.getTopic().equals(partitions.getTopic())) {
                log.warn("Topic in metadata storage [%s] doesn't match spec topic [%s], ignoring stored offsets", new Object[]{partitions.getTopic(), this.ioConfig.getTopic()});
                return ImmutableMap.of();
            }
            if (partitions.getPartitionOffsetMap() != null) {
                return partitions.getPartitionOffsetMap();
            }
        }
        return ImmutableMap.of();
    }

    private long getOffsetFromKafkaForPartition(int partition, boolean useEarliestOffset) {
        TopicPartition topicPartition = new TopicPartition(this.ioConfig.getTopic(), partition);
        if (!this.consumer.assignment().contains(topicPartition)) {
            this.consumer.assign((List)Lists.newArrayList((Object[])new TopicPartition[]{topicPartition}));
        }
        if (useEarliestOffset) {
            this.consumer.seekToBeginning(new TopicPartition[]{topicPartition});
        } else {
            this.consumer.seekToEnd(new TopicPartition[]{topicPartition});
        }
        return this.consumer.position(topicPartition);
    }

    private boolean isTaskCurrent(int taskGroupId, String taskId) {
        Optional taskOptional = this.taskStorage.getTask(taskId);
        if (!taskOptional.isPresent() || !(taskOptional.get() instanceof KafkaIndexTask)) {
            return false;
        }
        String taskSequenceName = ((KafkaIndexTask)((Object)taskOptional.get())).getIOConfig().getBaseSequenceName();
        return this.generateSequenceName(taskGroupId).equals(taskSequenceName);
    }

    private void stopTasksInGroup(TaskGroup taskGroup) {
        if (taskGroup == null) {
            return;
        }
        for (Map.Entry<String, TaskData> entry : taskGroup.tasks.entrySet()) {
            if (entry.getValue().status.isComplete()) continue;
            this.stopTask(entry.getKey(), false);
        }
    }

    private void stopTask(String id, boolean publish) {
        if (!this.taskInfoProvider.getTaskLocation(id).equals((Object)TaskLocation.unknown())) {
            try {
                this.taskClient.stop(id, publish);
            }
            catch (Exception e) {
                log.warn((Throwable)e, "Task [%s] failed to stop in a timely manner, killing task", new Object[]{id});
                this.killTask(id);
            }
        } else {
            this.killTask(id);
        }
    }

    private void killTask(String id) {
        Optional taskQueue = this.taskMaster.getTaskQueue();
        if (taskQueue.isPresent()) {
            ((TaskQueue)taskQueue.get()).shutdown(id);
        } else {
            log.error("Failed to get task queue because I'm not the leader!", new Object[0]);
        }
    }

    private DateTime getTaskStartTime(String id) {
        if (!this.taskInfoProvider.getTaskLocation(id).equals((Object)TaskLocation.unknown())) {
            DateTime startTime = this.taskClient.getStartTime(id, false);
            log.debug("Received start time of [%s] from task [%s]", new Object[]{startTime, id});
            return startTime;
        }
        return null;
    }

    private Optional<KafkaIndexTask.Status> getTaskStatus(String id) {
        if (!this.taskInfoProvider.getTaskLocation(id).equals((Object)TaskLocation.unknown())) {
            try {
                return Optional.of((Object)((Object)this.taskClient.getStatus(id)));
            }
            catch (Exception e) {
                log.warn((Throwable)e, "Failed to get status for task [%s]", new Object[]{id});
            }
        }
        return Optional.absent();
    }

    private Map<Integer, Long> getCurrentOffsets(String id, boolean retry) {
        if (!this.taskInfoProvider.getTaskLocation(id).equals((Object)TaskLocation.unknown())) {
            try {
                return this.taskClient.getCurrentOffsets(id, retry);
            }
            catch (Exception e) {
                log.info("Failed to get current offsets for task [%s]", new Object[]{id});
            }
        }
        return ImmutableMap.of();
    }

    private int getTaskGroupIdForPartition(int partition) {
        return partition % this.ioConfig.getTaskCount();
    }

    private boolean isTaskInPendingCompletionGroups(String taskId) {
        for (List<TaskGroup> taskGroups : this.pendingCompletionTaskGroups.values()) {
            for (TaskGroup taskGroup : taskGroups) {
                if (!taskGroup.tasks.containsKey(taskId)) continue;
                return true;
            }
        }
        return false;
    }

    private KafkaSupervisorReport generateReport(boolean includeOffsets) {
        int numPartitions = 0;
        for (Map<Integer, Long> map : this.partitionGroups.values()) {
            numPartitions += map.size();
        }
        KafkaSupervisorReport report = new KafkaSupervisorReport(this.dataSource, DateTime.now(), this.ioConfig.getTopic(), numPartitions, this.ioConfig.getReplicas(), this.ioConfig.getTaskDuration().getMillis() / 1000L);
        try {
            for (TaskGroup taskGroup : this.taskGroups.values()) {
                for (Map.Entry<String, TaskData> entry : taskGroup.tasks.entrySet()) {
                    String taskId = entry.getKey();
                    DateTime startTime = entry.getValue().startTime;
                    Long remainingSeconds = null;
                    if (startTime != null) {
                        remainingSeconds = Math.max(0L, this.ioConfig.getTaskDuration().getMillis() - (DateTime.now().getMillis() - startTime.getMillis())) / 1000L;
                    }
                    report.addActiveTask(taskId, includeOffsets ? taskGroup.partitionOffsets : null, includeOffsets ? this.getCurrentOffsets(taskId, false) : null, startTime, remainingSeconds);
                }
            }
            for (List<TaskGroup> taskGroups : this.pendingCompletionTaskGroups.values()) {
                for (TaskGroup taskGroup : taskGroups) {
                    for (Map.Entry<String, TaskData> entry : taskGroup.tasks.entrySet()) {
                        String taskId = entry.getKey();
                        DateTime startTime = entry.getValue().startTime;
                        Long remainingSeconds = null;
                        if (taskGroup.completionTimeout != null) {
                            remainingSeconds = Math.max(0L, taskGroup.completionTimeout.getMillis() - DateTime.now().getMillis()) / 1000L;
                        }
                        report.addPublishingTask(taskId, includeOffsets ? taskGroup.partitionOffsets : null, includeOffsets ? this.getCurrentOffsets(taskId, false) : null, startTime, remainingSeconds);
                    }
                }
            }
        }
        catch (Exception exception) {
            log.warn((Throwable)exception, "Failed to generate status report", new Object[0]);
        }
        return report;
    }

    private Runnable buildRunTask() {
        return new Runnable(){

            @Override
            public void run() {
                KafkaSupervisor.this.notices.add(new RunNotice());
            }
        };
    }

    private class ShutdownNotice
    implements Notice {
        private ShutdownNotice() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void handle() {
            KafkaSupervisor.this.consumer.close();
            Object object = KafkaSupervisor.this.stopLock;
            synchronized (object) {
                KafkaSupervisor.this.stopped = true;
                KafkaSupervisor.this.stopLock.notifyAll();
            }
        }
    }

    private class GracefulShutdownNotice
    extends ShutdownNotice {
        private GracefulShutdownNotice() {
        }

        @Override
        public void handle() {
            for (TaskGroup taskGroup : KafkaSupervisor.this.taskGroups.values()) {
                for (Map.Entry<String, TaskData> entry : taskGroup.tasks.entrySet()) {
                    if (KafkaSupervisor.this.taskInfoProvider.getTaskLocation(entry.getKey()).equals((Object)TaskLocation.unknown())) {
                        KafkaSupervisor.this.killTask(entry.getKey());
                        continue;
                    }
                    entry.getValue().startTime = new DateTime(0L);
                }
            }
            KafkaSupervisor.this.checkTaskDuration();
            super.handle();
        }
    }

    private class RunNotice
    implements Notice {
        private RunNotice() {
        }

        @Override
        public void handle() {
            long nowTime = System.currentTimeMillis();
            if (nowTime - KafkaSupervisor.this.lastRunTime < 1000L) {
                return;
            }
            KafkaSupervisor.this.lastRunTime = nowTime;
            KafkaSupervisor.this.runInternal();
        }
    }

    private static interface Notice {
        public void handle();
    }

    private class TaskData {
        TaskStatus status;
        DateTime startTime;

        private TaskData() {
        }
    }

    private class TaskGroup {
        final Map<Integer, Long> partitionOffsets;
        final Map<String, TaskData> tasks = new HashMap<String, TaskData>();
        final Optional<DateTime> minimumMessageTime;
        DateTime completionTimeout;

        public TaskGroup(Map<Integer, Long> partitionOffsets, Optional<DateTime> minimumMessageTime) {
            this.partitionOffsets = partitionOffsets;
            this.minimumMessageTime = minimumMessageTime;
        }
    }
}

