/*
 * Decompiled with CFR 0.152.
 */
package io.druid.server.coordinator.helper;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.inject.Inject;
import io.druid.client.indexing.IndexingServiceClient;
import io.druid.java.util.common.DateTimes;
import io.druid.java.util.common.JodaUtils;
import io.druid.java.util.common.logger.Logger;
import io.druid.metadata.MetadataSegmentManager;
import io.druid.server.coordinator.DruidCoordinatorConfig;
import io.druid.server.coordinator.DruidCoordinatorRuntimeParams;
import io.druid.server.coordinator.helper.DruidCoordinatorHelper;
import java.util.Collection;
import java.util.List;
import org.joda.time.Interval;
import org.joda.time.ReadableInstant;

public class DruidCoordinatorSegmentKiller
implements DruidCoordinatorHelper {
    private static final Logger log = new Logger(DruidCoordinatorSegmentKiller.class);
    private final long period;
    private final long retainDuration;
    private final int maxSegmentsToKill;
    private long lastKillTime = 0L;
    private final MetadataSegmentManager segmentManager;
    private final IndexingServiceClient indexingServiceClient;

    @Inject
    public DruidCoordinatorSegmentKiller(MetadataSegmentManager segmentManager, IndexingServiceClient indexingServiceClient, DruidCoordinatorConfig config) {
        this.period = config.getCoordinatorKillPeriod().getMillis();
        Preconditions.checkArgument((this.period > config.getCoordinatorIndexingPeriod().getMillis() ? 1 : 0) != 0, (Object)"coordinator kill period must be greater than druid.coordinator.period.indexingPeriod");
        this.retainDuration = config.getCoordinatorKillDurationToRetain().getMillis();
        Preconditions.checkArgument((this.retainDuration >= 0L ? 1 : 0) != 0, (Object)"coordinator kill retainDuration must be >= 0");
        this.maxSegmentsToKill = config.getCoordinatorKillMaxSegments();
        Preconditions.checkArgument((this.maxSegmentsToKill > 0 ? 1 : 0) != 0, (Object)"coordinator kill maxSegments must be > 0");
        log.info("Kill Task scheduling enabled with period [%s], retainDuration [%s], maxSegmentsToKill [%s]", new Object[]{this.period, this.retainDuration, this.maxSegmentsToKill});
        this.segmentManager = segmentManager;
        this.indexingServiceClient = indexingServiceClient;
    }

    @Override
    public DruidCoordinatorRuntimeParams run(DruidCoordinatorRuntimeParams params) {
        boolean killAllDataSources = params.getCoordinatorDynamicConfig().isKillAllDataSources();
        Collection<String> whitelist = params.getCoordinatorDynamicConfig().getKillDataSourceWhitelist();
        if (killAllDataSources && whitelist != null && !whitelist.isEmpty()) {
            log.error("killAllDataSources can't be true when killDataSourceWhitelist is non-empty, No kill tasks are scheduled.", new Object[0]);
            return params;
        }
        if (killAllDataSources) {
            whitelist = this.segmentManager.getAllDatasourceNames();
        }
        if (whitelist != null && whitelist.size() > 0 && this.lastKillTime + this.period < System.currentTimeMillis()) {
            this.lastKillTime = System.currentTimeMillis();
            for (String dataSource : whitelist) {
                Interval intervalToKill = this.findIntervalForKillTask(dataSource, this.maxSegmentsToKill);
                if (intervalToKill == null) continue;
                try {
                    this.indexingServiceClient.killSegments(dataSource, intervalToKill);
                }
                catch (Exception ex) {
                    log.error((Throwable)ex, "Failed to submit kill task for dataSource [%s]", new Object[]{dataSource});
                    if (!Thread.currentThread().isInterrupted()) continue;
                    log.warn("skipping kill task scheduling because thread is interrupted.", new Object[0]);
                    break;
                }
            }
        }
        return params;
    }

    @VisibleForTesting
    Interval findIntervalForKillTask(String dataSource, int limit) {
        List<Interval> unusedSegmentIntervals = this.segmentManager.getUnusedSegmentIntervals(dataSource, new Interval((ReadableInstant)DateTimes.EPOCH, (ReadableInstant)DateTimes.nowUtc().minus(this.retainDuration)), limit);
        if (unusedSegmentIntervals != null && unusedSegmentIntervals.size() > 0) {
            return JodaUtils.umbrellaInterval(unusedSegmentIntervals);
        }
        return null;
    }
}

