/*
 * Decompiled with CFR 0.152.
 */
package io.druid.storage.s3;

import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.io.ByteSource;
import com.google.common.io.Files;
import com.google.inject.Inject;
import com.metamx.common.CompressionUtils;
import com.metamx.common.FileUtils;
import com.metamx.common.IAE;
import com.metamx.common.ISE;
import com.metamx.common.MapUtils;
import com.metamx.common.StringUtils;
import com.metamx.common.UOE;
import com.metamx.common.logger.Logger;
import io.druid.segment.loading.DataSegmentPuller;
import io.druid.segment.loading.SegmentLoadingException;
import io.druid.segment.loading.URIDataPuller;
import io.druid.storage.s3.S3Utils;
import io.druid.timeline.DataSegment;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.net.URI;
import java.util.Map;
import java.util.concurrent.Callable;
import javax.tools.FileObject;
import org.apache.commons.io.FileUtils;
import org.jets3t.service.S3ServiceException;
import org.jets3t.service.ServiceException;
import org.jets3t.service.impl.rest.httpclient.RestS3Service;
import org.jets3t.service.model.StorageObject;

public class S3DataSegmentPuller
implements DataSegmentPuller,
URIDataPuller {
    public static final int DEFAULT_RETRY_COUNT = 3;
    public static final String scheme = "s3_zip";
    private static final Logger log = new Logger(S3DataSegmentPuller.class);
    protected static final String BUCKET = "bucket";
    protected static final String KEY = "key";
    protected final RestS3Service s3Client;

    public static FileObject buildFileObject(final URI uri, final RestS3Service s3Client) throws ServiceException {
        S3Coords coords = new S3Coords(S3DataSegmentPuller.checkURI(uri));
        final StorageObject s3Obj = s3Client.getObjectDetails(coords.bucket, coords.path);
        final String path = uri.getPath();
        return new FileObject(){
            final Object inputStreamOpener = new Object();
            volatile boolean streamAcquired = false;
            volatile StorageObject storageObject = s3Obj;

            @Override
            public URI toUri() {
                return uri;
            }

            @Override
            public String getName() {
                String ext = Files.getFileExtension((String)path);
                return Files.getNameWithoutExtension((String)path) + (Strings.isNullOrEmpty((String)ext) ? "" : "." + ext);
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public InputStream openInputStream() throws IOException {
                try {
                    Object object = this.inputStreamOpener;
                    synchronized (object) {
                        if (this.streamAcquired) {
                            return this.storageObject.getDataInputStream();
                        }
                        this.storageObject = s3Client.getObject(s3Obj.getBucketName(), s3Obj.getKey());
                        InputStream stream = this.storageObject.getDataInputStream();
                        this.streamAcquired = true;
                        return stream;
                    }
                }
                catch (ServiceException e) {
                    throw new IOException(StringUtils.safeFormat((String)"Could not load S3 URI [%s]", (Object[])new Object[]{uri}), e);
                }
            }

            @Override
            public OutputStream openOutputStream() throws IOException {
                throw new UOE("Cannot stream S3 output", new Object[0]);
            }

            @Override
            public Reader openReader(boolean ignoreEncodingErrors) throws IOException {
                throw new UOE("Cannot open reader", new Object[0]);
            }

            @Override
            public CharSequence getCharContent(boolean ignoreEncodingErrors) throws IOException {
                throw new UOE("Cannot open character sequence", new Object[0]);
            }

            @Override
            public Writer openWriter() throws IOException {
                throw new UOE("Cannot open writer", new Object[0]);
            }

            @Override
            public long getLastModified() {
                return s3Obj.getLastModifiedDate().getTime();
            }

            @Override
            public boolean delete() {
                throw new UOE("Cannot delete S3 items anonymously. jetS3t doesn't support authenticated deletes easily.", new Object[0]);
            }
        };
    }

    @Inject
    public S3DataSegmentPuller(RestS3Service s3Client) {
        this.s3Client = s3Client;
    }

    public void getSegmentFiles(DataSegment segment, File outDir) throws SegmentLoadingException {
        this.getSegmentFiles(new S3Coords(segment), outDir);
    }

    public FileUtils.FileCopyResult getSegmentFiles(S3Coords s3Coords, File outDir) throws SegmentLoadingException {
        log.info("Pulling index at path[%s] to outDir[%s]", new Object[]{s3Coords, outDir});
        if (!this.isObjectInBucket(s3Coords)) {
            throw new SegmentLoadingException("IndexFile[%s] does not exist.", new Object[]{s3Coords});
        }
        if (!outDir.exists()) {
            outDir.mkdirs();
        }
        if (!outDir.isDirectory()) {
            throw new ISE("outDir[%s] must be a directory.", new Object[]{outDir});
        }
        try {
            final URI uri = URI.create(String.format("s3://%s/%s", s3Coords.bucket, s3Coords.path));
            ByteSource byteSource = new ByteSource(){

                public InputStream openStream() throws IOException {
                    try {
                        return S3DataSegmentPuller.buildFileObject(uri, S3DataSegmentPuller.this.s3Client).openInputStream();
                    }
                    catch (ServiceException e) {
                        if (e.getCause() != null && S3Utils.S3RETRY.apply((Object)e)) {
                            throw new IOException("Recoverable exception", e);
                        }
                        throw Throwables.propagate((Throwable)e);
                    }
                }
            };
            if (CompressionUtils.isZip((String)s3Coords.path)) {
                FileUtils.FileCopyResult result = CompressionUtils.unzip((ByteSource)byteSource, (File)outDir, S3Utils.S3RETRY, (boolean)true);
                log.info("Loaded %d bytes from [%s] to [%s]", new Object[]{result.size(), s3Coords.toString(), outDir.getAbsolutePath()});
                return result;
            }
            if (CompressionUtils.isGz((String)s3Coords.path)) {
                String fname = Files.getNameWithoutExtension((String)uri.getPath());
                File outFile = new File(outDir, fname);
                FileUtils.FileCopyResult result = CompressionUtils.gunzip((ByteSource)byteSource, (File)outFile, S3Utils.S3RETRY);
                log.info("Loaded %d bytes from [%s] to [%s]", new Object[]{result.size(), s3Coords.toString(), outFile.getAbsolutePath()});
                return result;
            }
            throw new IAE("Do not know how to load file type at [%s]", new Object[]{uri.toString()});
        }
        catch (Exception e) {
            try {
                FileUtils.deleteDirectory((File)outDir);
            }
            catch (IOException ioe) {
                log.warn((Throwable)ioe, "Failed to remove output directory [%s] for segment pulled from [%s]", new Object[]{outDir.getAbsolutePath(), s3Coords.toString()});
            }
            throw new SegmentLoadingException((Throwable)e, e.getMessage(), new Object[0]);
        }
    }

    public static URI checkURI(URI uri) {
        if (uri.getScheme().equalsIgnoreCase(scheme)) {
            uri = URI.create("s3" + uri.toString().substring(scheme.length()));
        } else if (!uri.getScheme().equalsIgnoreCase("s3")) {
            throw new IAE("Don't know how to load scheme for URI [%s]", new Object[]{uri.toString()});
        }
        return uri;
    }

    public InputStream getInputStream(URI uri) throws IOException {
        try {
            return S3DataSegmentPuller.buildFileObject(uri, this.s3Client).openInputStream();
        }
        catch (ServiceException e) {
            throw new IOException(String.format("Could not load URI [%s]", uri.toString()), e);
        }
    }

    public Predicate<Throwable> shouldRetryPredicate() {
        return new Predicate<Throwable>(){

            public boolean apply(Throwable e) {
                if (e == null) {
                    return false;
                }
                if (e instanceof ServiceException) {
                    return S3Utils.isServiceExceptionRecoverable((ServiceException)e);
                }
                if (S3Utils.S3RETRY.apply((Object)e)) {
                    return true;
                }
                return this.apply(e.getCause());
            }
        };
    }

    public String getVersion(URI uri) throws IOException {
        try {
            FileObject object = S3DataSegmentPuller.buildFileObject(uri, this.s3Client);
            return String.format("%d", object.getLastModified());
        }
        catch (ServiceException e) {
            if (S3Utils.isServiceExceptionRecoverable(e)) {
                throw new IOException(String.format("Could not fetch last modified timestamp from URI [%s]", uri.toString()), e);
            }
            throw Throwables.propagate((Throwable)e);
        }
    }

    private String toFilename(String key, String suffix) {
        String filename = key.substring(key.lastIndexOf("/") + 1);
        filename = filename.substring(0, filename.length() - suffix.length());
        return filename;
    }

    private boolean isObjectInBucket(final S3Coords coords) throws SegmentLoadingException {
        try {
            return S3Utils.retryS3Operation(new Callable<Boolean>(){

                @Override
                public Boolean call() throws Exception {
                    return S3Utils.isObjectInBucket(S3DataSegmentPuller.this.s3Client, coords.bucket, coords.path);
                }
            });
        }
        catch (IOException | S3ServiceException e) {
            throw new SegmentLoadingException(e, "S3 fail! Key[%s]", new Object[]{coords});
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    protected static class S3Coords {
        String bucket;
        String path;

        public S3Coords(URI uri) {
            if (!"s3".equalsIgnoreCase(uri.getScheme())) {
                throw new IAE("Unsupported scheme: [%s]", new Object[]{uri.getScheme()});
            }
            this.bucket = uri.getHost();
            String path = uri.getPath();
            if (path.startsWith("/")) {
                path = path.substring(1);
            }
            this.path = path;
        }

        public S3Coords(DataSegment segment) {
            Map loadSpec = segment.getLoadSpec();
            this.bucket = MapUtils.getString((Map)loadSpec, (String)S3DataSegmentPuller.BUCKET);
            this.path = MapUtils.getString((Map)loadSpec, (String)S3DataSegmentPuller.KEY);
            if (this.path.startsWith("/")) {
                this.path = this.path.substring(1);
            }
        }

        public S3Coords(String bucket, String key) {
            this.bucket = bucket;
            this.path = key;
        }

        public String toString() {
            return String.format("s3://%s/%s", this.bucket, this.path);
        }
    }
}

