/*
 * Licensed to Metamarkets Group Inc. (Metamarkets) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. Metamarkets licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package io.druid.java.util.metrics;

import com.google.common.collect.ImmutableMap;
import io.druid.java.util.emitter.core.ConcurrentTimeCounter;
import io.druid.java.util.emitter.core.HttpPostEmitter;
import io.druid.java.util.emitter.service.ServiceEmitter;
import io.druid.java.util.emitter.service.ServiceMetricEvent;

public class HttpPostEmitterMonitor extends FeedDefiningMonitor
{
  private final HttpPostEmitter httpPostEmitter;
  private final ImmutableMap<String, String> extraDimensions;
  private final ServiceMetricEvent.Builder builder;
  private long lastTotalEmittedEvents = 0;
  private int lastDroppedBuffers = 0;

  public HttpPostEmitterMonitor(
      String feed,
      HttpPostEmitter httpPostEmitter,
      ImmutableMap<String, String> extraDimensions
  )
  {
    super(feed);
    this.httpPostEmitter = httpPostEmitter;
    this.extraDimensions = extraDimensions;
    this.builder = builder();
  }

  @Override
  public boolean doMonitor(ServiceEmitter emitter)
  {
    long newTotalEmittedEvents = httpPostEmitter.getTotalEmittedEvents();
    long totalEmittedEventsDiff = newTotalEmittedEvents - lastTotalEmittedEvents;
    emitter.emit(builder.build("emitter/events/emitted", totalEmittedEventsDiff));
    lastTotalEmittedEvents = newTotalEmittedEvents;

    int newDroppedBuffers = httpPostEmitter.getDroppedBuffers();
    int droppedBuffersDiff = newDroppedBuffers - lastDroppedBuffers;
    emitter.emit(builder.build("emitter/buffers/dropped", droppedBuffersDiff));
    lastDroppedBuffers = newDroppedBuffers;

    emitTimeCounterMetrics(emitter, httpPostEmitter.getBatchFillingTimeCounter(), "emitter/batchFilling/");
    emitTimeCounterMetrics(emitter, httpPostEmitter.getSuccessfulSendingTimeCounter(), "emitter/successfulSending/");
    emitTimeCounterMetrics(emitter, httpPostEmitter.getFailedSendingTimeCounter(), "emitter/failedSending/");

    emitter.emit(builder.build("emitter/events/emitQueue", httpPostEmitter.getEventsToEmit()));
    emitter.emit(builder.build("emitter/events/large/emitQueue", httpPostEmitter.getLargeEventsToEmit()));
    emitter.emit(builder.build("emitter/buffers/totalAllocated", httpPostEmitter.getTotalAllocatedBuffers()));
    emitter.emit(builder.build("emitter/buffers/emitQueue", httpPostEmitter.getBuffersToEmit()));
    emitter.emit(builder.build("emitter/buffers/failed", httpPostEmitter.getFailedBuffers()));
    emitter.emit(builder.build("emitter/buffers/reuseQueue", httpPostEmitter.getBuffersToReuse()));

    return true;
  }

  private void emitTimeCounterMetrics(ServiceEmitter emitter, ConcurrentTimeCounter timeCounter, String metricNameBase)
  {
    long timeSumAndCount = timeCounter.getTimeSumAndCountAndReset();
    emitter.emit(builder.build(metricNameBase + "timeMsSum", ConcurrentTimeCounter.timeSum(timeSumAndCount)));
    emitter.emit(builder.build(metricNameBase + "count", ConcurrentTimeCounter.count(timeSumAndCount)));
    emitter.emit(builder.build(metricNameBase + "maxTimeMs", timeCounter.getAndResetMaxTime()));
    emitter.emit(builder.build(metricNameBase + "minTimeMs", timeCounter.getAndResetMinTime()));
  }

  @Override
  protected ServiceMetricEvent.Builder builder()
  {
    ServiceMetricEvent.Builder builder = super.builder();
    extraDimensions.forEach(builder::setDimension);
    return builder;
  }
}
