/*
 * Copyright (c) 2008, 2009, 2011 Oracle, Inc. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.  The Eclipse Public License is available
 * at http://www.eclipse.org/legal/epl-v10.html and the Eclipse Distribution License
 * is available at http://www.eclipse.org/org/documents/edl-v10.php.
 */
package javax.persistence;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.*;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Defines a primary key generator that may be referenced by name when a generator element is specified for
 * the {@link GeneratedValue} annotation. A sequence generator may be specified on the
 * primary key field.
 * <p>
 * <pre>
 *   Example:
 *
 *   &#064;SequenceGenerator(name="EMP_SEQ", allocationSize=25)
 * </pre>
 *
 * @since Java Persistence 1.0
 */
@Target({FIELD})
@Retention(RUNTIME)
public @interface SequenceGenerator {
  /**
   * (Required) A unique generator name that can be referenced by one or more classes to be the generator
   * for primary key values.
   *
   * @return name
   */
  String name();

  /**
   * (Optional) The name of the database sequence object from which to obtain primary key values.
   * <p>
   * Defaults to a provider-chosen value.
   *
   * @return seq name
   */
  String sequenceName() default "";

  /**
   * (Optional) The catalog of the sequence generator.
   *
   * @return catalog
   * @since Java Persistence 2.0
   */
  String catalog() default "";

  /**
   * (Optional) The schema of the sequence generator.
   *
   * @return schema
   * @since Java Persistence 2.0
   */
  String schema() default "";

  /**
   * (Optional) The value from which the sequence object is to start generating.
   *
   * @return init value
   */
  int initialValue() default 1;

  /**
   * (Optional) The amount to increment by when allocating sequence numbers from the sequence.
   *
   * @return alloc size
   */
  int allocationSize() default 50;
}
