/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.cleaner;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.conf.ConfigurationObserver;
import org.apache.hadoop.hbase.io.HFileLink;
import org.apache.hadoop.hbase.master.cleaner.BaseHFileCleanerDelegate;
import org.apache.hadoop.hbase.master.cleaner.CleanerChore;
import org.apache.hadoop.hbase.master.cleaner.DirScanPool;
import org.apache.hadoop.hbase.regionserver.StoreFileInfo;
import org.apache.hadoop.hbase.util.StealJobQueue;

@InterfaceAudience.Private
public class HFileCleaner
extends CleanerChore<BaseHFileCleanerDelegate>
implements ConfigurationObserver {
    public static final String MASTER_HFILE_CLEANER_PLUGINS = "hbase.master.hfilecleaner.plugins";
    public static final String HFILE_DELETE_THROTTLE_THRESHOLD = "hbase.regionserver.thread.hfilecleaner.throttle";
    public static final int DEFAULT_HFILE_DELETE_THROTTLE_THRESHOLD = 0x4000000;
    public static final String LARGE_HFILE_QUEUE_INIT_SIZE = "hbase.regionserver.hfilecleaner.large.queue.size";
    public static final int DEFAULT_LARGE_HFILE_QUEUE_INIT_SIZE = 10240;
    public static final String SMALL_HFILE_QUEUE_INIT_SIZE = "hbase.regionserver.hfilecleaner.small.queue.size";
    public static final int DEFAULT_SMALL_HFILE_QUEUE_INIT_SIZE = 10240;
    public static final String LARGE_HFILE_DELETE_THREAD_NUMBER = "hbase.regionserver.hfilecleaner.large.thread.count";
    public static final int DEFAULT_LARGE_HFILE_DELETE_THREAD_NUMBER = 1;
    public static final String SMALL_HFILE_DELETE_THREAD_NUMBER = "hbase.regionserver.hfilecleaner.small.thread.count";
    public static final int DEFAULT_SMALL_HFILE_DELETE_THREAD_NUMBER = 1;
    public static final String HFILE_DELETE_THREAD_TIMEOUT_MSEC = "hbase.regionserver.hfilecleaner.thread.timeout.msec";
    @VisibleForTesting
    static final long DEFAULT_HFILE_DELETE_THREAD_TIMEOUT_MSEC = 60000L;
    public static final String HFILE_DELETE_THREAD_CHECK_INTERVAL_MSEC = "hbase.regionserver.hfilecleaner.thread.check.interval.msec";
    @VisibleForTesting
    static final long DEFAULT_HFILE_DELETE_THREAD_CHECK_INTERVAL_MSEC = 1000L;
    private static final Log LOG = LogFactory.getLog(HFileCleaner.class);
    StealJobQueue<HFileDeleteTask> largeFileQueue;
    BlockingQueue<HFileDeleteTask> smallFileQueue;
    private int throttlePoint;
    private int largeQueueInitSize;
    private int smallQueueInitSize;
    private int largeFileDeleteThreadNumber;
    private int smallFileDeleteThreadNumber;
    private long cleanerThreadTimeoutMsec;
    private long cleanerThreadCheckIntervalMsec;
    private List<Thread> threads = new ArrayList<Thread>();
    private boolean running;
    private AtomicLong deletedLargeFiles = new AtomicLong();
    private AtomicLong deletedSmallFiles = new AtomicLong();

    public HFileCleaner(int period, Stoppable stopper, Configuration conf, FileSystem fs, Path directory, DirScanPool pool) {
        this(period, stopper, conf, fs, directory, pool, null);
    }

    public HFileCleaner(int period, Stoppable stopper, Configuration conf, FileSystem fs, Path directory, DirScanPool pool, Map<String, Object> params) {
        super("HFileCleaner", period, stopper, conf, fs, directory, MASTER_HFILE_CLEANER_PLUGINS, pool, params);
        this.throttlePoint = conf.getInt(HFILE_DELETE_THROTTLE_THRESHOLD, 0x4000000);
        this.largeQueueInitSize = conf.getInt(LARGE_HFILE_QUEUE_INIT_SIZE, 10240);
        this.smallQueueInitSize = conf.getInt(SMALL_HFILE_QUEUE_INIT_SIZE, 10240);
        this.largeFileQueue = new StealJobQueue(this.largeQueueInitSize, this.smallQueueInitSize);
        this.smallFileQueue = this.largeFileQueue.getStealFromQueue();
        this.largeFileDeleteThreadNumber = conf.getInt(LARGE_HFILE_DELETE_THREAD_NUMBER, 1);
        this.smallFileDeleteThreadNumber = conf.getInt(SMALL_HFILE_DELETE_THREAD_NUMBER, 1);
        this.cleanerThreadTimeoutMsec = conf.getLong(HFILE_DELETE_THREAD_TIMEOUT_MSEC, 60000L);
        this.cleanerThreadCheckIntervalMsec = conf.getLong(HFILE_DELETE_THREAD_CHECK_INTERVAL_MSEC, 1000L);
        this.startHFileDeleteThreads();
    }

    @Override
    protected boolean validate(Path file) {
        if (HFileLink.isBackReferencesDir(file) || HFileLink.isBackReferencesDir(file.getParent())) {
            return true;
        }
        return StoreFileInfo.validateStoreFileName(file.getName());
    }

    public List<BaseHFileCleanerDelegate> getDelegatesForTesting() {
        return this.cleanersChain;
    }

    @Override
    public int deleteFiles(Iterable<FileStatus> filesToDelete) {
        int deletedFiles = 0;
        ArrayList<HFileDeleteTask> tasks = new ArrayList<HFileDeleteTask>();
        for (FileStatus file : filesToDelete) {
            HFileDeleteTask task = this.deleteFile(file);
            if (task == null) continue;
            tasks.add(task);
        }
        for (HFileDeleteTask task : tasks) {
            if (!task.getResult(this.cleanerThreadCheckIntervalMsec)) continue;
            ++deletedFiles;
        }
        return deletedFiles;
    }

    private HFileDeleteTask deleteFile(FileStatus file) {
        HFileDeleteTask task = new HFileDeleteTask(file, this.cleanerThreadTimeoutMsec);
        boolean enqueued = this.dispatch(task);
        return enqueued ? task : null;
    }

    private boolean dispatch(HFileDeleteTask task) {
        if (task.fileLength >= (long)this.throttlePoint) {
            if (!this.largeFileQueue.offer(task)) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)"Large file deletion queue is full");
                }
                return false;
            }
        } else if (!this.smallFileQueue.offer(task)) {
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)"Small file deletion queue is full");
            }
            return false;
        }
        return true;
    }

    @Override
    public void cleanup() {
        super.cleanup();
        this.stopHFileDeleteThreads();
    }

    private void startHFileDeleteThreads() {
        int i;
        String n = Thread.currentThread().getName();
        this.running = true;
        for (i = 0; i < this.largeFileDeleteThreadNumber; ++i) {
            Thread large = new Thread(){

                @Override
                public void run() {
                    HFileCleaner.this.consumerLoop(HFileCleaner.this.largeFileQueue);
                }
            };
            large.setDaemon(true);
            large.setName(n + "-HFileCleaner.large." + i + "-" + System.currentTimeMillis());
            large.start();
            LOG.debug((Object)("Starting hfile cleaner for large files: " + large.getName()));
            this.threads.add(large);
        }
        for (i = 0; i < this.smallFileDeleteThreadNumber; ++i) {
            Thread small = new Thread(){

                @Override
                public void run() {
                    HFileCleaner.this.consumerLoop(HFileCleaner.this.smallFileQueue);
                }
            };
            small.setDaemon(true);
            small.setName(n + "-HFileCleaner.small." + i + "-" + System.currentTimeMillis());
            small.start();
            LOG.debug((Object)("Starting hfile cleaner for small files: " + small.getName()));
            this.threads.add(small);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void consumerLoop(BlockingQueue<HFileDeleteTask> queue) {
        try {
            while (this.running) {
                boolean succeed;
                HFileDeleteTask task = null;
                try {
                    task = queue.take();
                }
                catch (InterruptedException e) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug((Object)"Interrupted while trying to take a task from queue", (Throwable)e);
                    }
                    break;
                }
                if (task == null) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("Removing: " + task.filePath + " from archive"));
                }
                try {
                    succeed = this.fs.delete(task.filePath, false);
                }
                catch (IOException e) {
                    LOG.warn((Object)("Failed to delete file " + task.filePath), (Throwable)e);
                    succeed = false;
                }
                task.setResult(succeed);
                if (!succeed) continue;
                this.countDeletedFiles(task.fileLength >= (long)this.throttlePoint, queue == this.largeFileQueue);
            }
        }
        finally {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Exit thread: " + Thread.currentThread()));
            }
        }
    }

    private void countDeletedFiles(boolean isLargeFile, boolean fromLargeQueue) {
        if (isLargeFile) {
            if (this.deletedLargeFiles.get() == Long.MAX_VALUE) {
                LOG.info((Object)"Deleted more than Long.MAX_VALUE large files, reset counter to 0");
                this.deletedLargeFiles.set(0L);
            }
            this.deletedLargeFiles.incrementAndGet();
        } else {
            if (this.deletedSmallFiles.get() == Long.MAX_VALUE) {
                LOG.info((Object)"Deleted more than Long.MAX_VALUE small files, reset counter to 0");
                this.deletedSmallFiles.set(0L);
            }
            if (fromLargeQueue && LOG.isTraceEnabled()) {
                LOG.trace((Object)"Stolen a small file deletion task in large file thread");
            }
            this.deletedSmallFiles.incrementAndGet();
        }
    }

    private void stopHFileDeleteThreads() {
        this.running = false;
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)"Stopping file delete threads");
        }
        for (Thread thread : this.threads) {
            thread.interrupt();
        }
    }

    @VisibleForTesting
    public List<Thread> getCleanerThreads() {
        return this.threads;
    }

    @VisibleForTesting
    public long getNumOfDeletedLargeFiles() {
        return this.deletedLargeFiles.get();
    }

    @VisibleForTesting
    public long getNumOfDeletedSmallFiles() {
        return this.deletedSmallFiles.get();
    }

    @VisibleForTesting
    public long getLargeQueueInitSize() {
        return this.largeQueueInitSize;
    }

    @VisibleForTesting
    public long getSmallQueueInitSize() {
        return this.smallQueueInitSize;
    }

    @VisibleForTesting
    public long getThrottlePoint() {
        return this.throttlePoint;
    }

    @VisibleForTesting
    long getCleanerThreadTimeoutMsec() {
        return this.cleanerThreadTimeoutMsec;
    }

    @VisibleForTesting
    long getCleanerThreadCheckIntervalMsec() {
        return this.cleanerThreadCheckIntervalMsec;
    }

    @Override
    public void onConfigurationChange(Configuration conf) {
        if (!this.checkAndUpdateConfigurations(conf)) {
            LOG.debug((Object)"Update configuration triggered but nothing changed for this cleaner");
            return;
        }
        this.stopHFileDeleteThreads();
        ArrayList<HFileDeleteTask> leftOverTasks = new ArrayList<HFileDeleteTask>(this.largeFileQueue.size() + this.smallFileQueue.size());
        for (HFileDeleteTask task : this.largeFileQueue) {
            leftOverTasks.add(task);
        }
        for (HFileDeleteTask task : this.smallFileQueue) {
            leftOverTasks.add(task);
        }
        this.largeFileQueue = new StealJobQueue(this.largeQueueInitSize, this.smallQueueInitSize);
        this.smallFileQueue = this.largeFileQueue.getStealFromQueue();
        this.threads.clear();
        this.startHFileDeleteThreads();
        for (HFileDeleteTask task : leftOverTasks) {
            this.dispatch(task);
        }
    }

    private boolean checkAndUpdateConfigurations(Configuration conf) {
        long cleanerThreadCheckIntervalMsec;
        long cleanerThreadTimeoutMsec;
        int smallFileDeleteThreadNumber;
        int largeFileDeleteThreadNumber;
        int smallQueueInitSize;
        int largeQueueInitSize;
        boolean updated = false;
        int throttlePoint = conf.getInt(HFILE_DELETE_THROTTLE_THRESHOLD, 0x4000000);
        if (throttlePoint != this.throttlePoint) {
            LOG.debug((Object)("Updating throttle point, from " + this.throttlePoint + " to " + throttlePoint));
            this.throttlePoint = throttlePoint;
            updated = true;
        }
        if ((largeQueueInitSize = conf.getInt(LARGE_HFILE_QUEUE_INIT_SIZE, 10240)) != this.largeQueueInitSize) {
            LOG.debug((Object)("Updating largeQueueInitSize, from " + this.largeQueueInitSize + " to " + largeQueueInitSize));
            this.largeQueueInitSize = largeQueueInitSize;
            updated = true;
        }
        if ((smallQueueInitSize = conf.getInt(SMALL_HFILE_QUEUE_INIT_SIZE, 10240)) != this.smallQueueInitSize) {
            LOG.debug((Object)("Updating smallQueueInitSize, from " + this.smallQueueInitSize + " to " + smallQueueInitSize));
            this.smallQueueInitSize = smallQueueInitSize;
            updated = true;
        }
        if ((largeFileDeleteThreadNumber = conf.getInt(LARGE_HFILE_DELETE_THREAD_NUMBER, 1)) != this.largeFileDeleteThreadNumber) {
            LOG.debug((Object)("Updating largeFileDeleteThreadNumber, from " + this.largeFileDeleteThreadNumber + " to " + largeFileDeleteThreadNumber));
            this.largeFileDeleteThreadNumber = largeFileDeleteThreadNumber;
            updated = true;
        }
        if ((smallFileDeleteThreadNumber = conf.getInt(SMALL_HFILE_DELETE_THREAD_NUMBER, 1)) != this.smallFileDeleteThreadNumber) {
            LOG.debug((Object)("Updating smallFileDeleteThreadNumber, from " + this.smallFileDeleteThreadNumber + " to " + smallFileDeleteThreadNumber));
            this.smallFileDeleteThreadNumber = smallFileDeleteThreadNumber;
            updated = true;
        }
        if ((cleanerThreadTimeoutMsec = conf.getLong(HFILE_DELETE_THREAD_TIMEOUT_MSEC, 60000L)) != this.cleanerThreadTimeoutMsec) {
            this.cleanerThreadTimeoutMsec = cleanerThreadTimeoutMsec;
            updated = true;
        }
        if ((cleanerThreadCheckIntervalMsec = conf.getLong(HFILE_DELETE_THREAD_CHECK_INTERVAL_MSEC, 1000L)) != this.cleanerThreadCheckIntervalMsec) {
            this.cleanerThreadCheckIntervalMsec = cleanerThreadCheckIntervalMsec;
            updated = true;
        }
        return updated;
    }

    static class HFileDeleteTask
    implements Comparable<HFileDeleteTask> {
        boolean done = false;
        boolean result;
        final Path filePath;
        final long fileLength;
        final long timeoutMsec;

        public HFileDeleteTask(FileStatus file, long timeoutMsec) {
            this.filePath = file.getPath();
            this.fileLength = file.getLen();
            this.timeoutMsec = timeoutMsec;
        }

        public synchronized void setResult(boolean result) {
            this.done = true;
            this.result = result;
            this.notify();
        }

        public synchronized boolean getResult(long waitIfNotFinished) {
            long waitTimeMsec = 0L;
            try {
                while (!this.done) {
                    long startTimeNanos = System.nanoTime();
                    this.wait(waitIfNotFinished);
                    waitTimeMsec += TimeUnit.MILLISECONDS.convert(System.nanoTime() - startTimeNanos, TimeUnit.NANOSECONDS);
                    if (this.done) {
                        return this.result;
                    }
                    if (waitTimeMsec <= this.timeoutMsec) continue;
                    LOG.warn((Object)("Wait more than " + this.timeoutMsec + " ms for deleting " + this.filePath + ", exit..."));
                    return false;
                }
            }
            catch (InterruptedException e) {
                LOG.warn((Object)("Interrupted while waiting for result of deleting " + this.filePath + ", will return false"), (Throwable)e);
                return false;
            }
            return this.result;
        }

        @Override
        public int compareTo(HFileDeleteTask o) {
            long sub = this.fileLength - o.fileLength;
            return sub > 0L ? -1 : (sub < 0L ? 1 : 0);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof HFileDeleteTask)) {
                return false;
            }
            HFileDeleteTask otherTask = (HFileDeleteTask)o;
            return this.filePath.equals((Object)otherTask.filePath) && this.fileLength == otherTask.fileLength;
        }

        public int hashCode() {
            return this.filePath.hashCode();
        }
    }
}

