/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class LossyCounting<T> {
    private static final Logger LOG = LoggerFactory.getLogger(LossyCounting.class);
    private final ExecutorService executor;
    private long bucketSize;
    private int currentTerm;
    private Map<T, Integer> data;
    private long totalDataCount;
    private final String name;
    private LossyCountingListener<T> listener;
    private static AtomicReference<Future<?>> fut = new AtomicReference<Object>(null);

    LossyCounting(String name, double errorRate) {
        this(name, errorRate, null);
    }

    public LossyCounting(String name, double errorRate, LossyCountingListener<T> listener) {
        if (errorRate < 0.0 || errorRate > 1.0) {
            throw new IllegalArgumentException(" Lossy Counting error rate should be within range [0,1]");
        }
        this.name = name;
        this.bucketSize = (long)Math.ceil(1.0 / errorRate);
        this.currentTerm = 1;
        this.totalDataCount = 0L;
        this.data = new ConcurrentHashMap<T, Integer>();
        this.listener = listener;
        this.calculateCurrentTerm();
        this.executor = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("lossy-count-%d").build());
    }

    LossyCounting(String name, Configuration conf) {
        this(name, conf, null);
    }

    public LossyCounting(String name, Configuration conf, LossyCountingListener<T> listener) {
        this(name, conf.getDouble("hbase.util.default.lossycounting.errorrate", 0.02), listener);
    }

    private void addByOne(T key) {
        Integer i = this.data.get(key);
        if (i == null) {
            i = this.currentTerm != 0 ? this.currentTerm - 1 : 0;
        }
        this.data.put(key, i + 1);
        ++this.totalDataCount;
        this.calculateCurrentTerm();
    }

    public void add(T key) {
        this.addByOne(key);
        if (this.totalDataCount % this.bucketSize == 0L) {
            Future<?> future = fut.get();
            if (future != null && !future.isDone()) {
                return;
            }
            future = this.executor.submit(new SweepRunnable());
            fut.set(future);
        }
    }

    @VisibleForTesting
    public void sweep() {
        for (Map.Entry<T, Integer> entry : this.data.entrySet()) {
            if (entry.getValue() >= this.currentTerm) continue;
            T metric = entry.getKey();
            this.data.remove(metric);
            if (this.listener == null) continue;
            this.listener.sweep(metric);
        }
    }

    private void calculateCurrentTerm() {
        this.currentTerm = (int)Math.ceil(1.0 * (double)this.totalDataCount / (double)this.bucketSize);
    }

    public long getBucketSize() {
        return this.bucketSize;
    }

    public long getDataSize() {
        return this.data.size();
    }

    public boolean contains(T key) {
        return this.data.containsKey(key);
    }

    public Set<T> getElements() {
        return this.data.keySet();
    }

    public long getCurrentTerm() {
        return this.currentTerm;
    }

    @VisibleForTesting
    public Future<?> getSweepFuture() {
        return fut.get();
    }

    class SweepRunnable
    implements Runnable {
        SweepRunnable() {
        }

        @Override
        public void run() {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Starting sweep of lossyCounting-" + LossyCounting.this.name);
            }
            try {
                LossyCounting.this.sweep();
            }
            catch (Exception exception) {
                LOG.debug("Error while sweeping lossyCounting-" + LossyCounting.this.name, (Throwable)exception);
            }
        }
    }

    public static interface LossyCountingListener<T> {
        public void sweep(T var1);
    }
}

