/*
 * Decompiled with CFR 0.152.
 */
package org.apache.twill.internal.zookeeper;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import com.google.common.util.concurrent.Uninterruptibles;
import java.util.UUID;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nullable;
import org.apache.twill.common.Cancellable;
import org.apache.twill.common.Threads;
import org.apache.twill.zookeeper.NodeChildren;
import org.apache.twill.zookeeper.NodeData;
import org.apache.twill.zookeeper.OperationFuture;
import org.apache.twill.zookeeper.ZKClient;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReentrantDistributedLock
implements Lock {
    private static final Logger LOG = LoggerFactory.getLogger(ReentrantDistributedLock.class);
    private final ZKClient zkClient;
    private final String path;
    private final ThreadLocal<String> localLockNode;
    private final ReentrantLock lock;

    public ReentrantDistributedLock(ZKClient zkClient, String path) {
        this.zkClient = zkClient;
        this.path = path.startsWith("/") ? path : "/" + path;
        this.localLockNode = new ThreadLocal();
        this.lock = new ReentrantLock();
    }

    @Override
    public void lock() {
        this.lock.lock();
        try {
            this.acquire(false, true);
        }
        catch (Exception e) {
            this.lock.unlock();
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public void lockInterruptibly() throws InterruptedException {
        this.lock.lockInterruptibly();
        try {
            this.acquire(true, true);
        }
        catch (Exception e) {
            this.lock.unlock();
            Throwables.propagateIfInstanceOf((Throwable)e, InterruptedException.class);
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public boolean tryLock() {
        if (!this.lock.tryLock()) {
            return false;
        }
        try {
            if (this.acquire(false, false)) {
                return true;
            }
            this.lock.unlock();
            return false;
        }
        catch (Exception e) {
            this.lock.unlock();
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public boolean tryLock(long time, TimeUnit unit) throws InterruptedException {
        long startTime = System.nanoTime();
        if (!this.lock.tryLock(time, unit)) {
            return false;
        }
        long timeoutNano = unit.toNanos(time) - (System.nanoTime() - startTime);
        try {
            if (this.acquire(true, true, timeoutNano, TimeUnit.NANOSECONDS)) {
                return true;
            }
            this.lock.unlock();
            return false;
        }
        catch (ExecutionException e) {
            this.lock.unlock();
            throw Throwables.propagate((Throwable)e.getCause());
        }
        catch (TimeoutException e) {
            this.lock.unlock();
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void unlock() {
        block9: {
            if (!this.lock.isHeldByCurrentThread()) {
                throw new IllegalStateException("Cannot unlock without holding a lock by thread " + Thread.currentThread());
            }
            try {
                if (this.lock.getHoldCount() != 1) break block9;
                try {
                    Uninterruptibles.getUninterruptibly(this.zkClient.delete(this.localLockNode.get()));
                }
                catch (ExecutionException e) {
                    throw Throwables.propagate((Throwable)e.getCause());
                }
                finally {
                    this.localLockNode.remove();
                }
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    @Override
    public Condition newCondition() {
        throw new UnsupportedOperationException("Condition not supported.");
    }

    private boolean acquire(boolean interruptible, boolean waitForLock) throws InterruptedException, ExecutionException {
        try {
            return this.acquire(interruptible, waitForLock, Long.MAX_VALUE, TimeUnit.SECONDS);
        }
        catch (TimeoutException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private boolean acquire(boolean interruptible, final boolean waitForLock, long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread(), (Object)"Not owner of local lock.");
        if (this.lock.getHoldCount() > 1) {
            return true;
        }
        final SettableFuture completion = SettableFuture.create();
        final Cancellable watcherCancellable = this.zkClient.addConnectionWatcher(new Watcher(){

            public void process(WatchedEvent event) {
                if (event.getState() == Watcher.Event.KeeperState.Expired) {
                    completion.setException((Throwable)new IllegalStateException("ZK session expired"));
                }
            }
        });
        completion.addListener(new Runnable(){

            @Override
            public void run() {
                watcherCancellable.cancel();
            }
        }, Threads.SAME_THREAD_EXECUTOR);
        final String guid = UUID.randomUUID().toString();
        final String lockPath = String.format("%s/%s-", this.path, guid);
        OperationFuture<String> future = this.zkClient.create(lockPath, null, CreateMode.EPHEMERAL_SEQUENTIAL, true);
        Futures.addCallback(future, (FutureCallback)new FutureCallback<String>(){

            public void onSuccess(String lockNode) {
                ReentrantDistributedLock.this.deleteNodeOnFailure((ListenableFuture)completion, lockNode);
                if (completion.isDone()) {
                    return;
                }
                ReentrantDistributedLock.this.doAcquire((SettableFuture<String>)completion, waitForLock, guid, lockNode);
            }

            public void onFailure(Throwable t) {
                if (t instanceof KeeperException.ConnectionLossException) {
                    ReentrantDistributedLock.this.doAcquire((SettableFuture<String>)completion, waitForLock, guid, null);
                } else {
                    LOG.error("Exception raised when creating lock node at {}", (Object)lockPath, (Object)t);
                    completion.setException(t);
                }
            }
        });
        try {
            if (interruptible) {
                this.localLockNode.set((String)completion.get(timeout, unit));
            } else {
                this.localLockNode.set((String)Uninterruptibles.getUninterruptibly((Future)completion, (long)timeout, (TimeUnit)unit));
            }
            return true;
        }
        catch (InterruptedException e) {
            completion.cancel(true);
            throw e;
        }
        catch (TimeoutException e) {
            completion.cancel(true);
            throw e;
        }
        catch (CancellationException e) {
            return false;
        }
    }

    private void doAcquire(final SettableFuture<String> completion, final boolean waitForLock, final String guid, final @Nullable String lockPath) {
        Futures.addCallback(this.zkClient.getChildren(this.path), (FutureCallback)new FutureCallback<NodeChildren>(){

            public void onSuccess(NodeChildren children) {
                String nodeToWatch;
                String lockNode;
                String string = lockNode = lockPath == null ? ReentrantDistributedLock.this.findLockNode(children.getChildren(), guid) : lockPath;
                if (lockNode == null) {
                    completion.setException(new IllegalStateException("Failed to acquire lock").fillInStackTrace());
                    return;
                }
                if (lockPath == null) {
                    ReentrantDistributedLock.this.deleteNodeOnFailure((ListenableFuture)completion, lockNode);
                }
                if ((nodeToWatch = ReentrantDistributedLock.this.findNodeToWatch(children, lockNode, guid)) == null) {
                    completion.set((Object)lockNode);
                } else if (!waitForLock) {
                    completion.cancel(true);
                }
                if (completion.isDone()) {
                    return;
                }
                OperationFuture<NodeData> getDataFuture = ReentrantDistributedLock.this.zkClient.getData(nodeToWatch, new Watcher(){

                    public void process(WatchedEvent event) {
                        if (!completion.isDone()) {
                            ReentrantDistributedLock.this.doAcquire((SettableFuture<String>)completion, waitForLock, guid, lockNode);
                        }
                    }
                });
                Futures.addCallback(getDataFuture, (FutureCallback)new FutureCallback<NodeData>(){

                    public void onSuccess(NodeData nodeData) {
                    }

                    public void onFailure(Throwable t) {
                        if (t instanceof KeeperException.NoNodeException && !completion.isDone()) {
                            ReentrantDistributedLock.this.doAcquire((SettableFuture<String>)completion, waitForLock, guid, lockNode);
                        } else {
                            completion.setException(t);
                        }
                    }
                });
            }

            public void onFailure(Throwable t) {
                if (lockPath != null) {
                    completion.setException(t);
                } else {
                    ReentrantDistributedLock.this.doAcquire((SettableFuture<String>)completion, waitForLock, guid, null);
                }
            }
        });
    }

    private void deleteNodeOnFailure(final ListenableFuture<?> future, final String node) {
        future.addListener(new Runnable(){

            @Override
            public void run() {
                try {
                    future.get();
                }
                catch (Exception e) {
                    ReentrantDistributedLock.this.zkClient.delete(node);
                }
            }
        }, Threads.SAME_THREAD_EXECUTOR);
    }

    private String findNodeToWatch(NodeChildren children, String lockPath, String guid) {
        int guidLen = guid.length();
        int id = Integer.parseInt(lockPath.substring(this.path.length() + guidLen + 2));
        String nodeToWatch = null;
        int maxOfMins = Integer.MIN_VALUE;
        for (String node : children.getChildren()) {
            int nodeId = Integer.parseInt(node.substring(guidLen + 1));
            if (nodeId >= id || nodeId <= maxOfMins) continue;
            maxOfMins = nodeId;
            nodeToWatch = this.path + "/" + node;
        }
        return nodeToWatch;
    }

    private String findLockNode(Iterable<String> nodes, String prefix) {
        for (String child : nodes) {
            if (!child.startsWith(prefix)) continue;
            return this.path + "/" + child;
        }
        return null;
    }
}

