package io.embrace.android.embracesdk;

import com.google.gson.annotations.SerializedName;

/**
 * Custom breadcrumbs can be created to reflect events into the user timeline.
 * The max number of characters for this breadcrumb message is
 * {@link CustomBreadcrumb#BREADCRUMB_MESSAGE_MAX_LENGTH}
 */
class CustomBreadcrumb implements Breadcrumb {

    private static final int BREADCRUMB_MESSAGE_MAX_LENGTH = 256;

    /**
     * Message for the custom breadcrumb event.
     * If the message exceeds the {@link CustomBreadcrumb#BREADCRUMB_MESSAGE_MAX_LENGTH} characters
     * it will be ellipsized.
     */
    @SerializedName("m")
    final String message;

    /**
     * The timestamp at which the event occurred.
     */
    @SerializedName("ts")
    private final long timestamp;

    CustomBreadcrumb(String message, long timestamp) {
        this.timestamp = timestamp;
        this.message = ellipsizeBreadcrumbMessage(message);
    }

    private static String ellipsizeBreadcrumbMessage(String input) {
        if (input == null || input.length() < BREADCRUMB_MESSAGE_MAX_LENGTH) {
            return input;
        }
        return input.substring(0, BREADCRUMB_MESSAGE_MAX_LENGTH - 3) + "...";
    }

    @Override
    public long getStartTime() {
        return timestamp;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        CustomBreadcrumb that = (CustomBreadcrumb) o;

        if (timestamp != that.timestamp) {
            return false;
        }
        return message != null ? message.equals(that.message) : that.message == null;
    }

    @Override
    public int hashCode() {
        int result = message != null ? message.hashCode() : 0;
        result = 31 * result + (int) (timestamp ^ (timestamp >>> 32));
        return result;
    }
}
