package io.embrace.android.embracesdk;

import static io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger.logger;

import android.content.Context;
import android.webkit.ConsoleMessage;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.jetbrains.annotations.NotNull;

import java.util.Map;

import io.embrace.android.embracesdk.config.ConfigService;
import io.embrace.android.embracesdk.internal.utils.ThrowableUtilsKt;
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.network.EmbraceNetworkRequest;
import io.embrace.android.embracesdk.network.EmbraceNetworkRequestV2;
import io.embrace.android.embracesdk.network.http.HttpMethod;
import io.embrace.android.embracesdk.network.http.NetworkCaptureData;

/**
 * Entry point for the SDK. This class is part of the Embrace Public API.
 * <p>
 * Contains a singleton instance of itself, and is used for initializing the SDK.
 */
@SuppressWarnings("unused")
public final class Embrace implements EmbraceAndroidApi {

    /**
     * Singleton instance of the Embrace SDK.
     */
    private static final Embrace embrace = new Embrace();
    private static EmbraceImpl impl = new EmbraceImpl();

    /**
     * Gets the singleton instance of the Embrace SDK.
     *
     * @return the instance of the Embrace SDK
     */
    @NonNull
    public static Embrace getInstance() {
        return embrace;
    }

    /**
     * Gets the EmbraceImpl instance. This provides access to internally visible functions
     * intended for use in the Android SDK only
     */
    @NonNull
    static EmbraceImpl getImpl() {
        return impl;
    }

    static void setImpl(@Nullable EmbraceImpl instance) {
        impl = instance;
    }

    /**
     * Capture stacktraces during startup to better understand what is happening during startup.
     * <p>
     * This API is experimental and subject to change.
     */
    @Deprecated
    public static void enableStartupTracing(@NonNull Context context) {
    }

    /**
     * Starts capturing ANRs before the rest of the Embrace SDK has initialized. This allows
     * you to get insight into what is causing ANRs early on in an application.
     * <p>
     * This API is experimental and subject to change.
     */
    @Deprecated
    public static void enableEarlyAnrCapture(@NonNull Context context) {
    }

    @Override
    public void start(@NonNull Context context) {
        start(context, true, AppFramework.NATIVE);
    }

    @Override
    public void start(@NonNull Context context, boolean enableIntegrationTesting) {
        start(context, enableIntegrationTesting, AppFramework.NATIVE);
    }

    @Override
    public void start(@NonNull Context context, boolean enableIntegrationTesting, @NonNull AppFramework appFramework) {
        impl.start(context, enableIntegrationTesting, appFramework);
    }

    /**
     * Whether or not the SDK has been started.
     *
     * @return true if the SDK is started, false otherwise
     */
    public boolean isStarted() {
        return impl.isStarted();
    }

    /**
     * Sets a custom app ID that overrides the one specified at build time. Must be called before
     * the SDK is started.
     *
     * @param appId custom app ID
     * @return true if the app ID could be set, false otherwise.
     */
    public boolean setAppId(@NonNull String appId) {
        return impl.setAppId(appId);
    }

    /**
     * This method sets a logging level, but this logging level is never used.
     *
     * @param severity the severity
     * @deprecated as the log level is never used. Use {@link EmbraceLogger}.
     */
    @Deprecated
    public void setLogLevel(@NonNull EmbraceLogger.Severity severity) {
        // This method does not do anything and is purely retained for backwards-compatibility
    }

    /**
     * This method enables debug logging.
     */
    @Deprecated
    public void enableDebugLogging() {
        // This method does not do anything and is purely retained for backwards-compatibility
    }

    /**
     * This method disables debug logging.
     */
    @Deprecated
    public void disableDebugLogging() {
        // This method does not do anything and is purely retained for backwards-compatibility
    }

    @Override
    public void setUserIdentifier(@Nullable String userId) {
        impl.setUserIdentifier(userId);
    }

    @Override
    public void clearUserIdentifier() {
        impl.clearUserIdentifier();
    }

    @Override
    public void setUserEmail(@Nullable String email) {
        impl.setUserEmail(email);
    }

    @Override
    public void clearUserEmail() {
        impl.clearUserEmail();
    }

    @Override
    public void setUserAsPayer() {
        impl.setUserAsPayer();
    }

    @Override
    public void clearUserAsPayer() {
        impl.clearUserAsPayer();
    }

    @Override
    public void setUserPersona(@NonNull String persona) {
        impl.setUserPersona(persona);
    }

    @Override
    public void clearUserPersona(@NonNull String persona) {
        impl.clearUserPersona(persona);
    }

    @Override
    public void clearAllUserPersonas() {
        impl.clearAllUserPersonas();
    }

    @Override
    public boolean addSessionProperty(@NonNull String key, @NonNull String value, boolean permanent) {
        return impl.addSessionProperty(key, value, permanent);
    }

    @Override
    public boolean removeSessionProperty(@NonNull String key) {
        return impl.removeSessionProperty(key);
    }

    @Override
    @Nullable
    public Map<String, String> getSessionProperties() {
        return impl.getSessionProperties();
    }

    @Override
    public void setUsername(@Nullable String username) {
        impl.setUsername(username);
    }

    @Override
    public void clearUsername() {
        impl.clearUsername();
    }

    @Override
    public void startEvent(@NonNull String name) {
        startEvent(name, null, false, null);
    }

    @Override
    public void startEvent(@NonNull String name, @Nullable String identifier) {
        startEvent(name, identifier, false, null);
    }

    @Override
    public void startEvent(@NonNull String name, @Nullable String identifier, boolean allowScreenshot) {
        startEvent(name, identifier, allowScreenshot, null);
    }

    @Override
    public void startEvent(@NonNull String name,
                           @Nullable String identifier,
                           @Nullable Map<String, Object> properties) {
        startEvent(name, identifier, false, properties);
    }

    @Override
    public void startEvent(@NonNull String name,
                           @Nullable String identifier,
                           boolean allowScreenshot,
                           @Nullable Map<String, Object> properties) {
        impl.startEvent(name, identifier, allowScreenshot, properties);
    }

    @Override
    public void endEvent(@NonNull String name) {
        endEvent(name, null, null);
    }

    @Override
    public void endEvent(@NonNull String name, @Nullable String identifier) {
        endEvent(name, identifier, null);
    }

    @Override
    public void endEvent(@NonNull String name, @Nullable Map<String, Object> properties) {
        endEvent(name, null, properties);
    }

    @Override
    public void endEvent(@NonNull String name, @Nullable String identifier, @Nullable Map<String, Object> properties) {
        impl.endEvent(name, identifier, properties);
    }

    @Override
    public void endAppStartup() {
        impl.endAppStartup(null);
    }

    @Override
    public void endAppStartup(@NonNull Map<String, Object> properties) {
        impl.endAppStartup(properties);
    }

    /**
     * Retrieve the HTTP request header to extract trace ID from.
     *
     * @return the Trace ID header.
     */
    @NonNull
    public String getTraceIdHeader() {
        return impl.getTraceIdHeader();
    }

    /**
     * Manually logs a network request.
     *
     * @param url           the URL of the network call
     * @param httpMethod    the int value of the HTTP method of the network call
     * @param startTime     the time that the network call started
     * @param endTime       the time that the network call was completed
     * @param bytesSent     the number of bytes sent as part of the network call
     * @param bytesReceived the number of bytes returned by the server in response to the network call
     * @param statusCode    the status code returned by the server
     * @param error         the error returned by the exception
     */
    public void logNetworkRequest(@NonNull String url,
                                  int httpMethod,
                                  long startTime,
                                  long endTime,
                                  int bytesSent,
                                  int bytesReceived,
                                  int statusCode,
                                  @Nullable String error) {
        impl.logNetworkRequest(url, httpMethod, startTime, endTime, bytesSent, bytesReceived, statusCode, error);
    }

    /**
     * Manually logs a network request.
     *
     * @param request An EmbraceNetworkRequestV2 with at least the following set: url, method, start time,
     *                end time, and either status code or error
     */
    public void logNetworkRequest(@NonNull EmbraceNetworkRequestV2 request) {
        impl.logNetworkRequest(request);
    }

    /**
     * Manually logs a network request.
     *
     * @param request An EmbraceNetworkRequest with at least the following set: url, method, start time,
     *                end time, and either status code or error
     */
    public void logNetworkRequest(@NonNull EmbraceNetworkRequest request) {
        impl.logNetworkRequest(request);
    }

    /**
     * Logs the fact that a network call occurred. These are recorded and sent to Embrace as part
     * of a particular session.
     *
     * @param url           the URL of the network call
     * @param httpMethod    the HTTP method of the network call
     * @param statusCode    the status code returned by the server
     * @param startTime     the time that the network call started
     * @param endTime       the time that the network call was completed
     * @param bytesSent     the number of bytes sent as part of the network call
     * @param bytesReceived the number of bytes returned by the server in response to the network call
     */
    public void logNetworkCall(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        int statusCode,
        long startTime,
        long endTime,
        long bytesSent,
        long bytesReceived) {
        logNetworkCall(url, httpMethod, statusCode, startTime, endTime, bytesSent, bytesReceived, null, null);
    }

    @Override
    public void logNetworkCall(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        int statusCode,
        long startTime,
        long endTime,
        long bytesSent,
        long bytesReceived,
        @Nullable String traceId) {
        impl.logNetworkCall(url, httpMethod, statusCode, startTime, endTime, bytesSent, bytesReceived, traceId, null);
    }

    @Override
    public void logNetworkCall(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        int statusCode,
        long startTime,
        long endTime,
        long bytesSent,
        long bytesReceived,
        @Nullable String traceId,
        @Nullable NetworkCaptureData networkCaptureData) {
        impl.logNetworkCall(url, httpMethod, statusCode, startTime, endTime, bytesSent, bytesReceived, traceId, networkCaptureData);
    }

    /**
     * Logs the fact that an exception was thrown when attempting to make a network call.
     * <p>
     * These are client-side exceptions and not server-side exceptions, such as a DNS error or
     * failure to connect to the remote server.
     *
     * @param url          the URL of the network call
     * @param httpMethod   the HTTP method of the network call
     * @param startTime    the time that the network call started
     * @param endTime      the time that the network call was completed
     * @param errorType    the type of the exception
     * @param errorMessage the message returned by the exception
     */
    public void logNetworkClientError(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        long startTime,
        long endTime,
        @NonNull String errorType,
        @NonNull String errorMessage) {
        logNetworkClientError(url, httpMethod, startTime, endTime, errorType, errorMessage, null, null);
    }

    @Override
    public void logNetworkClientError(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        long startTime,
        long endTime,
        @NonNull String errorType,
        @NonNull String errorMessage,
        @Nullable String traceId) {
        impl.logNetworkClientError(url, httpMethod, startTime, endTime, errorType, errorMessage, traceId, null);
    }

    @Override
    public void logNetworkClientError(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        long startTime,
        long endTime,
        @NonNull String errorType,
        @NonNull String errorMessage,
        @Nullable String traceId,
        @Nullable NetworkCaptureData networkCaptureData) {
        impl.logNetworkClientError(url, httpMethod, startTime, endTime, errorType, errorMessage, traceId, networkCaptureData);
    }

    @Override
    public void logInfo(@NonNull String message) {
        logInfo(message, null);
    }

    @Override
    public void logInfo(@NonNull String message, @Nullable Map<String, Object> properties) {
        impl.logMessage(EmbraceEvent.Type.INFO_LOG, message, properties, false, null, null, LogExceptionType.NONE, null, null);
    }

    @Override
    public void logWarning(@NonNull String message) {
        logWarning(message, null, false, null);
    }

    @Override
    public void logWarning(@NonNull String message, @Nullable Map<String, Object> properties) {
        logWarning(message, properties, false, null);
    }

    @Override
    public void logWarning(@NonNull String message, @Nullable Map<String, Object> properties, boolean allowScreenshot) {
        logWarning(message, properties, allowScreenshot, null);
    }

    @Override
    public void logWarning(@NonNull String message,
                           @Nullable Map<String, Object> properties,
                           boolean allowScreenshot,
                           @Nullable String javascriptStackTrace) {
        impl.logMessage(
            EmbraceEvent.Type.WARNING_LOG,
            message,
            properties,
            allowScreenshot,
            null,
            javascriptStackTrace,
            LogExceptionType.NONE,
            null,
            null);
    }

    @Override
    public void logError(@NonNull String message) {
        logError(message, null, true, null);
    }

    @Override
    public void logError(@NonNull String message, @Nullable Map<String, Object> properties) {
        logError(message, properties, true, null);
    }

    @Override
    public void logError(@NonNull String message, @Nullable Map<String, Object> properties, boolean allowScreenshot) {
        logError(message, properties, allowScreenshot, null);
    }

    @Override
    public void logError(@NonNull String message,
                         @Nullable Map<String, Object> properties,
                         boolean allowScreenshot,
                         @Nullable String javascriptStackTrace) {
        impl.logMessage(
            EmbraceEvent.Type.ERROR_LOG,
            message,
            properties,
            allowScreenshot,
            null,
            javascriptStackTrace,
            LogExceptionType.NONE,
            null,
            null);
    }

    @Override
    public void logError(@NonNull String message,
                         @Nullable Map<String, Object> properties,
                         boolean allowScreenshot,
                         @Nullable String javascriptStackTrace,
                         boolean isException) {
        impl.logMessage(EmbraceEvent.Type.ERROR_LOG,
            message,
            properties,
            allowScreenshot,
            null,
            javascriptStackTrace,
            LogExceptionType.NONE,
            null,
            null);
    }

    @Override
    public void logError(@NonNull Throwable e) {
        logError(e, null, false);
    }

    @Override
    public void logError(@NonNull Throwable e, @Nullable Map<String, Object> properties) {
        logError(e, properties, false);
    }

    @Override
    public void logError(@NonNull Throwable e, @Nullable Map<String, Object> properties, boolean allowScreenshot) {
        logError(e, e.getLocalizedMessage() != null ? e.getLocalizedMessage() : "", properties, allowScreenshot);
    }

    @Override
    public void logError(@NonNull Throwable e, @NonNull String message, @Nullable Map<String, Object> properties, boolean allowScreenshot) {
        impl.logMessage(EmbraceEvent.Type.ERROR_LOG,
            message,
            properties,
            allowScreenshot,
            ThrowableUtilsKt.getSafeStackTrace(e),
            null,
            LogExceptionType.NONE,
            null,
            null,
            e.getClass().getSimpleName(),
            e.getMessage());
    }

    /**
     * Logs a React Native Redux Action.
     */
    public void logRnAction(@NonNull String name, long startTime, long endTime,
                            @NonNull Map<String, Object> properties, int bytesSent, @NonNull String output) {
        impl.logRnAction(name, startTime, endTime, properties, bytesSent, output);
    }

    @Override
    public void logBreadcrumb(@NonNull String message) {
        impl.logBreadcrumb(message);
    }

    @Override
    public void logHandledException(@NonNull Throwable throwable, @NonNull LogType type) {
        logHandledException(throwable, type, null, null, false);
    }

    @Override
    public void logHandledException(@NonNull Throwable throwable, @NonNull LogType type, @NonNull StackTraceElement[] customStackTrace) {
        logHandledException(throwable, type, null, customStackTrace, false);
    }

    @Override
    public void logHandledException(@NonNull Throwable throwable, @NonNull LogType type, @NonNull Map<String, Object> properties) {
        logHandledException(throwable, type, properties, null, false);
    }

    @SuppressWarnings("ConstantConditions")
    @Override
    public void logHandledException(@NonNull Throwable throwable,
                                    @NonNull LogType type,
                                    @Nullable Map<String, Object> properties,
                                    @Nullable StackTraceElement[] customStackTrace,
                                    boolean takeScreenshot) {
        if (throwable == null) {
            InternalStaticEmbraceLogger.logWarning("Throwable must not be null.");
        }
        if (type == null) {
            InternalStaticEmbraceLogger.logWarning("Type must not be null.");
        }

        EmbraceEvent.Type eventType = type.toEventType();

        impl.logMessage(eventType,
            throwable.getMessage() != null ? throwable.getMessage() : "",
            properties,
            takeScreenshot,
            customStackTrace != null ? customStackTrace : throwable.getStackTrace(),
            null,
            LogExceptionType.NONE,
            null,
            null,
            throwable.getClass().getSimpleName(),
            throwable.getMessage());
    }

    /**
     * Logs a javascript unhandled exception.
     *
     * @param name       name of the exception.
     * @param message    exception message.
     * @param type       error type.
     * @param stacktrace exception stacktrace.
     */
    @InternalApi
    @Deprecated
    public void logUnhandledJsException(@NonNull String name,
                                        @NonNull String message,
                                        @Nullable String type,
                                        @Nullable String stacktrace) {
        impl.logUnhandledJsException(name, message, type, stacktrace);
    }

    /**
     * Logs a Unity unhandled exception.
     *
     * @param message    exception message.
     * @param stacktrace exception stacktrace.
     */
    @InternalApi
    @Deprecated
    public void logUnhandledUnityException(@NonNull String message, @Nullable String stacktrace) {
        impl.logUnityException(null, message, stacktrace, LogExceptionType.UNHANDLED);
    }

    /**
     * Logs a Unity unhandled exception.
     *
     * @param name       exception name.
     * @param message    exception message.
     * @param stacktrace exception stacktrace.
     */
    @InternalApi
    public void logUnhandledUnityException(@NonNull String name, @NonNull String message, @Nullable String stacktrace) {
        impl.logUnityException(name, message, stacktrace, LogExceptionType.UNHANDLED);
    }

    /**
     * Logs a Unity handled exception.
     *
     * @param name       exception name.
     * @param message    exception message.
     * @param stacktrace exception stacktrace.
     */
    @InternalApi
    public void logHandledUnityException(@NonNull String name, @NonNull String message, @Nullable String stacktrace) {
        impl.logUnityException(name, message, stacktrace, LogExceptionType.HANDLED);
    }

    /**
     * Sets the react native version number.
     *
     * @param version react native version number.
     */
    @Deprecated
    public void setReactNativeVersionNumber(@NonNull String version) {
        impl.setReactNativeVersionNumber(version);
    }

    /**
     * Sets javascript patch number.
     *
     * @param number javascript patch number.
     */
    @Deprecated
    public void setJavaScriptPatchNumber(@NonNull String number) {
        impl.setJavaScriptPatchNumber(number);
    }

    /**
     * Sets the Embrace RN SDK version.
     *
     * @param version embrace sdk version.
     */
    public void setReactNativeSdkVersion(@NonNull String version) {
        impl.setReactNativeSdkVersion(version);
    }

    /**
     * Sets the path of the javascript bundle.
     *
     * @param url path of the javascript bundle.
     */
    @SuppressWarnings("AbbreviationAsWordInNameCheck")
    @Deprecated
    public void setJavaScriptBundleURL(@NonNull String url) {
        impl.setJavaScriptBundleURL(url);
    }

    /**
     * Sets the Unity version and Unity build id.
     * (Keep this overload to maintain backward compatibility with older versions of the Unity SDK)
     *
     * @param unityVersion of the Unity Editor
     * @param buildGuid    if the Unity build
     */
    @InternalApi
    @Deprecated
    public void setUnityMetaData(@NonNull String unityVersion, @NonNull String buildGuid) {
        setUnityMetaData(unityVersion, buildGuid, null);
    }

    /**
     * Sets the Unity version, Unity build id and Unity SDK version.
     *
     * @param unityVersion    of the Unity Editor
     * @param buildGuid       if the Unity build
     * @param unitySdkVersion of the Unity SDK
     */
    @InternalApi
    @Deprecated
    public void setUnityMetaData(@NonNull String unityVersion, @NonNull String buildGuid, @Nullable String unitySdkVersion) {
        impl.setUnityMetaData(unityVersion, buildGuid, unitySdkVersion);
    }

    /**
     * Logs an internal error to the Embrace SDK - this is not intended for public use.
     */
    @InternalApi
    public void logInternalError(@Nullable String message, @Nullable String details) {
        impl.logInternalError(message, details);
    }

    /**
     * Logs a Dart error to the Embrace SDK - this is not intended for public use.
     */
    @InternalApi
    @Deprecated
    public void logDartError(
        @Nullable String stack,
        @Nullable String message,
        @Nullable String context,
        @Nullable String library
    ) {
        impl.logDartError(stack, null, message, context, library, LogExceptionType.UNHANDLED);
    }

    /**
     * Logs a Dart error to the Embrace SDK - this is not intended for public use.
     */
    @InternalApi
    @Deprecated
    public void logDartErrorWithType(
        @Nullable String stack,
        @Nullable String message,
        @Nullable String context,
        @Nullable String library,
        @Nullable String errorType
    ) {
        impl.logDartError(stack, errorType, message, context, library, LogExceptionType.UNHANDLED);
    }

    /**
     * Logs a handled Dart error to the Embrace SDK - this is not intended for public use.
     */
    @InternalApi
    public void logHandledDartError(
        @Nullable String stack,
        @Nullable String name,
        @Nullable String message,
        @Nullable String context,
        @Nullable String library
    ) {
        impl.logDartError(stack, name, message, context, library, LogExceptionType.HANDLED);
    }

    /**
     * Logs an unhandled Dart error to the Embrace SDK - this is not intended for public use.
     */
    @InternalApi
    public void logUnhandledDartError(
        @Nullable String stack,
        @Nullable String name,
        @Nullable String message,
        @Nullable String context,
        @Nullable String library
    ) {
        impl.logDartError(stack, name, message, context, library, LogExceptionType.UNHANDLED);
    }

    /**
     * Sets the Embrace Flutter SDK version - this is not intended for public use.
     */
    @InternalApi
    @Deprecated
    public void setEmbraceFlutterSdkVersion(@Nullable String version) {
        impl.setEmbraceFlutterSdkVersion(version);
    }

    /**
     * Sets the Dart version - this is not intended for public use.
     */
    @InternalApi
    @Deprecated
    public void setDartVersion(@Nullable String version) {
        impl.setDartVersion(version);
    }

    /**
     * Registers a {@link ConnectionQualityListener}, notifying the listener each time that there is
     * a change in the connection quality.
     *
     * @param listener the listener to register
     */
    @Deprecated
    public void addConnectionQualityListener(@NonNull ConnectionQualityListener listener) {
        impl.addConnectionQualityListener(listener);
    }

    /**
     * Removes a registered {@link ConnectionQualityListener}, suspending connection quality
     * notifications.
     *
     * @param listener the listener to remove
     */
    @Deprecated
    public void removeConnectionQualityListener(@NonNull ConnectionQualityListener listener) {
        impl.removeConnectionQualityListener(listener);
    }

    @Override
    public synchronized void endSession() {
        endSession(false);
    }

    @Override
    public synchronized void endSession(boolean clearUserInfo) {
        impl.endSession(clearUserInfo);
    }

    @Override
    @NonNull
    public String getDeviceId() {
        return impl.getDeviceId();
    }

    /**
     * Causes a crash with an exception. Use this for test purposes only
     */
    public void throwException() {
        impl.throwException();
    }

    @Override
    public boolean startFragment(@NonNull String name) {
        return impl.startFragment(name);
    }

    @Override
    public boolean endFragment(@NonNull String name) {
        return impl.endFragment(name);
    }

    /**
     * Logs the fact that a particular view was entered.
     * <p>
     * If the previously logged view has the same name, a duplicate view breadcrumb will not be
     * logged.
     *
     * @param screen the name of the view to log
     */
    @InternalApi
    public void logRnView(@NonNull String screen) {
        impl.logRnView(screen);
    }

    @Nullable
    @InternalApi
    public ConfigService getConfigService() {
        return impl.getConfigService();
    }

    @InternalApi
    void installUnityThreadSampler() {
        getImpl().installUnityThreadSampler();
    }

    /**
     * The AppFramework that is in use.
     */
    public enum AppFramework {
        NATIVE(1),
        REACT_NATIVE(2),
        UNITY(3),
        FLUTTER(4);

        private final int value;

        AppFramework(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }

    /**
     * Gets the {@link ReactNativeInternalInterface} that should be used as the sole source of
     * communication with the Android SDK for React Native.
     */
    @Nullable
    @InternalApi
    public ReactNativeInternalInterface getReactNativeInternalInterface() {
        return impl.getReactNativeInternalInterface();
    }

    /**
     * Gets the {@link UnityInternalInterface} that should be used as the sole source of
     * communication with the Android SDK for Unity.
     */
    @Nullable
    @InternalApi
    public UnityInternalInterface getUnityInternalInterface() {
        return impl.getUnityInternalInterface();
    }

    /**
     * Gets the {@link FlutterInternalInterface} that should be used as the sole source of
     * communication with the Android SDK for Flutter.
     */
    @Nullable
    @InternalApi
    public FlutterInternalInterface getFlutterInternalInterface() {
        return impl.getFlutterInternalInterface();
    }

    @InternalApi
    public void sampleCurrentThreadDuringAnrs() {
        impl.sampleCurrentThreadDuringAnrs();
    }

    /**
     * Allows Unity customers to verify their integration.
     */
    void verifyUnityIntegration() {
        EmbraceSamples.verifyIntegration();
    }

    /**
     * Saves captured push notification information into session payload
     *
     * @param title                    the title of the notification as a string (or null)
     * @param body                     the body of the notification as a string (or null)
     * @param topic                    the notification topic (if a user subscribed to one), or null
     * @param id                       A unique ID identifying the message
     * @param notificationPriority     the notificationPriority of the message (as resolved on the device)
     * @param messageDeliveredPriority the priority of the message (as resolved on the server)
     * @param isNotification           if it is a notification message.
     * @param hasData                  if the message contains payload data.
     */
    public void logPushNotification(
        @Nullable String title,
        @Nullable String body,
        @Nullable String topic,
        @Nullable String id,
        @Nullable Integer notificationPriority,
        @NotNull Integer messageDeliveredPriority,
        @NotNull Boolean isNotification,
        @NotNull Boolean hasData) {

        impl.logPushNotification(
            title,
            body,
            topic,
            id,
            notificationPriority,
            messageDeliveredPriority,
            PushNotificationBreadcrumb.NotificationType.Builder.notificationTypeFor(hasData, isNotification)
        );
    }

    /**
     * Saves captured push notification information into session payload
     *
     * @param title                    the title of the notification as a string (or null)
     * @param body                     the body of the notification as a string (or null)
     * @param topic                    the notification topic (if a user subscribed to one), or null
     * @param id                       A unique ID identifying the message
     * @param notificationPriority     the notificationPriority (as resolved on the device)
     * @param messageDeliveredPriority the priority of the message (as resolved on the server)
     * @param notificationType         the type of the push notification
     */
    @InternalApi
    @Deprecated
    public void logPushNotification(
        @Nullable String title,
        @Nullable String body,
        @Nullable String topic,
        @Nullable String id,
        @Nullable Integer notificationPriority,
        @NotNull Integer messageDeliveredPriority,
        @NotNull PushNotificationBreadcrumb.NotificationType notificationType) {

        impl.logPushNotification(
            title,
            body,
            topic,
            id,
            notificationPriority,
            messageDeliveredPriority,
            notificationType
        );
    }

    /**
     * Determine if a network call should be captured based on the network capture rules
     *
     * @param url    the url of the network call
     * @param method the method of the network call
     * @return the network capture rule to apply or null
     */
    @InternalApi
    public boolean shouldCaptureNetworkBody(@NonNull String url, @NonNull String method) {
        if (isStarted()) {
            return impl.shouldCaptureNetworkCall(url, method);
        } else {
            EmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot check for capture rules.");
            return false;
        }
    }

    @InternalApi
    public void setProcessStartedByNotification() {
        impl.setProcessStartedByNotification();
    }

    /**
     * Listen to performance-tracking JavaScript previously embedded in the website's code.
     * The WebView being tracked must have JavaScript enabled.
     *
     * @param tag            a name used to identify the WebView being tracked
     * @param consoleMessage the console message collected from the WebView
     */
    public void trackWebViewPerformance(@NonNull String tag, @NonNull ConsoleMessage consoleMessage) {
        if (consoleMessage.message() != null) {
            trackWebViewPerformance(tag, consoleMessage.message());
        } else {
            logger.logDebug("Empty WebView console message.");
        }
    }

    /**
     * Listen to performance-tracking JavaScript previously embedded in the website's code.
     * The WebView being tracked must have JavaScript enabled.
     *
     * @param tag     a name used to identify the WebView being tracked
     * @param message the console message collected from the WebView
     */
    public void trackWebViewPerformance(@NonNull String tag, @NonNull String message) {
        impl.trackWebViewPerformance(tag, message);
    }

    /**
     * Get the end state of the last run of the application.
     *
     * @return LastRunEndState enum value representing the end state of the last run.
     */
    @NonNull
    @Override
    public LastRunEndState getLastRunEndState() {
        return impl.getLastRunEndState();
    }

    /**
     * Enum representing the end state of the last run of the application.
     */
    public enum LastRunEndState {
        /**
         * The SDK has not been started yet.
         */
        INVALID(0),

        /**
         * The last run resulted in a crash.
         */
        CRASH(1),

        /**
         * The last run did not result in a crash.
         */
        CLEAN_EXIT(2);

        private final int value;

        LastRunEndState(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }
}
