package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.Map;

import io.embrace.android.embracesdk.network.http.HttpMethod;
import io.embrace.android.embracesdk.network.http.NetworkCaptureData;

/**
 * Declares the functions that consist of Embrace's public API. You should not use
 * {@link EmbraceApi} directly or implement it in your own custom classes,
 * as new functions may be added in future. Use the {@link Embrace} class instead.
 */
interface EmbraceApi {

    /**
     * Sets the user ID. This would typically be some form of unique identifier such as a UUID or database key for the user.
     * This ID will persist across app launches until it is explicitly removed using {@link #clearUserIdentifier()}
     * or otherwise cleared.
     *
     * @param userId the unique identifier for the user
     */
    void setUserIdentifier(@Nullable String userId);

    /**
     * Clears the currently set user ID. For example, if the user logs out.
     */
    void clearUserIdentifier();

    /**
     * Sets the current user's email address.
     *
     * @param email the email address of the current user
     */
    void setUserEmail(@Nullable String email);

    /**
     * Clears the currently set user's email address.
     */
    void clearUserEmail();

    /**
     * Sets this user as a paying user. This adds a persona to the user's identity.
     */
    void setUserAsPayer();

    /**
     * Clears this user as a paying user. This would typically be called if a user is no longer
     * paying for the service and has reverted back to a basic user.
     */
    void clearUserAsPayer();

    /**
     * Sets a custom user persona. A persona is a trait associated with a given user.
     *
     * @param persona the persona to set
     */
    void setUserPersona(@NonNull String persona);

    /**
     * Clears the custom user persona, if it is set.
     *
     * @param persona the persona to clear
     */
    void clearUserPersona(@NonNull String persona);

    /**
     * Clears all custom user personas from the user.
     */
    void clearAllUserPersonas();

    /**
     * Adds a property to the current session, overwriting any previous property set with the given key. If a permanent property
     * already exists with the given name and a non-permanent one is to be added, the permanent one will be removed (and vice versa).
     *
     * @param key       The case-sensitive key to be used for this property. The maximum length for this is 128 characters. A key passed in
     *                  that exceeds the maximum length will be truncated.
     * @param value     The value associated with the given key. The maximum length for this is 1024 characters. A value passed in that
     *                  exceeds the maximum length will be truncated.
     * @param permanent True if this property should be added to subsequent sessions going forward, persisting through app launches.
     *
     * @return True if the property was successfully added. Reasons this may fail include an invalid key or value, or if the
     *         session has exceeded its total properties limit.
     */
    boolean addSessionProperty(@NonNull String key,
                               @NonNull String value,
                               boolean permanent);

    /**
     * Removes a property from the current session.
     *
     * @return true if a property with that name had previously existed.
     */
    boolean removeSessionProperty(@NonNull String key);

    /**
     * Retrieves a map of the current session properties.
     *
     * @return a new immutable map containing the current session properties, or null if the SDK has not been started or has been stopped.
     */
    @Nullable
    Map<String, String> getSessionProperties();

    /**
     * Sets the username of the currently logged in user.
     *
     * @param username the username to set
     */
    void setUsername(@Nullable String username);

    /**
     * Clears the username of the currently logged in user, for example if the user has logged out.
     */
    void clearUsername();

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name a name identifying the event
     */
    void startEvent(@NonNull String name);

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name       a name identifying the event
     * @param identifier an identifier distinguishing between multiple events with the same name
     */
    void startEvent(@NonNull String name, @Nullable String identifier);

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name            a name identifying the event
     * @param identifier      an identifier distinguishing between multiple events with the same name
     * @param allowScreenshot true if a screenshot should be taken for a late event, false otherwise
     */
    void startEvent(@NonNull String name,
                    @Nullable String identifier,
                    boolean allowScreenshot);

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name       a name identifying the event
     * @param identifier an identifier distinguishing between multiple events with the same name
     * @param properties custom key-value pairs to provide with the event
     */
    void startEvent(@NonNull String name,
                    @Nullable String identifier,
                    @Nullable Map<String, Object> properties);

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name            a name identifying the event
     * @param identifier      an identifier distinguishing between multiple events with the same name
     * @param allowScreenshot true if a screenshot should be taken for a late event, false otherwise
     * @param properties      custom key-value pairs to provide with the event
     */
    void startEvent(@NonNull String name,
                    @Nullable String identifier,
                    boolean allowScreenshot,
                    @Nullable Map<String, Object> properties);

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name the name of the event to end
     */
    void endEvent(@NonNull String name);

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name       the name of the event to end
     * @param identifier the identifier of the event to end, distinguishing between events with the same name
     */
    void endEvent(@NonNull String name, @Nullable String identifier);

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name       the name of the event to end
     * @param properties custom key-value pairs to provide with the event
     */
    void endEvent(@NonNull String name,
                  @Nullable Map<String, Object> properties);

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name       the name of the event to end
     * @param identifier the identifier of the event to end, distinguishing between events with the same name
     * @param properties custom key-value pairs to provide with the event
     */
    void endEvent(@NonNull String name,
                  @Nullable String identifier,
                  @Nullable Map<String, Object> properties);

    /**
     * Signals that the app has completed startup.
     */
    void endAppStartup();

    /**
     * Signals that the app has completed startup.
     *
     * @param properties properties to include as part of the startup moment
     */
    void endAppStartup(@NonNull Map<String, Object> properties);


    /**
     * Remotely logs a message at INFO level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message the message to remotely log
     */
    void logInfo(@NonNull String message);

    /**
     * Remotely logs a message at INFO level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    void logInfo(@NonNull String message,
                 @Nullable Map<String, Object> properties);


    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message the message to remotely log
     */
    void logWarning(@NonNull String message);

    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    void logWarning(@NonNull String message,
                    @Nullable Map<String, Object> properties);

    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message         the message to remotely log
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    void logWarning(@NonNull String message,
                    @Nullable Map<String, Object> properties,
                    boolean allowScreenshot);

    /**
     * Remotely logs a message at WARN level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     *
     * @param message              the message to remotely log
     * @param properties           custom key-value pairs to include with the log message
     * @param allowScreenshot      true if a screenshot should be taken for this message, false otherwise
     * @param javascriptStackTrace javascript stack trace coming from the the RN side
     */
    void logWarning(@NonNull String message,
                    @Nullable Map<String, Object> properties,
                    boolean allowScreenshot,
                    @Nullable String javascriptStackTrace);

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message the message to remotely log
     */
    void logError(@NonNull String message);

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message    the message to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    void logError(@NonNull String message,
                  @Nullable Map<String, Object> properties);

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message         the message to remotely log
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    void logError(@NonNull String message,
                  @Nullable Map<String, Object> properties,
                  boolean allowScreenshot);

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message              the message to remotely log
     * @param properties           custom key-value pairs to include with the log message
     * @param allowScreenshot      true if a screenshot should be taken for this message, false otherwise
     * @param javascriptStackTrace javascript stack trace coming from the the RN side
     */
    void logError(@NonNull String message,
                  @Nullable Map<String, Object> properties,
                  boolean allowScreenshot,
                  @Nullable String javascriptStackTrace);

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     *
     * @param message              the message to remotely log
     * @param properties           custom key-value pairs to include with the log message
     * @param allowScreenshot      true if a screenshot should be taken for this message, false otherwise
     * @param javascriptStackTrace javascript stack trace coming from the the RN side
     * @param isException          mark the log as an exception
     */

    void logError(@NonNull String message,
                  @Nullable Map<String, Object> properties,
                  boolean allowScreenshot,
                  @Nullable String javascriptStackTrace,
                  boolean isException);

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param e the throwable to remotely log
     */
    void logError(@NonNull Throwable e);

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param e          the throwable to remotely log
     * @param properties custom key-value pairs to include with the log message
     */
    void logError(@NonNull Throwable e,
                  @Nullable Map<String, Object> properties);

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param e               the throwable to remotely log
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    void logError(@NonNull Throwable e,
                  @Nullable Map<String, Object> properties,
                  boolean allowScreenshot);

    /**
     * Remotely logs a message at ERROR level. These log messages will appear as part of the session
     * timeline, and can be used to describe what was happening at a particular time within the app.
     * <p>
     * If enabled for the current app, a screenshot will automatically be taken when the error is
     * logged, and displayed on the dashboard with the error message.
     * <p>
     * The stacktrace from the throwable will be used instead of the stack trace from where this method is called.
     *
     * @param exc             the throwable to remotely log
     * @param message         message to log (overrides getLocalizedMessage() from e)
     * @param properties      custom key-value pairs to include with the log message
     * @param allowScreenshot true if a screenshot should be taken for this message, false otherwise
     */
    void logError(@NonNull Throwable exc,
                  @NonNull String message,
                  @Nullable Map<String, Object> properties,
                  boolean allowScreenshot);

    /**
     * Logs a breadcrumb.
     * <p>
     * Breadcrumbs track a user's journey through the application and will be shown on the timeline.
     *
     * @param message the name of the breadcrumb to log
     */
    void logBreadcrumb(@NonNull String message);

    /**
     * Logs a handled exception to the Embrace.io API for aggregation and viewing on the dashboard.
     * The log will include the stacktrace found on the exception.
     *
     * @param throwable The handled exception object, which will be serialized and combined
     *                  with the stack trace to aggregate the errors on the dashboard view.
     * @param type      Will flag the message as one of info, warning, or error for filtering on the dashboard
     */
    void logHandledException(@NonNull Throwable throwable, @NonNull LogType type);

    /**
     * Logs a handled exception to the Embrace.io API for aggregation and viewing on the dashboard.
     * The log will include the stacktrace found on the exception.
     *
     * @param throwable        The handled exception object.
     * @param type             Will flag the message as one of info, warning, or error for filtering on the dashboard
     * @param customStackTrace A custom Stack Trace to be logged and displayed on the dashboard view.
     */
    void logHandledException(@NonNull Throwable throwable, @NonNull LogType type, StackTraceElement[] customStackTrace);


    /**
     * Logs a handled exception to the Embrace.io API for aggregation and viewing on the dashboard.
     * The log will include the stacktrace found on the exception.
     *
     * @param throwable  The handled exception object, which will be serialized and combined with
     *                   the stack trace to aggregate the errors on the dashboard view.
     * @param type       Will flag the message as one of info, warning, or error for filtering on the dashboard
     * @param properties An optional dictionary of custom key/value properties to be sent with the error log.
     */
    void logHandledException(@NonNull Throwable throwable, @NonNull LogType type, Map<String, Object> properties);

    /**
     * Logs a handled exception to the Embrace.io API for aggregation and viewing on the dashboard.
     * The log will include the stacktrace found on the exception.
     *
     * @param throwable        The handled exception object, which will be serialized and combined with the stack trace
     *                         to aggregate the errors on the dashboard view.
     * @param type             Will flag the message as one of info, warning, or error for filtering on the dashboard
     * @param properties       An optional dictionary of custom key/value properties to be sent with the error log.
     * @param customStackTrace A custom Stack Trace to be logged and displayed on the dashboard view. If this object is not null,
     *                         the Exception Stack Trace will be ignored.
     * @param takeScreenshot   A flag for whether the SDK should take a screenshot of the application window to display on the dashboard
     */
    void logHandledException(@NonNull Throwable throwable,
                             @NonNull LogType type,
                             Map<String, Object> properties,
                             StackTraceElement[] customStackTrace,
                             boolean takeScreenshot);


    /**
     * Ends the current session and starts a new one.
     */
    void endSession();

    /**
     * Ends the current session and starts a new one.
     *
     * @param clearUserInfo Pass in true to clear all user info set on this device.
     */
    void endSession(boolean clearUserInfo);

    /**
     * Get the user identifier assigned to the device by Embrace
     *
     * @return the device identifier created by Embrace
     */
    @NonNull
    String getDeviceId();

    /**
     * Logs the fact that a network call occurred. These are recorded and sent to Embrace as part
     * of a particular session.
     *
     * @param url           the URL of the network call
     * @param httpMethod    the HTTP method of the network call
     * @param statusCode    the status code returned by the server
     * @param startTime     the time that the network call started
     * @param endTime       the time that the network call was completed
     * @param bytesSent     the number of bytes sent as part of the network call
     * @param bytesReceived the number of bytes returned by the server in response to the network call
     * @param traceId       the optional trace id that can be used to trace a particular request
     */
    void logNetworkCall(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        int statusCode,
        long startTime,
        long endTime,
        long bytesSent,
        long bytesReceived,
        @Nullable String traceId);

    /**
     * Logs the fact that a network call occurred. These are recorded and sent to Embrace as part
     * of a particular session.
     *
     * @param url                the URL of the network call
     * @param httpMethod         the HTTP method of the network call
     * @param statusCode         the status code returned by the server
     * @param startTime          the time that the network call started
     * @param endTime            the time that the network call was completed
     * @param bytesSent          the number of bytes sent as part of the network call
     * @param bytesReceived      the number of bytes returned by the server in response to the network call
     * @param traceId            the optional trace id that can be used to trace a particular request
     * @param networkCaptureData the additional data captured if network body capture is enabled for the URL
     */
    void logNetworkCall(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        int statusCode,
        long startTime,
        long endTime,
        long bytesSent,
        long bytesReceived,
        @Nullable String traceId,
        @Nullable NetworkCaptureData networkCaptureData);

    /**
     * Logs the fact that an exception was thrown when attempting to make a network call.
     * <p>
     * These are client-side exceptions and not server-side exceptions, such as a DNS error or
     * failure to connect to the remote server.
     *
     * @param url          the URL of the network call
     * @param httpMethod   the HTTP method of the network call
     * @param startTime    the time that the network call started
     * @param endTime      the time that the network call was completed
     * @param errorType    the type of the exception
     * @param errorMessage the message returned by the exception
     * @param traceId      the optional trace id that can be used to trace a particular request
     */
    void logNetworkClientError(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        long startTime,
        long endTime,
        @NonNull String errorType,
        @NonNull String errorMessage,
        @Nullable String traceId);

    /**
     * Logs the fact that an exception was thrown when attempting to make a network call.
     * <p>
     * These are client-side exceptions and not server-side exceptions, such as a DNS error or
     * failure to connect to the remote server.
     *
     * @param url                the URL of the network call
     * @param httpMethod         the HTTP method of the network call
     * @param startTime          the time that the network call started
     * @param endTime            the time that the network call was completed
     * @param errorType          the type of the exception
     * @param errorMessage       the message returned by the exception
     * @param traceId            the optional trace id that can be used to trace a particular request
     * @param networkCaptureData the additional data captured if network body capture is enabled for the URL
     */
    void logNetworkClientError(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        long startTime,
        long endTime,
        @NonNull String errorType,
        @NonNull String errorMessage,
        @Nullable String traceId,
        @Nullable NetworkCaptureData networkCaptureData);

    /**
     * Get the end state of the last run of the application.
     *
     * @return LastRunEndState enum value representing the end state of the last run.
     */
    @NonNull
    Embrace.LastRunEndState getLastRunEndState();


}


