package io.embrace.android.embracesdk;

import android.app.Activity;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import java.util.concurrent.atomic.AtomicInteger;

import io.embrace.android.embracesdk.capture.PerformanceInfoService;
import io.embrace.android.embracesdk.capture.user.UserService;
import io.embrace.android.embracesdk.clock.Clock;
import io.embrace.android.embracesdk.comms.delivery.DeliveryService;
import io.embrace.android.embracesdk.config.ConfigListener;
import io.embrace.android.embracesdk.config.ConfigService;
import io.embrace.android.embracesdk.config.behavior.BackgroundActivityBehavior;
import io.embrace.android.embracesdk.internal.spans.EmbraceAttributes;
import io.embrace.android.embracesdk.internal.spans.SpansService;
import io.embrace.android.embracesdk.worker.BackgroundWorker;
import kotlin.Lazy;

final class EmbraceBackgroundActivityService implements BackgroundActivityService, ActivityListener, ConfigListener {

    /**
     * Signals to the API that this is a background session.
     */
    private static final String APPLICATION_STATE_BACKGROUND = "background";

    /**
     * Signals to the API the end of a session.
     */
    private static final String MESSAGE_TYPE_END = "en";

    /**
     * Minimum time between writes of the background activity to disk
     */
    private static final long MIN_INTERVAL_BETWEEN_SAVES = 5000;

    /**
     * Embrace service dependencies of the background activity session service.
     */
    private final Clock clock;
    private final PerformanceInfoService performanceInfoService;
    private final MetadataService metadataService;
    private final BreadcrumbService breadcrumbService;
    private final EventService eventService;
    private final EmbraceRemoteLogger remoteLogger;
    private final UserService userService;
    private final EmbraceInternalErrorService exceptionService;
    private final DeliveryService deliveryService;
    private final ConfigService configService;
    private final NdkService ndkService;
    private final SpansService spansService;
    private final Lazy<BackgroundWorker> workerSupplier;
    private BackgroundWorker backgroundActivityCacheWorker;
    private long lastSaved;
    private boolean willBeSaved = false;

    /**
     * The active background activity session.
     */
    @VisibleForTesting
    volatile BackgroundActivity backgroundActivity;
    private final AtomicInteger manualBkgSessionsSent = new AtomicInteger(0);
    @VisibleForTesting
    long lastSendAttempt;

    private boolean isEnabled = true;

    public EmbraceBackgroundActivityService(
        @NonNull PerformanceInfoService performanceInfoService,
        @NonNull MetadataService metadataService,
        @NonNull BreadcrumbService breadcrumbService,
        @NonNull ActivityService activityService,
        @NonNull EventService eventService,
        @NonNull EmbraceRemoteLogger remoteLogger,
        @NonNull UserService userService,
        @NonNull EmbraceInternalErrorService exceptionService,
        @NonNull DeliveryService deliveryService,
        @NonNull ConfigService configService,
        @NonNull NdkService ndkService,
        @NonNull Clock clock,
        @NonNull SpansService spansService,
        @NonNull Lazy<BackgroundWorker> workerSupplier) {
        this.clock = clock;
        this.performanceInfoService = performanceInfoService;
        this.metadataService = metadataService;
        this.breadcrumbService = breadcrumbService;
        this.eventService = eventService;
        this.remoteLogger = remoteLogger;
        this.userService = userService;
        this.exceptionService = exceptionService;
        this.deliveryService = deliveryService;
        this.configService = configService;
        this.ndkService = ndkService;
        this.spansService = spansService;
        this.workerSupplier = workerSupplier;

        activityService.addListener(this);

        this.lastSendAttempt = clock.now();

        configService.addListener(this);

        if (activityService.isInBackground()) {
            // start background activity capture from a cold start
            startBackgroundActivityCapture(clock.now(), true, BackgroundActivity.LifeEventType.BKGND_STATE);
        }
    }

    @Override
    public void sendBackgroundActivity() {
        if (!isEnabled || !verifyManualSendThresholds()) {
            return;
        }

        long now = clock.now();
        BackgroundActivityMessage backgroundActivityMessage =
            stopBackgroundActivityCapture(now, BackgroundActivity.LifeEventType.BKGND_MANUAL, null);
        // start a new background activity session
        startBackgroundActivityCapture(clock.now(), false, BackgroundActivity.LifeEventType.BKGND_MANUAL);
        if (backgroundActivityMessage != null) {
            deliveryService.sendBackgroundActivity(backgroundActivityMessage);
        }
    }

    @Override
    public void handleCrash(@NonNull String crashId) {
        if (isEnabled && backgroundActivity != null) {
            long now = clock.now();
            BackgroundActivityMessage backgroundActivityMessage =
                stopBackgroundActivityCapture(now, BackgroundActivity.LifeEventType.BKGND_STATE, crashId);
            if (backgroundActivityMessage != null) {
                deliveryService.saveBackgroundActivity(backgroundActivityMessage);
            }
            startBackgroundActivityCapture(clock.now(), false, BackgroundActivity.LifeEventType.BKGND_STATE);
        }
    }

    @Override
    public void onForeground(boolean coldStart, long startupTime, long timestamp) {
        if (isEnabled) {
            BackgroundActivityMessage backgroundActivityMessage =
                stopBackgroundActivityCapture(timestamp - 1, BackgroundActivity.LifeEventType.BKGND_STATE, null);
            if (backgroundActivityMessage != null) {
                deliveryService.saveBackgroundActivity(backgroundActivityMessage);
            }
            deliveryService.sendBackgroundActivities();
        }
    }

    @Override
    public void onBackground(long timestamp) {
        if (isEnabled) {
            startBackgroundActivityCapture(timestamp + 1, false, BackgroundActivity.LifeEventType.BKGND_STATE);
        }
    }

    @Override
    public void onConfigChange(@NonNull ConfigService configService) {
        if (isEnabled && !configService.isBackgroundActivityCaptureEnabled()) {
            disableService();
        } else if (!isEnabled && configService.isBackgroundActivityCaptureEnabled()) {
            enableService();
        }
    }

    /**
     * Save the background activity to disk
     */
    public void save() {
        if (isEnabled && backgroundActivity != null) {
            if (clock.now() - lastSaved > MIN_INTERVAL_BETWEEN_SAVES) {
                saveNow();
            } else if (!willBeSaved) {
                willBeSaved = true;
                saveLater();
            }
        }
    }

    private void saveNow() {
        getCacheWorker().submitSafe(() -> {
            cacheBackgroundActivity();
            return null;
        });
        willBeSaved = false;
    }

    private void saveLater() {
        Handler handler = new Handler(Looper.getMainLooper());
        handler.postDelayed(this::saveNow, MIN_INTERVAL_BETWEEN_SAVES);
    }

    private void disableService() {
        isEnabled = false;
    }

    private void enableService() {
        isEnabled = true;
    }

    /**
     * Start the background activity capture by starting the cache service and creating the background
     * session.
     *
     * @param coldStart defines if the action comes from an application cold start or not
     * @param startType defines which is the lifecycle of the session
     */
    private void startBackgroundActivityCapture(long startTime, boolean coldStart, BackgroundActivity.LifeEventType startType) {
        backgroundActivity = BackgroundActivity.createStartMessage(
            Uuid.getEmbUuid(),
            startTime,
            coldStart,
            startType,
            APPLICATION_STATE_BACKGROUND,
            userService.loadUserInfoFromDisk()
        );

        metadataService.setActiveSessionId(backgroundActivity.getSessionId());

        if (configService.getAutoDataCaptureBehavior().isNdkEnabled()) {
            ndkService.updateSessionId(backgroundActivity.getSessionId());
        }

        saveNow();
    }

    /**
     * Stop the background activity capture by stopping the cache service and putting the background
     * session to its final state with all the data collected up to the current point.
     * Build the next background message and attempt to send it.
     *
     * @param endType defines what kind of event ended the background activity capture
     */
    private synchronized BackgroundActivityMessage stopBackgroundActivityCapture(
        long endTime,
        BackgroundActivity.LifeEventType endType,
        String crashId) {
        if (backgroundActivity == null) {
            EmbraceLogger.logError("No background activity to report");
            return null;
        }

        long startTime = backgroundActivity.getStartTime() != null ? backgroundActivity.getStartTime() : 0;
        BackgroundActivity sendBackgroundActivity = BackgroundActivity.createStopMessage(
            backgroundActivity,
                APPLICATION_STATE_BACKGROUND,
                MESSAGE_TYPE_END,
                endTime,
                eventService.findEventIdsForSession(startTime, endTime),
                remoteLogger.findInfoLogIds(startTime, endTime),
                remoteLogger.findWarningLogIds(startTime, endTime),
                remoteLogger.findErrorLogIds(startTime, endTime),
                remoteLogger.getInfoLogsAttemptedToSend(),
                remoteLogger.getWarnLogsAttemptedToSend(),
                remoteLogger.getErrorLogsAttemptedToSend(),
                exceptionService.getCurrentExceptionError(),
                endTime,
                endType,
                remoteLogger.getUnhandledExceptionsSent(),
                crashId
            );
        backgroundActivity = null;

        return buildBackgroundActivityMessage(sendBackgroundActivity, true);
    }

    /**
     * Verify if the amount of background activities captured reach the limit or if the last send
     * attempt was less than 5 sec ago.
     *
     * @return false if the verify failed, true otherwise
     */
    private boolean verifyManualSendThresholds() {
        BackgroundActivityBehavior behavior = configService.getBackgroundActivityBehavior();
        int manualBackgroundActivityLimit = behavior.getManualBackgroundActivityLimit();
        long minBackgroundActivityDuration = behavior.getMinBackgroundActivityDuration();

        if (manualBkgSessionsSent.getAndIncrement() >= manualBackgroundActivityLimit) {
            EmbraceLogger.logWarning("Warning, failed to send background activity. " +
                "The amount of background activity that can be sent reached the limit..");
            return false;
        }

        if (lastSendAttempt < minBackgroundActivityDuration) {
            EmbraceLogger.logWarning("Warning, failed to send background activity. The last attempt " +
                "to send background activity was less than 5 seconds ago.");
            return false;
        }

        return true;
    }

    /**
     * Create the background session message with the current state of the background activity.
     *
     * @param backgroundActivity      the current state of a background activity
     * @param isBackgroundActivityEnd true if the message is being built for the termination of the background activity
     * @return a background activity message for backend
     */
    private BackgroundActivityMessage buildBackgroundActivityMessage(
        BackgroundActivity backgroundActivity,
        boolean isBackgroundActivityEnd
    ) {
        if (backgroundActivity != null) {
            long startTime = backgroundActivity.getStartTime() != null ? backgroundActivity.getStartTime() : 0L;
            long endTime = backgroundActivity.getEndTime() != null ? backgroundActivity.getEndTime() : clock.now();
            final boolean isCrash = backgroundActivity.getCrashReportId() != null;

            return new BackgroundActivityMessage(
                backgroundActivity,
                backgroundActivity.getUser(),
                metadataService.getAppInfo(),
                metadataService.getDeviceInfo(),
                performanceInfoService.getSessionPerformanceInfo(
                    startTime, endTime, Boolean.TRUE.equals(backgroundActivity.isColdStart()), null, false),
                breadcrumbService.getBreadcrumbs(startTime, endTime),
                isBackgroundActivityEnd ?
                    spansService.flushSpans(isCrash ? EmbraceAttributes.AppTerminationCause.CRASH : null) : spansService.completedSpans()
                );
        }
        return null;
    }

    /**
     * Cache the activity, with performance information generated up to the current point.
     */
    private void cacheBackgroundActivity() {
        try {
            if (backgroundActivity != null) {
                lastSaved = clock.now();

                long startTime = backgroundActivity.getStartTime() != null ? backgroundActivity.getStartTime() : 0L;
                long endTime = backgroundActivity.getEndTime() != null ? backgroundActivity.getEndTime() : clock.now();

                BackgroundActivity cachedActivity = BackgroundActivity.createStopMessage(
                    backgroundActivity,
                    APPLICATION_STATE_BACKGROUND,
                    MESSAGE_TYPE_END,
                    null,
                    eventService.findEventIdsForSession(startTime, endTime),
                    remoteLogger.findInfoLogIds(startTime, endTime),
                    remoteLogger.findWarningLogIds(startTime, endTime),
                    remoteLogger.findErrorLogIds(startTime, endTime),
                    remoteLogger.getInfoLogsAttemptedToSend(),
                    remoteLogger.getWarnLogsAttemptedToSend(),
                    remoteLogger.getErrorLogsAttemptedToSend(),
                    exceptionService.getCurrentExceptionError(),
                    clock.now(),
                    null,
                    remoteLogger.getUnhandledExceptionsSent(),
                    null
                );

                BackgroundActivityMessage message = buildBackgroundActivityMessage(cachedActivity, false);
                if (message == null) {
                    EmbraceLogger.logDebug("Failed to cache background activity message.");
                    return;
                }

                deliveryService.saveBackgroundActivity(message);
            }
        } catch (Exception ex) {
            EmbraceLogger.logDebug("Error while caching active session", ex);
        }
    }

    /**
     * Lazy load the worker for writing to cache
     */
    private synchronized BackgroundWorker getCacheWorker() {
        if (backgroundActivityCacheWorker == null) {
            this.backgroundActivityCacheWorker = workerSupplier.getValue();
        }
        return backgroundActivityCacheWorker;
    }

    @Override
    public void applicationStartupComplete() {
    }

    @Override
    public void onView(@NonNull Activity activity) {
    }

    @Override
    public void onViewClose(@NonNull Activity activity) {
    }

    @Override
    public void onActivityCreated(@NonNull Activity activity, @Nullable Bundle bundle) {
    }
}
