package io.embrace.android.embracesdk;

import static io.embrace.android.embracesdk.EmbraceEventService.STARTUP_EVENT_NAME;

import android.app.Application;
import android.content.Context;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import io.embrace.android.embracesdk.anr.AnrService;
import io.embrace.android.embracesdk.anr.ndk.EmbraceNativeThreadSamplerServiceKt;
import io.embrace.android.embracesdk.anr.ndk.NativeThreadSamplerInstaller;
import io.embrace.android.embracesdk.anr.ndk.NativeThreadSamplerService;
import io.embrace.android.embracesdk.capture.crumbs.activity.ActivityLifecycleBreadcrumbService;
import io.embrace.android.embracesdk.capture.memory.MemoryService;
import io.embrace.android.embracesdk.capture.strictmode.StrictModeService;
import io.embrace.android.embracesdk.capture.user.UserService;
import io.embrace.android.embracesdk.capture.webview.WebViewService;
import io.embrace.android.embracesdk.clock.Clock;
import io.embrace.android.embracesdk.clock.SystemClock;
import io.embrace.android.embracesdk.config.ConfigService;
import io.embrace.android.embracesdk.config.behavior.NetworkBehavior;
import io.embrace.android.embracesdk.config.behavior.SessionBehavior;
import io.embrace.android.embracesdk.injection.AnrModuleImpl;
import io.embrace.android.embracesdk.injection.CoreModule;
import io.embrace.android.embracesdk.injection.CoreModuleImpl;
import io.embrace.android.embracesdk.injection.CrashModule;
import io.embrace.android.embracesdk.injection.CrashModuleImpl;
import io.embrace.android.embracesdk.injection.CustomerLogModuleImpl;
import io.embrace.android.embracesdk.injection.DataCaptureServiceModule;
import io.embrace.android.embracesdk.injection.DataCaptureServiceModuleImpl;
import io.embrace.android.embracesdk.injection.DataContainerModule;
import io.embrace.android.embracesdk.injection.DataContainerModuleImpl;
import io.embrace.android.embracesdk.injection.DeliveryModule;
import io.embrace.android.embracesdk.injection.DeliveryModuleImpl;
import io.embrace.android.embracesdk.injection.EssentialServiceModule;
import io.embrace.android.embracesdk.injection.EssentialServiceModuleImpl;
import io.embrace.android.embracesdk.injection.SdkObservabilityModule;
import io.embrace.android.embracesdk.injection.SdkObservabilityModuleImpl;
import io.embrace.android.embracesdk.injection.SystemServiceModule;
import io.embrace.android.embracesdk.injection.SystemServiceModuleImpl;
import io.embrace.android.embracesdk.internal.ApkToolsConfig;
import io.embrace.android.embracesdk.internal.crash.LastRunCrashVerifier;
import io.embrace.android.embracesdk.internal.spans.EmbraceSpansService;
import io.embrace.android.embracesdk.internal.spans.SpansService;
import io.embrace.android.embracesdk.logging.InternalEmbraceLogger;
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.network.EmbraceNetworkRequest;
import io.embrace.android.embracesdk.network.EmbraceNetworkRequestV2;
import io.embrace.android.embracesdk.network.http.HttpMethod;
import io.embrace.android.embracesdk.network.http.NetworkCaptureData;
import io.embrace.android.embracesdk.registry.ServiceRegistry;
import io.embrace.android.embracesdk.worker.WorkerName;
import io.embrace.android.embracesdk.worker.WorkerThreadModule;
import io.embrace.android.embracesdk.worker.WorkerThreadModuleImpl;
import kotlin.jvm.functions.Function0;
import kotlin.jvm.functions.Function2;
import kotlin.jvm.functions.Function4;

/**
 * Implementation class of the SDK. Embrace.java forms our public API and calls functions in this
 * class.
 * <p>
 * Any non-public APIs or functionality related to the Embrace.java client should ideally be put
 * here instead.
 */
final class EmbraceImpl {

    private static final String ERROR_USER_UPDATES_DISABLED = "User updates are disabled, ignoring user persona update.";

    /**
     * Whether the Embrace SDK has been started yet.
     */
    private final AtomicBoolean started = new AtomicBoolean(false);

    /**
     * Custom app ID that overrides the one specified at build time
     */
    private static String customAppId;

    /**
     * The application being instrumented by the SDK.
     */
    private volatile Application application;

    /**
     * The application framework. It should be Android Native, Unity or React Native
     */
    private volatile Embrace.AppFramework appFramework;

    /**
     * The breadcrumbs service.
     */
    private volatile BreadcrumbService breadcrumbService;

    /**
     * The session handling service.
     */
    private volatile SessionService sessionService;

    /**
     * The background activity service.
     */
    private volatile BackgroundActivityService backgroundActivityService;

    /**
     * The device metadata service.
     */
    private volatile MetadataService metadataService;

    /**
     * The Activity service.
     */
    private volatile ActivityService activityService;

    /**
     * The network call logging service.
     */
    private volatile NetworkLoggingService networkLoggingService;


    /**
     * The ANR Monitoring service. This is static to allow for customers to initialize it
     * before the main SDK has initialized if they wish to capture early ANRs.
     */
    @Nullable
    private volatile AnrService anrService;

    /**
     * The EmbraceRemoteLogger.
     */
    private volatile EmbraceRemoteLogger remoteLogger;

    /**
     * The Configuration service.
     */
    private volatile ConfigService configService;

    /**
     * The Embrace prefences service.
     */
    private volatile PreferencesService preferencesService;

    /**
     * The Embrace event service.
     */
    private volatile EventService eventService;

    /**
     * The User service.
     */
    private volatile UserService userService;

    /**
     * The Embrace exception class service.
     */
    private volatile EmbraceInternalErrorService exceptionsService;

    /**
     * The Embrace NDK class service.
     */
    private volatile NdkService ndkService;

    /**
     * The Embrace network capture service.
     */
    private volatile NetworkCaptureService networkCaptureService;

    /**
     * The Embrace Webview info collector service.
     */
    private volatile WebViewService webViewService;

    /**
     * The Internal Embrace Logger.
     */
    private final InternalEmbraceLogger internalEmbraceLogger = InternalStaticEmbraceLogger.logger;

    private NativeThreadSamplerService nativeThreadSampler;
    private NativeThreadSamplerInstaller nativeThreadSamplerInstaller;

    @Nullable
    private ReactNativeInternalInterface reactNativeInternalInterface;

    @Nullable
    private UnityInternalInterface unityInternalInterface;

    @Nullable
    private FlutterInternalInterface flutterInternalInterface;

    private PushNotificationCaptureService pushNotificationService;
    private WorkerThreadModule workerThreadModule;

    private volatile SpansService spansService;
    private ServiceRegistry serviceRegistry;

    private LastRunCrashVerifier crashVerifier;

    // Set this to some clock initially so we will never be null - change this to clock instance the SDK uses when it's initialized
    @NonNull
    private Clock sdkClock = new SystemClock();

    @NonNull
    private final Function2<Context, Embrace.AppFramework, CoreModule> coreModuleSupplier;

    @NonNull
    private final Function0<WorkerThreadModule> workerThreadModuleSupplier;

    @NonNull
    private final Function4<CoreModule, EssentialServiceModule, DataCaptureServiceModule, WorkerThreadModule, DeliveryModule>
        deliveryModuleSupplier;

    EmbraceImpl(
        @NonNull Function2<Context, Embrace.AppFramework, CoreModule> coreModuleSupplier,
        @NonNull Function0<WorkerThreadModule> workerThreadModuleSupplier,
        @NonNull Function4<CoreModule, EssentialServiceModule, DataCaptureServiceModule, WorkerThreadModule, DeliveryModule>
            deliveryModuleSupplier
    ) {
        this.coreModuleSupplier = coreModuleSupplier;
        this.workerThreadModuleSupplier = workerThreadModuleSupplier;
        this.deliveryModuleSupplier = deliveryModuleSupplier;
    }

    EmbraceImpl() {
        this(CoreModuleImpl::new, WorkerThreadModuleImpl::new, DeliveryModuleImpl::new);
    }

    /**
     * Starts instrumentation of the Android application using the Embrace SDK. This should be
     * called during creation of the application, as early as possible.
     * <p>
     * See <a href="https://docs.embrace.io/docs/android-integration-guide">Embrace Docs</a> for
     * integration instructions. For compatibility with other networking SDKs such as Akamai,
     * the Embrace SDK must be initialized after any other SDK.
     *
     * @param context                  an instance of context
     * @param enableIntegrationTesting if true, debug sessions (those which are not part of a
     *                                 release APK) will go to the live integration testing tab
     *                                 of the dashboard. If false, they will appear in 'recent
     *                                 sessions'.
     */
    public void start(@NonNull Context context,
                      boolean enableIntegrationTesting,
                      @NonNull Embrace.AppFramework appFramework) {
        try {
            startImpl(context, enableIntegrationTesting, appFramework);
        } catch (Exception ex) {
            internalEmbraceLogger.logError(
                "Exception occurred while initializing the Embrace SDK. Instrumentation may be disabled.", ex, true);
        }
    }

    private void startImpl(@NonNull Context context,
                           boolean enableIntegrationTesting,
                           @NonNull Embrace.AppFramework appFramework) {
        if (this.application != null) {
            // We don't hard fail if the SDK has been already initialized.
            InternalStaticEmbraceLogger.logWarning("Embrace SDK has already been initialized");
            return;
        }
        if (ApkToolsConfig.IS_SDK_DISABLED) {
            internalEmbraceLogger.logInfo("SDK disabled through ApkToolsConfig");
            stop();
            return;
        }

        final CoreModule coreModule = coreModuleSupplier.invoke(context, appFramework);
        serviceRegistry = coreModule.getServiceRegistry();
        SystemServiceModule systemServiceModule = new SystemServiceModuleImpl(coreModule);

        this.application = coreModule.getApplication();
        this.appFramework = coreModule.getAppFramework();
        this.sdkClock = coreModule.getClock();

        internalEmbraceLogger.logDeveloper("Embrace", "Starting SDK for framework " + appFramework.name());

        long startTime = sdkClock.now();
        /* ------------------------------------------------------------------------------------
         *  Device instrumentation (power, memory, CPU, network, preferences, CPU)
         *  */
        BuildInfo buildInfo = BuildInfo.fromResources(coreModule.getResources(), coreModule.getContext().getPackageName());
        this.workerThreadModule = workerThreadModuleSupplier.invoke();

        // bootstrap initialization. ConfigService not created yet...
        EssentialServiceModule essentialServiceModule = new EssentialServiceModuleImpl(
            coreModule,
            systemServiceModule,
            workerThreadModule,
            buildInfo,
            customAppId,
            enableIntegrationTesting,
            () -> {
                Embrace.getImpl().stop();
                return null;
            });

        this.activityService = essentialServiceModule.getActivityService();
        this.preferencesService = essentialServiceModule.getPreferencesService();
        this.metadataService = essentialServiceModule.getMetadataService();
        this.configService = essentialServiceModule.getConfigService();

        // example usage.
        serviceRegistry.registerServices(
            activityService,
            preferencesService,
            metadataService,
            configService
        );

        // only call after ConfigService has initialized.
        metadataService.precomputeValues();

        DataCaptureServiceModule dataCaptureServiceModule = new DataCaptureServiceModuleImpl(
            coreModule,
            systemServiceModule,
            essentialServiceModule,
            workerThreadModule
        );
        this.webViewService = dataCaptureServiceModule.getWebviewService();
        MemoryService memoryService = dataCaptureServiceModule.getMemoryService();
        ((EmbraceActivityService) essentialServiceModule.getActivityService())
            .setMemoryService(dataCaptureServiceModule.getMemoryService());
        serviceRegistry.registerServices(
            webViewService,
            memoryService
        );

        /*
         * Since onForeground() is called sequential in the order that services registered for it,
         * it is important to initialize the `EmbraceAnrService`, and thus register the `onForeground()
         * listener for it, before the `EmbraceSessionService`.
         * The onForeground() call inside the EmbraceAnrService should be called before the
         * EmbraceSessionService call. This is necessary since the EmbraceAnrService should be able to
         * force a Main thread health check and close the pending ANR intervals that happened on the
         * background before the next session is created.
         */
        AnrModuleImpl anrModule = new AnrModuleImpl(
            coreModule,
            systemServiceModule,
            essentialServiceModule
        );
        AnrService anrServiceImpl = anrModule.getAnrService();
        this.anrService = anrServiceImpl;
        serviceRegistry.registerService(anrService);

        // set callbacks and pass in non-placeholder config.
        anrServiceImpl.finishInitialization(
            essentialServiceModule.getConfigService()
        );

        serviceRegistry.registerService(dataCaptureServiceModule.getPowerSaveModeService());

        // initialize the logger early so that logged exceptions have a good chance of
        // being appended to the exceptions service rather than logcat
        SdkObservabilityModule sdkObservabilityModule = new SdkObservabilityModuleImpl(
            coreModule,
            essentialServiceModule
        );
        this.exceptionsService = sdkObservabilityModule.getExceptionService();
        serviceRegistry.registerService(exceptionsService);
        internalEmbraceLogger.addLoggerAction(sdkObservabilityModule.getInternalErrorLogger());

        serviceRegistry.registerService(dataCaptureServiceModule.getNetworkConnectivityService());

        /* ------------------------------------------------------------------------------------
         *  API services
         *  */
        final DeliveryModule deliveryModule = deliveryModuleSupplier.invoke(coreModule,
            essentialServiceModule,
            dataCaptureServiceModule,
            workerThreadModule
        );

        serviceRegistry.registerService(deliveryModule.getDeliveryService());

        final EmbraceSessionProperties sessionProperties = new EmbraceSessionProperties(
            essentialServiceModule.getPreferencesService(),
            coreModule.getLogger(),
            essentialServiceModule.getConfigService());

        if (!essentialServiceModule.getConfigService().isSdkEnabled()) {
            internalEmbraceLogger.logDeveloper("Embrace", "the SDK is disabled");
            stop();
            return;
        }

        this.exceptionsService.setConfigService(configService);
        this.breadcrumbService = dataCaptureServiceModule.getBreadcrumbService();
        this.pushNotificationService = dataCaptureServiceModule.getPushNotificationService();
        serviceRegistry.registerServices(breadcrumbService, pushNotificationService, dataCaptureServiceModule.getAppExitInfoService());

        this.userService = essentialServiceModule.getUserService();
        serviceRegistry.registerServices(userService);

        CustomerLogModuleImpl customerLogModule = new CustomerLogModuleImpl(
            essentialServiceModule,
            coreModule,
            deliveryModule,
            sessionProperties,
            dataCaptureServiceModule,
            workerThreadModule
        );
        this.remoteLogger = customerLogModule.getRemoteLogger();
        this.networkCaptureService = customerLogModule.getNetworkCaptureService();
        this.networkLoggingService = customerLogModule.getNetworkLoggingService();
        serviceRegistry.registerServices(
            customerLogModule.getScreenshotService(),
            remoteLogger,
            networkCaptureService,
            networkLoggingService
        );

        NativeModule nativeModule = new NativeModuleImpl(
            coreModule,
            essentialServiceModule,
            deliveryModule,
            sessionProperties,
            workerThreadModule
        );

        DataContainerModule dataContainerModule = new DataContainerModuleImpl(
            essentialServiceModule,
            coreModule,
            dataCaptureServiceModule,
            anrModule,
            customerLogModule,
            deliveryModule,
            nativeModule,
            sessionProperties,
            workerThreadModule,
            startTime
        );

        this.spansService = dataContainerModule.getSpansService();
        this.eventService = dataContainerModule.getEventService();
        serviceRegistry.registerServices(
            dataContainerModule.getPerformanceInfoService(),
            spansService,
            eventService
        );

        this.ndkService = nativeModule.getNdkService();
        nativeThreadSampler = nativeModule.getNativeThreadSamplerService();
        nativeThreadSamplerInstaller = nativeModule.getNativeThreadSamplerInstaller();

        serviceRegistry.registerServices(
            ndkService,
            nativeThreadSampler
        );

        if (nativeThreadSampler != null && nativeThreadSamplerInstaller != null) {
            // install the native thread sampler
            nativeThreadSampler.setupNativeSampler();

            // In Unity this should always run on the Unity thread.
            if (coreModule.getAppFramework() == Embrace.AppFramework.UNITY && EmbraceNativeThreadSamplerServiceKt.isUnityMainThread()) {
                sampleCurrentThreadDuringAnrs();
            }
        } else {
            internalEmbraceLogger.logDeveloper("Embrace", "Failed to load SO file embrace-native");
        }

        SessionModule sessionModule = new SessionModuleImpl(
            coreModule,
            essentialServiceModule,
            nativeModule,
            dataContainerModule,
            deliveryModule,
            sessionProperties,
            dataCaptureServiceModule,
            customerLogModule,
            sdkObservabilityModule,
            workerThreadModule
        );

        this.sessionService = sessionModule.getSessionService();
        this.backgroundActivityService = sessionModule.getBackgroundActivityService();
        serviceRegistry.registerServices(sessionService, backgroundActivityService);

        if (backgroundActivityService != null) {
            internalEmbraceLogger.logInfo("Background activity capture enabled");
        } else {
            internalEmbraceLogger.logInfo("Background activity capture disabled");
        }

        CrashModule crashModule = new CrashModuleImpl(
            essentialServiceModule,
            deliveryModule,
            nativeModule,
            sessionModule,
            anrModule,
            dataContainerModule,
            coreModule
        );

        loadCrashVerifier(crashModule, workerThreadModule);

        Thread.setDefaultUncaughtExceptionHandler(crashModule.getAutomaticVerificationExceptionHandler());
        serviceRegistry.registerService(crashModule.getCrashService());

        StrictModeService strictModeService = dataCaptureServiceModule.getStrictModeService();
        serviceRegistry.registerService(strictModeService);
        strictModeService.start();

        serviceRegistry.registerService(dataCaptureServiceModule.getThermalStatusService());

        ActivityLifecycleBreadcrumbService collector = dataCaptureServiceModule.getActivityLifecycleBreadcrumbService();
        if (collector instanceof Application.ActivityLifecycleCallbacks) {
            coreModule.getApplication().registerActivityLifecycleCallbacks((Application.ActivityLifecycleCallbacks) collector);
            serviceRegistry.registerService(collector);
        }

        // initialize internal interfaces
        InternalInterfaceModuleImpl internalInterfaceModule = new InternalInterfaceModuleImpl(
            coreModule,
            essentialServiceModule,
            this,
            crashModule
        );
        EmbraceInternalInterface embraceInternalInterface = internalInterfaceModule.getEmbraceInternalInterface();
        reactNativeInternalInterface = internalInterfaceModule.getReactNativeInternalInterface();
        unityInternalInterface = internalInterfaceModule.getUnityInternalInterface();
        flutterInternalInterface = internalInterfaceModule.getFlutterInternalInterface();

        String startMsg = "Embrace SDK started. App ID: " + configService.getSdkModeBehavior().getAppId() +
            " Version: " + BuildConfig.VERSION_NAME;
        internalEmbraceLogger.logInfo(startMsg);

        NetworkBehavior networkBehavior = configService.getNetworkBehavior();
        if (networkBehavior.isNativeNetworkingMonitoringEnabled()) {
            // Intercept Android network calls
            internalEmbraceLogger.logDeveloper("Embrace", "Native Networking Monitoring enabled");
            StreamHandlerFactoryInstaller.registerFactory(networkBehavior.isRequestContentLengthCaptureEnabled());
        }

        final long endTime = sdkClock.now();
        started.set(true);

        workerThreadModule.backgroundWorker(WorkerName.BACKGROUND_REGISTRATION).submit(() -> {
            ((EmbraceSpansService) spansService).initializeService(TimeUnit.MILLISECONDS.toNanos(startTime),
                TimeUnit.MILLISECONDS.toNanos(endTime));
            return null;
        });

        long startupDuration = endTime - startTime;
        ((EmbraceSessionService) sessionService).setSdkStartupDuration(startupDuration);
        internalEmbraceLogger.logDeveloper("Embrace", "Startup duration: " + startupDuration + " millis");

        // Sets up the registered services. This method is called after the SDK has been started and
        // no more services can be added to the registry. It sets listeners for any services that were
        // registered.
        serviceRegistry.closeRegistration();
        serviceRegistry.registerActivityListeners(activityService);
        serviceRegistry.registerConfigListeners(configService);
        serviceRegistry.registerMemoryCleanerListeners(essentialServiceModule.getMemoryCleanerService());

        // Attempt to send the startup event if the app is already in the foreground. We registered to send this when
        // we went to the foreground, but if an activity had already gone to the foreground, we may have missed
        // sending this, so to ensure the startup message is sent, we force it to be sent here.
        if (!activityService.isInBackground()) {
            internalEmbraceLogger.logDeveloper("Embrace", "Sending startup moment");
            eventService.sendStartupMoment();
        }
    }

    /**
     * Whether or not the SDK has been started.
     *
     * @return true if the SDK is started, false otherwise
     */
    public boolean isStarted() {
        return started.get();
    }

    /**
     * Sets a custom app ID that overrides the one specified at build time. Must be called before
     * the SDK is started.
     *
     * @param appId custom app ID
     * @return true if the app ID could be set, false otherwise.
     */
    public boolean setAppId(@NonNull String appId) {
        if (isStarted()) {
            internalEmbraceLogger.logError("You must set the custom app ID before the SDK is started.");
            return false;
        }
        if (appId == null || appId.isEmpty()) {
            internalEmbraceLogger.logError("App ID cannot be null or empty.");
            return false;
        }
        if (!MetadataUtils.isValidAppId(appId)) {
            internalEmbraceLogger.logError("Invalid app ID. Must be a 5-character string with " +
                "characters from the set [A-Za-z0-9], but it was \"" + appId + "\".");
            return false;
        }

        customAppId = appId;
        internalEmbraceLogger.logDeveloper("Embrace", "App Id set");
        return true;
    }

    /**
     * Shuts down the Embrace SDK.
     */
    void stop() {
        if (started.compareAndSet(true, false)) {
            internalEmbraceLogger.logInfo("Shutting down Embrace SDK.");
            try {
                this.application = null;
                internalEmbraceLogger.logDeveloper("Embrace", "Attempting to close services...");
                serviceRegistry.close();
                internalEmbraceLogger.logDeveloper("Embrace", "Services closed");
                workerThreadModule.close();
            } catch (Exception ex) {
                internalEmbraceLogger.logError("Error while shutting down Embrace SDK", ex);
            }
        }
    }

    /**
     * Sets the user ID. This would typically be some form of unique identifier such as a UUID or
     * database key for the user.
     *
     * @param userId the unique identifier for the user
     */
    public void setUserIdentifier(@Nullable String userId) {
        if (isStarted()) {
            if (!configService.getDataCaptureEventBehavior().isMessageTypeEnabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring identifier update.");
                return;
            }
            userService.setUserIdentifier(userId);
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
            if (userId != null) {
                internalEmbraceLogger.logDebug("Set user ID to " + userId);
            } else {
                internalEmbraceLogger.logDebug("Cleared user ID by setting to null");
            }
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set user identifier");
        }
    }

    /**
     * Clears the currently set user ID. For example, if the user logs out.
     */
    public void clearUserIdentifier() {
        if (isStarted()) {
            if (!configService.getDataCaptureEventBehavior().isMessageTypeEnabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring identifier update.");
                return;
            }
            userService.clearUserIdentifier();
            internalEmbraceLogger.logDebug("Cleared user ID");
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user identifier");
        }
    }

    /**
     * Sets the current user's email address.
     *
     * @param email the email address of the current user
     */
    public void setUserEmail(@Nullable String email) {
        if (isStarted()) {
            if (!configService.getDataCaptureEventBehavior().isMessageTypeEnabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring email update.");
                return;
            }
            userService.setUserEmail(email);
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
            if (email != null) {
                internalEmbraceLogger.logDebug("Set email to " + email);
            } else {
                internalEmbraceLogger.logDebug("Cleared email by setting to null");
            }
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user email");
        }
    }

    /**
     * Clears the currently set user's email address.
     */
    public void clearUserEmail() {
        if (isStarted()) {
            if (!configService.getDataCaptureEventBehavior().isMessageTypeEnabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring email update.");
                return;
            }
            userService.clearUserEmail();
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
            internalEmbraceLogger.logDebug("Cleared email");
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user email");
        }
    }

    /**
     * Sets this user as a paying user. This adds a persona to the user's identity.
     */
    public void setUserAsPayer() {
        if (isStarted()) {
            if (!configService.getDataCaptureEventBehavior().isMessageTypeEnabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring payer user update.");
                return;
            }
            userService.setUserAsPayer();
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set user as payer");
        }
    }

    /**
     * Clears this user as a paying user. This would typically be called if a user is no longer
     * paying for the service and has reverted back to a basic user.
     */
    public void clearUserAsPayer() {
        if (isStarted()) {
            if (!configService.getDataCaptureEventBehavior().isMessageTypeEnabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring payer user update.");
                return;
            }
            userService.clearUserAsPayer();
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user as payer");
        }
    }

    /**
     * Sets a custom user persona. A persona is a trait associated with a given user.
     *
     * @param persona the persona to set
     */
    public void setUserPersona(@NonNull String persona) {
        if (isStarted()) {
            if (!configService.getDataCaptureEventBehavior().isMessageTypeEnabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning(ERROR_USER_UPDATES_DISABLED);
                return;
            }
            userService.setUserPersona(persona);
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set user persona");
        }
    }

    /**
     * Clears the custom user persona, if it is set.
     *
     * @param persona the persona to clear
     */
    public void clearUserPersona(@NonNull String persona) {
        if (isStarted()) {
            if (!configService.getDataCaptureEventBehavior().isMessageTypeEnabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning(ERROR_USER_UPDATES_DISABLED);
                return;
            }
            userService.clearUserPersona(persona);
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user persona");
        }
    }

    /**
     * Clears all custom user personas from the user.
     */
    public void clearAllUserPersonas() {
        if (isStarted()) {
            if (!configService.getDataCaptureEventBehavior().isMessageTypeEnabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning(ERROR_USER_UPDATES_DISABLED);
                return;
            }
            userService.clearAllUserPersonas();
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear user personas");
        }
    }

    /**
     * Adds a property to the current session.
     */
    public boolean addSessionProperty(@NonNull String key, @NonNull String value, boolean permanent) {
        if (isStarted()) {
            return sessionService.addProperty(key, value, permanent);
        }
        internalEmbraceLogger.logSDKNotInitialized("cannot add session property");
        return false;
    }

    /**
     * Removes a property from the current session.
     */
    public boolean removeSessionProperty(@NonNull String key) {
        if (isStarted()) {
            return sessionService.removeProperty(key);
        }

        internalEmbraceLogger.logSDKNotInitialized("remove session property");
        return false;
    }

    /**
     * Retrieves a map of the current session properties.
     */
    @Nullable
    public Map<String, String> getSessionProperties() {
        if (isStarted()) {
            return sessionService.getProperties();
        }

        internalEmbraceLogger.logSDKNotInitialized("gets session properties");
        return null;
    }

    /**
     * Sets the username of the currently logged in user.
     *
     * @param username the username to set
     */
    public void setUsername(@Nullable String username) {
        if (isStarted()) {
            if (!configService.getDataCaptureEventBehavior().isMessageTypeEnabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring username update.");
                return;
            }
            userService.setUsername(username);
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
            if (username != null) {
                internalEmbraceLogger.logDebug("Set username to " + username);
            } else {
                internalEmbraceLogger.logDebug("Cleared username by setting to null");
            }
        } else {
            internalEmbraceLogger.logSDKNotInitialized("set username");
        }
    }

    /**
     * Clears the username of the currently logged in user, for example if the user has logged out.
     */
    public void clearUsername() {
        if (isStarted()) {
            if (!configService.getDataCaptureEventBehavior().isMessageTypeEnabled(MessageType.USER)) {
                internalEmbraceLogger.logWarning("User updates are disabled, ignoring username update.");
                return;
            }
            userService.clearUsername();
            // Update user info in NDK service
            ndkService.onUserInfoUpdate();
            internalEmbraceLogger.logDebug("Cleared username");
        } else {
            internalEmbraceLogger.logSDKNotInitialized("clear username");
        }
    }

    /**
     * Starts an event or 'moment'. Events are used for encapsulating particular activities within
     * the app, such as a user adding an item to their shopping cart.
     * <p>
     * The length of time an event takes to execute is recorded, and a screenshot can be taken if
     * an event is 'late'.
     *
     * @param name            a name identifying the event
     * @param identifier      an identifier distinguishing between multiple events with the same name
     * @param allowScreenshot true if a screenshot should be taken for a late event, false otherwise
     * @param properties      custom key-value pairs to provide with the event
     */
    public void startEvent(@NonNull String name,
                           @Nullable String identifier,
                           boolean allowScreenshot,
                           @Nullable Map<String, Object> properties) {
        if (isStarted()) {
            eventService.startEvent(name, identifier, allowScreenshot, normalizeProperties(properties));
            onActivityReported();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log event");
        }
    }

    /**
     * Signals the end of an event with the specified name.
     * <p>
     * The duration of the event is computed, and a screenshot taken (if enabled) if the event was
     * late.
     *
     * @param name       the name of the event to end
     * @param identifier the identifier of the event to end, distinguishing between events with the same name
     * @param properties custom key-value pairs to provide with the event
     */
    public void endEvent(@NonNull String name, @Nullable String identifier, @Nullable Map<String, Object> properties) {
        if (isStarted()) {
            eventService.endEvent(name, identifier, normalizeProperties(properties));
            onActivityReported();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log event");
        }
    }

    /**
     * Signals that the app has completed startup.
     *
     * @param properties properties to include as part of the startup moment
     */
    public void endAppStartup(@Nullable Map<String, Object> properties) {
        endEvent(STARTUP_EVENT_NAME, null, properties);
    }

    /**
     * Retrieve the HTTP request header to extract trace ID from.
     *
     * @return the Trace ID header.
     */
    @NonNull
    public String getTraceIdHeader() {
        if (isStarted() && configService != null) {
            return configService.getNetworkBehavior().getTraceIdHeader();
        }
        return NetworkBehavior.CONFIG_TRACE_ID_HEADER_DEFAULT_VALUE;
    }

    /**
     * Manually log a network request. In most cases the Embrace SDK.
     *
     * @param url           the URL of the network call
     * @param httpMethod    the int value of the HTTP method of the network call
     * @param startTime     the time that the network call started
     * @param endTime       the time that the network call was completed
     * @param bytesSent     the number of bytes sent as part of the network call
     * @param bytesReceived the number of bytes returned by the server in response to the network call
     * @param statusCode    the status code returned by the server
     * @param error         the error returned by the exception
     */
    public void logNetworkRequest(@NonNull String url,
                                  int httpMethod,
                                  long startTime,
                                  long endTime,
                                  int bytesSent,
                                  int bytesReceived,
                                  int statusCode,
                                  @Nullable String error) {
        EmbraceNetworkRequestV2.Builder requestBuilder = EmbraceNetworkRequestV2.newBuilder()
            .withUrl(url)
            .withHttpMethod(httpMethod)
            .withStartTime(startTime)
            .withEndTime(endTime)
            .withBytesIn(bytesReceived)
            .withBytesOut(bytesSent)
            .withResponseCode(statusCode);

        if (error != null && !error.isEmpty()) {
            internalEmbraceLogger.logDeveloper("Embrace", "Log network with error: " + error);
            requestBuilder.withError(new Throwable(error));
        } else {
            internalEmbraceLogger.logDeveloper("Embrace", "Log network request without errors");
        }

        logNetworkRequest(requestBuilder.build());
    }

    /**
     * Manually log a network request. In most cases the Embrace SDK.
     *
     * @param request An EmbraceNetworkRequestV2 with at least the following set: url, method, start time,
     *                end time, and either status code or error
     */
    public void logNetworkRequest(@NonNull EmbraceNetworkRequestV2 request) {
        if (isStarted()) {
            if (request == null) {
                internalEmbraceLogger.logDeveloper("Embrace", "Request is null");
                return;
            }
            if (!request.canSend()) {
                internalEmbraceLogger.logDeveloper("Embrace", "Request can't be sent");
                return;
            }
            if (request.getError() != null) {
                networkLoggingService.logNetworkError(
                    request.getUrl(),
                    request.getHttpMethod(),
                    request.getStartTime(),
                    request.getEndTime() != null ? request.getEndTime() : 0,
                    request.getError().getClass().getCanonicalName(),
                    request.getError().getLocalizedMessage(),
                    request.getTraceId(),
                    null);
            } else {
                networkLoggingService.logNetworkCall(
                    request.getUrl(),
                    request.getHttpMethod(),
                    request.getResponseCode() != null ? request.getResponseCode() : 0,
                    request.getStartTime(),
                    request.getEndTime() != null ? request.getEndTime() : 0,
                    request.getBytesOut(),
                    request.getBytesIn(),
                    request.getTraceId(),
                    null);
            }
            onActivityReported();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log network request");
        }
    }

    /**
     * Manually log a network request. In most cases the Embrace SDK.
     *
     * @param request An EmbraceNetworkRequest with at least the following set: url, method, start time,
     *                end time, and either status code or error
     */
    public void logNetworkRequest(@NonNull EmbraceNetworkRequest request) {
        if (isStarted()) {
            if (request == null) {
                internalEmbraceLogger.logDeveloper("Embrace", "Request is null");
                return;
            }
            if (!request.canSend()) {
                internalEmbraceLogger.logDeveloper("Embrace", "Request can't be sent");
                return;
            }
            if (request.getError() != null) {
                networkLoggingService.logNetworkError(
                    request.getUrl(),
                    request.getHttpMethod(),
                    request.getStartTime(),
                    request.getEndTime() != null ? request.getEndTime() : 0,
                    request.getError().getClass().getCanonicalName(),
                    request.getError().getLocalizedMessage(),
                    request.getTraceId(),
                    null);
            } else {
                networkLoggingService.logNetworkCall(
                    request.getUrl(),
                    request.getHttpMethod(),
                    request.getResponseCode() != null ? request.getResponseCode() : 0,
                    request.getStartTime(),
                    request.getEndTime() != null ? request.getEndTime() : 0,
                    request.getBytesOut(),
                    request.getBytesIn(),
                    request.getTraceId(),
                    null);
            }
            onActivityReported();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log network request");
        }
    }

    /**
     * Logs the fact that a network call occurred. These are recorded and sent to Embrace as part
     * of a particular session.
     *
     * @param url                the URL of the network call
     * @param httpMethod         the HTTP method of the network call
     * @param statusCode         the status code returned by the server
     * @param startTime          the time that the network call started
     * @param endTime            the time that the network call was completed
     * @param bytesSent          the number of bytes sent as part of the network call
     * @param bytesReceived      the number of bytes returned by the server in response to the network call
     * @param traceId            the optional trace id that can be used to trace a particular request
     * @param networkCaptureData the network body captured to log.
     */
    public void logNetworkCall(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        int statusCode,
        long startTime,
        long endTime,
        long bytesSent,
        long bytesReceived,
        @Nullable String traceId,
        @Nullable NetworkCaptureData networkCaptureData) {

        if (ApkToolsConfig.IS_NETWORK_CAPTURE_DISABLED) {
            return;
        }
        if (isStarted()) {
            internalEmbraceLogger.logDeveloper("Embrace", "Attempting to log network call");

            if (!configService.getNetworkBehavior().isUrlEnabled(url)) {
                internalEmbraceLogger.logWarning("Recording of network calls disabled for url: " + url);
                return;
            }

            internalEmbraceLogger.logDeveloper("Embrace", "Log network call");
            networkLoggingService.logNetworkCall(
                url,
                httpMethod.name(),
                statusCode,
                startTime,
                endTime,
                bytesSent,
                bytesReceived,
                traceId,
                networkCaptureData);
            onActivityReported();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log network call");
        }
    }

    /**
     * Logs the fact that an exception was thrown when attempting to make a network call.
     * <p>
     * These are client-side exceptions and not server-side exceptions, such as a DNS error or
     * failure to connect to the remote server.
     *
     * @param url          the URL of the network call
     * @param httpMethod   the HTTP method of the network call
     * @param startTime    the time that the network call started
     * @param endTime      the time that the network call was completed
     * @param errorType    the type of the exception
     * @param errorMessage the message returned by the exception
     * @param traceId      the optional trace id that can be used to trace a particular request
     */
    public void logNetworkClientError(
        @NonNull String url,
        @NonNull HttpMethod httpMethod,
        long startTime,
        long endTime,
        @NonNull String errorType,
        @NonNull String errorMessage,
        @Nullable String traceId,
        @Nullable NetworkCaptureData networkCaptureData) {
        internalEmbraceLogger.logDeveloper("Embrace", "Attempting to log network client error");
        if (isStarted()) {
            if (!configService.getNetworkBehavior().isUrlEnabled(url)) {
                internalEmbraceLogger.logWarning("Recording of network calls disabled for url: " + url);
                return;
            }

            internalEmbraceLogger.logDeveloper("Embrace", "Log network client error");
            networkLoggingService.logNetworkError(
                url,
                httpMethod.name(),
                startTime,
                endTime,
                errorType,
                errorMessage,
                traceId,
                networkCaptureData);
            onActivityReported();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log network error");
        }
    }

    void logMessage(
        @NonNull EmbraceEvent.Type type,
        @NonNull String message,
        @Nullable Map<String, Object> properties,
        boolean allowScreenshot,
        @Nullable StackTraceElement[] stackTraceElements,
        @Nullable String customStackTrace,
        @NonNull LogExceptionType logExceptionType,
        @Nullable String context,
        @Nullable String library) {
        logMessage(type,
            message,
            properties,
            allowScreenshot,
            stackTraceElements,
            customStackTrace,
            logExceptionType,
            context,
            library,
            null,
            null);
    }

    void logMessage(
        @NonNull EmbraceEvent.Type type,
        @NonNull String message,
        @Nullable Map<String, Object> properties,
        boolean allowScreenshot,
        @Nullable StackTraceElement[] stackTraceElements,
        @Nullable String customStackTrace,
        @NonNull LogExceptionType logExceptionType,
        @Nullable String context,
        @Nullable String library,
        @Nullable String exceptionName,
        @Nullable String exceptionMessage) {
        internalEmbraceLogger.logDeveloper("Embrace", "Attempting to log message");
        if (isStarted()) {
            try {
                this.remoteLogger.log(
                    message,
                    type,
                    allowScreenshot,
                    logExceptionType,
                    normalizeProperties(properties),
                    stackTraceElements,
                    customStackTrace,
                    this.appFramework,
                    context,
                    library,
                    exceptionName,
                    exceptionMessage);
                onActivityReported();
            } catch (Exception ex) {
                internalEmbraceLogger.logDebug("Failed to log message using Embrace SDK.", ex);
            }
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log message");
        }
    }

    /**
     * Logs a breadcrumb.
     * <p>
     * Breadcrumbs track a user's journey through the application and will be shown on the timeline.
     *
     * @param message the name of the breadcrumb to log
     */
    public void logBreadcrumb(@NonNull String message) {
        internalEmbraceLogger.logDeveloper("Embrace", "Attempting to log breadcrumb");
        if (isStarted()) {
            this.breadcrumbService.logCustom(message, sdkClock.now());
            onActivityReported();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("log breadcrumb");
        }
    }

    /**
     * Logs a React Native Redux Action.
     */
    public void logRnAction(@NonNull String name, long startTime, long endTime,
                            @NonNull Map<String, Object> properties, int bytesSent, @NonNull String output) {
        if (isStarted()) {
            this.breadcrumbService.logRnAction(name, startTime, endTime, properties, bytesSent, output);
        } else {
            internalEmbraceLogger.logWarning("Embrace SDK is not initialized yet, cannot log breadcrumb.");
        }
    }

    /**
     * Logs a javascript unhandled exception.
     *
     * @param name       name of the exception.
     * @param message    exception message.
     * @param type       error type.
     * @param stacktrace exception stacktrace.
     */
    @InternalApi
    public void logUnhandledJsException(@NonNull String name, @NonNull String message, @Nullable String type, @Nullable String stacktrace) {
        if (reactNativeInternalInterface != null) {
            reactNativeInternalInterface.logUnhandledJsException(name, message, type, stacktrace);
            onActivityReported();
        }
    }

    /**
     * Logs a Unity unhandled exception.
     *
     * @param name       exception name. @Nullable for backwards compatibility.
     * @param message    exception message.
     * @param stacktrace exception stacktrace.
     */
    @InternalApi
    public void logUnityException(@Nullable String name, @NonNull String message, @Nullable String stacktrace,
                                  @NonNull LogExceptionType logExceptionType) {
        if (unityInternalInterface != null) {
            String sanitizedName = name == null ? "" : name;
            if (logExceptionType == LogExceptionType.UNHANDLED) {
                unityInternalInterface.logUnhandledUnityException(sanitizedName, message, stacktrace);
            } else if (logExceptionType == LogExceptionType.HANDLED) {
                unityInternalInterface.logHandledUnityException(sanitizedName, message, stacktrace);
            }
            onActivityReported();
        }
    }

    /**
     * Sets the react native version number.
     *
     * @param version react native version number.
     */
    public void setReactNativeVersionNumber(@NonNull String version) {
        if (reactNativeInternalInterface != null) {
            reactNativeInternalInterface.setReactNativeVersionNumber(version);
        }
    }

    /**
     * Sets javascript patch number.
     *
     * @param number javascript patch number.
     */
    public void setJavaScriptPatchNumber(@NonNull String number) {
        if (reactNativeInternalInterface != null) {
            reactNativeInternalInterface.setJavaScriptPatchNumber(number);
        }
    }

    /**
     * Sets the Embrace RN SDK version.
     *
     * @param version embrace sdk version.
     */
    public void setReactNativeSdkVersion(@NonNull String version) {
        if (isStarted()) {
            metadataService.setRnSdkVersion(version);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("setReactNativeSDKVersion");
        }
    }

    /**
     * Sets the path of the javascript bundle.
     *
     * @param url path of the javascript bundle.
     */
    @SuppressWarnings("AbbreviationAsWordInNameCheck")
    public void setJavaScriptBundleURL(@NonNull String url) {
        if (reactNativeInternalInterface != null) {
            reactNativeInternalInterface.setJavaScriptBundleUrl(application, url);
        }
    }

    /**
     * Sets the Unity version, Unity build id and Unity SDK version.
     *
     * @param unityVersion    of the Unity Editor
     * @param buildGuid       if the Unity build
     * @param unitySdkVersion of the Unity SDK
     */
    @InternalApi
    public void setUnityMetaData(@NonNull String unityVersion, @NonNull String buildGuid, @Nullable String unitySdkVersion) {
        if (unityInternalInterface != null) {
            unityInternalInterface.setUnityMetaData(unityVersion, buildGuid, unitySdkVersion);
        }
    }

    /**
     * Logs an internal error to the Embrace SDK - this is not intended for public use.
     */
    @InternalApi
    public void logInternalError(@Nullable String message, @Nullable String details) {
        if (isStarted()) {
            if (message == null) {
                return;
            }
            String msg;

            if (details != null) {
                msg = message + ": " + details;
            } else {
                msg = message;
            }
            exceptionsService.handleInternalError(new InternalErrorLogger.DartError(msg));
        } else {
            internalEmbraceLogger.logSDKNotInitialized("logInternalError");
        }
    }

    /**
     * Logs a Dart error to the Embrace SDK - this is not intended for public use.
     */
    @InternalApi
    public void logDartError(
        @Nullable String stack,
        @Nullable String name,
        @Nullable String message,
        @Nullable String context,
        @Nullable String library,
        @NonNull LogExceptionType logExceptionType
    ) {
        if (flutterInternalInterface != null) {
            if (logExceptionType == LogExceptionType.HANDLED) {
                flutterInternalInterface.logHandledDartError(stack, name, message, context, library);
            } else if (logExceptionType == LogExceptionType.UNHANDLED) {
                flutterInternalInterface.logUnhandledDartError(stack, name, message, context, library);
            }
            onActivityReported();
        }
    }

    /**
     * Sets the Embrace Flutter SDK version - this is not intended for public use.
     */
    @InternalApi
    public void setEmbraceFlutterSdkVersion(@Nullable String version) {
        if (flutterInternalInterface != null) {
            flutterInternalInterface.setEmbraceFlutterSdkVersion(version);
        }
    }

    /**
     * Sets the Dart version - this is not intended for public use.
     */
    @InternalApi
    public void setDartVersion(@Nullable String version) {
        if (flutterInternalInterface != null) {
            flutterInternalInterface.setDartVersion(version);
        }
    }

    /**
     * Registers a {@link ConnectionQualityListener}, notifying the listener each time that there is
     * a change in the connection quality.
     *
     * @param listener the listener to register
     */
    @Deprecated
    public void addConnectionQualityListener(@NonNull ConnectionQualityListener listener) {
        internalEmbraceLogger.logWarning("Warning: failed to remove connection quality listener. " +
            "The signal quality service is deprecated.");
    }

    /**
     * Removes a registered {@link ConnectionQualityListener}, suspending connection quality
     * notifications.
     *
     * @param listener the listener to remove
     */
    @Deprecated
    public void removeConnectionQualityListener(@NonNull ConnectionQualityListener listener) {
        internalEmbraceLogger.logWarning("Warning: failed to remove connection quality listener. " +
            "The signal quality service is deprecated.");
    }

    /**
     * Ends the current session and starts a new one.
     * <p>
     * Cleans all the user info on the device.
     */
    public synchronized void endSession(boolean clearUserInfo) {
        if (isStarted()) {
            SessionBehavior sessionBehavior = configService.getSessionBehavior();
            if (sessionBehavior.getMaxSessionSecondsAllowed() != null) {
                internalEmbraceLogger.logWarning("Can't close the session, automatic session close enabled.");
                return;
            }

            if (sessionBehavior.isAsyncEndEnabled()) {
                internalEmbraceLogger.logWarning("Can't close the session, session ending in background thread enabled.");
                return;
            }

            if (clearUserInfo) {
                userService.clearAllUserInfo();
                // Update user info in NDK service
                ndkService.onUserInfoUpdate();
            }

            sessionService.triggerStatelessSessionEnd(Session.SessionLifeEventType.MANUAL);
        } else {
            internalEmbraceLogger.logSDKNotInitialized("end session");
        }
    }

    /**
     * Get the user identifier assigned to the device by Embrace
     *
     * @return the device identifier created by Embrace
     */
    @NonNull
    public String getDeviceId() {
        return preferencesService.getDeviceIdentifier();
    }

    /**
     * Causes a crash with an exception. Use this for test purposes only
     */
    public void throwException() {
        throw new RuntimeException("EmbraceException", new Throwable("Embrace test exception"));
    }

    /**
     * Log the start of a fragment.
     * <p>
     * A matching call to endFragment must be made.
     *
     * @param name the name of the fragment to log
     */
    public boolean startFragment(@NonNull String name) {
        if (isStarted()) {
            internalEmbraceLogger.logDeveloper("Embrace", "Starting fragment: " + name);
            return this.breadcrumbService.startFragment(name);
        }

        internalEmbraceLogger.logDeveloper("Embrace", "Cannot start fragment, SDK is not started");
        return false;
    }

    /**
     * Log the end of a fragment.
     * <p>
     * A matching call to startFragment must be made before this is called.
     *
     * @param name the name of the fragment to log
     */
    public boolean endFragment(@NonNull String name) {
        if (isStarted()) {
            internalEmbraceLogger.logDeveloper("Embrace", "Ending fragment: " + name);
            return this.breadcrumbService.endFragment(name);
        }

        internalEmbraceLogger.logDeveloper("Embrace", "Cannot end fragment, SDK is not started");
        return false;
    }

    @InternalApi
    public void sampleCurrentThreadDuringAnrs() {
        try {
            AnrService service = anrService;
            if (service != null && nativeThreadSamplerInstaller != null) {
                nativeThreadSamplerInstaller.monitorCurrentThread(
                    nativeThreadSampler,
                    configService,
                    service
                );
            } else {
                internalEmbraceLogger.logDeveloper("Embrace", "nativeThreadSamplerInstaller not started, cannot sample current thread");
            }
        } catch (Exception exc) {
            internalEmbraceLogger.logError("Failed to sample current thread during ANRs", exc);
        }
    }

    /**
     * Logs the fact that a particular view was entered.
     * <p>
     * If the previously logged view has the same name, a duplicate view breadcrumb will not be
     * logged.
     *
     * @param screen the name of the view to log
     */
    void logView(String screen) {
        if (isStarted()) {
            this.breadcrumbService.logView(screen, sdkClock.now());
            onActivityReported();
        }

        internalEmbraceLogger.logDeveloper("Embrace", "SDK not started, cannot log view");
    }

    /**
     * Logs the fact that a particular view was entered.
     * <p>
     * If the previously logged view has the same name, a duplicate view breadcrumb will not be
     * logged.
     *
     * @param screen the name of the view to log
     */
    public void logRnView(@NonNull String screen) {
        if (this.appFramework != Embrace.AppFramework.REACT_NATIVE) {
            InternalStaticEmbraceLogger.logWarning("[Embrace] logRnView is only available on React Native");
            return;
        }

        this.logView(screen);
    }

    /**
     * Logs that a particular WebView URL was loaded.
     *
     * @param url the url to log
     */
    void logWebView(String url) {
        if (isStarted()) {
            this.breadcrumbService.logWebView(url, sdkClock.now());
            onActivityReported();
        }

        internalEmbraceLogger.logDeveloper("Embrace", "SDK not started, cannot log view");
    }

    /**
     * Logs the fact that a particular view was entered.
     * <p>
     * If the previously logged view has the same name, a duplicate view breadcrumb will be
     * logged, and not treated as a duplicate.
     *
     * @param screen the name of the view to log
     */
    void forceLogView(String screen) {
        if (isStarted()) {
            this.breadcrumbService.forceLogView(screen, sdkClock.now());
            onActivityReported();
        } else {
            internalEmbraceLogger.logDeveloper("Embrace", "SDK not started, cannot force log view");
        }
    }

    /**
     * Logs a tap on a screen element.
     *
     * @param point       the coordinates of the screen tap
     * @param elementName the name of the element which was tapped
     * @param type        the type of tap that occurred
     */
    void logTap(Pair<Float, Float> point, String elementName, TapBreadcrumb.TapBreadcrumbType type) {
        if (isStarted()) {
            this.breadcrumbService.logTap(point, elementName, sdkClock.now(), type);
            onActivityReported();
        } else {
            internalEmbraceLogger.logDeveloper("Embrace", "SDK not started, cannot log tap");
        }
    }

    @Nullable
    @InternalApi
    public ConfigService getConfigService() {
        if (isStarted()) {
            return configService;
        } else {
            internalEmbraceLogger.logSDKNotInitialized("get local config");
        }
        return null;
    }

    EventService getEventService() {
        return eventService;
    }

    ActivityService getActivityService() {
        return activityService;
    }

    EmbraceRemoteLogger getRemoteLogger() {
        return remoteLogger;
    }

    EmbraceInternalErrorService getExceptionsService() {
        return exceptionsService;
    }

    MetadataService getMetadataService() {
        return metadataService;
    }

    SessionService getSessionService() {
        return sessionService;
    }

    Application getApplication() {
        return application;
    }

    @Nullable
    private Map<String, Object> normalizeProperties(@Nullable Map<String, Object> properties) {
        Map<String, Object> normalizedProperties = new HashMap<>();
        if (properties != null) {
            try {
                internalEmbraceLogger.logDeveloper("Embrace", "normalizing properties");
                normalizedProperties = PropertyUtils.sanitizeProperties(properties);
            } catch (Exception e) {
                internalEmbraceLogger.logError("Exception occurred while normalizing the properties.", e);
            }
            return normalizedProperties;
        } else {
            return null;
        }
    }

    /**
     * Gets the {@link ReactNativeInternalInterface} that should be used as the sole source of
     * communication with the Android SDK for React Native.
     */
    @Nullable
    ReactNativeInternalInterface getReactNativeInternalInterface() {
        return reactNativeInternalInterface;
    }

    /**
     * Gets the {@link UnityInternalInterface} that should be used as the sole source of
     * communication with the Android SDK for Unity.
     */
    @Nullable
    UnityInternalInterface getUnityInternalInterface() {
        return unityInternalInterface;
    }

    /**
     * Gets the {@link FlutterInternalInterface} that should be used as the sole source of
     * communication with the Android SDK for Flutter.
     */
    @Nullable
    FlutterInternalInterface getFlutterInternalInterface() {
        return flutterInternalInterface;
    }

    public void installUnityThreadSampler() {
        if (isStarted()) {
            sampleCurrentThreadDuringAnrs();
        } else {
            internalEmbraceLogger.logSDKNotInitialized("installUnityThreadSampler");
        }
    }

    /**
     * Saves captured push notification information into session payload
     *
     * @param title                    the title of the notification as a string (or null)
     * @param body                     the body of the notification as a string (or null)
     * @param topic                    the notification topic (if a user subscribed to one), or null
     * @param id                       A unique ID identifying the message
     * @param notificationPriority     the notificationPriority of the message (as resolved on the device)
     * @param messageDeliveredPriority the priority of the message (as resolved on the server)
     */
    void logPushNotification(
        @Nullable String title,
        @Nullable String body,
        @Nullable String topic,
        @Nullable String id,
        @Nullable Integer notificationPriority,
        Integer messageDeliveredPriority,
        PushNotificationBreadcrumb.NotificationType type) {

        pushNotificationService.logPushNotification(
            title,
            body,
            topic,
            id,
            notificationPriority,
            messageDeliveredPriority,
            type
        );
        onActivityReported();
    }

    private void onActivityReported() {
        if (backgroundActivityService != null) {
            backgroundActivityService.save();
        }
    }

    public boolean shouldCaptureNetworkCall(String url, String method) {
        return !networkCaptureService.getNetworkCaptureRules(url, method).isEmpty();
    }

    public void setProcessStartedByNotification() {
        eventService.setProcessStartedByNotification();
    }

    public void trackWebViewPerformance(@NonNull String tag, @NonNull String message) {
        if (configService.getWebViewVitalsBehavior().isWebViewVitalsEnabled()) {
            webViewService.collectWebData(tag, message);
        }
    }

    /**
     * Get the end state of the last run of the application.
     *
     * @return LastRunEndState enum value representing the end state of the last run.
     */
    @NonNull
    public Embrace.LastRunEndState getLastRunEndState() {
        if (isStarted() && this.crashVerifier != null) {
            if (this.crashVerifier.didLastRunCrash()) {
                return Embrace.LastRunEndState.CRASH;
            } else {
                return Embrace.LastRunEndState.CLEAN_EXIT;
            }
        } else {
            return Embrace.LastRunEndState.INVALID;
        }
    }

    /**
     * Loads the crash verifier to get the end state of the app crashed in the last run.
     * This method is called when the app starts.
     *
     * @param crashModule        an instance of {@link CrashModule}
     * @param workerThreadModule an instance of {@link WorkerThreadModule}
     */
    private void loadCrashVerifier(CrashModule crashModule, WorkerThreadModule workerThreadModule) {
        this.crashVerifier = crashModule.getLastRunCrashVerifier();
        this.crashVerifier.readAndCleanMarkerAsync(
            workerThreadModule.backgroundWorker(WorkerName.BACKGROUND_REGISTRATION)
        );
    }
}
