package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.Map;

import io.embrace.android.embracesdk.network.http.NetworkCaptureData;

/**
 * Provides an internal interface to Embrace that is intended for use by hosted SDKs as their
 * sole source of communication with the Android SDK.
 */
interface EmbraceInternalInterface {

    /**
     * {@see Embrace#logInfo}
     */
    void logInfo(@NonNull String message,
                 @Nullable Map<String, Object> properties);

    /**
     * {@see Embrace#logWarning}
     */
    void logWarning(@NonNull String message,
                    @Nullable Map<String, Object> properties,
                    boolean allowScreenshot,
                    @Nullable String stacktrace);

    /**
     * {@see Embrace#logError}
     */
    void logError(@NonNull String message,
                  @Nullable Map<String, Object> properties,
                  boolean allowScreenshot,
                  @Nullable String stacktrace,
                  boolean isException);

    /**
     * {@see Embrace#logHandledException}
     */
    void logHandledException(@NonNull Throwable throwable,
                             @NonNull LogType type,
                             @Nullable Map<String, Object> properties,
                             @Nullable StackTraceElement[] customStackTrace,
                             boolean takeScreenshot);

    /**
     * {@see Embrace#logBreadcrumb}
     */
    void logBreadcrumb(@NonNull String message);

    /**
     * {@see Embrace#getDeviceId}
     */
    @NonNull
    String getDeviceId();

    /**
     * {@see Embrace#setUsername}
     */
    void setUsername(@Nullable String username);

    /**
     * {@see Embrace#clearUsername}
     */
    void clearUsername();

    /**
     * {@see Embrace#setUserIdentifier}
     */
    void setUserIdentifier(@Nullable String userId);

    /**
     * {@see Embrace#clearUserIdentifier}
     */
    void clearUserIdentifier();

    /**
     * {@see Embrace#setUserEmail}
     */
    void setUserEmail(@Nullable String email);

    /**
     * {@see Embrace#clearUserEmail}
     */
    void clearUserEmail();

    /**
     * {@see Embrace#setUserAsPayer}
     */
    void setUserAsPayer();

    /**
     * {@see Embrace#clearUserAsPayer}
     */
    void clearUserAsPayer();

    /**
     * {@see Embrace#setUserPersona}
     */
    void setUserPersona(@NonNull String persona);

    /**
     * {@see Embrace#clearUserPersona}
     */
    void clearUserPersona(@NonNull String persona);

    /**
     * {@see Embrace#clearAllUserPersonas}
     */
    void clearAllUserPersonas();

    /**
     * {@see Embrace#addSessionProperty}
     */
    boolean addSessionProperty(@NonNull String key,
                               @NonNull String value,
                               boolean permanent);

    /**
     * {@see Embrace#removeSessionProperty}
     */
    boolean removeSessionProperty(@NonNull String key);

    /**
     * {@see Embrace#getSessionProperties}
     */
    @Nullable
    Map<String, String> getSessionProperties();

    /**
     * {@see Embrace#startEvent}
     */
    void startEvent(@NonNull String name,
                    @Nullable String identifier,
                    boolean allowScreenshot,
                    @Nullable Map<String, Object> properties);

    /**
     * {@see Embrace#endEvent}
     */
    void endEvent(@NonNull String name,
                  @Nullable String identifier,
                  @Nullable Map<String, Object> properties);

    /**
     * {@see Embrace#startFragment}
     */
    boolean startView(@NonNull String name);

    /**
     * {@see Embrace#endFragment}
     */
    boolean endView(@NonNull String name);

    /**
     * {@see Embrace#endAppStartup}
     */
    void endAppStartup(@NonNull Map<String, Object> properties);

    /**
     * {@see Embrace#logInternalError}
     */
    void logInternalError(@Nullable String message, @Nullable String details);

    /**
     * {@see Embrace#endSession}
     */
    void endSession(boolean clearUserInfo);

    /**
     * {@see Embrace#logNetworkClientError}
     */
    void logNetworkClientError(
        @NonNull String url,
        @NonNull String httpMethod,
        long startTime,
        long endTime,
        @NonNull String errorType,
        @NonNull String errorMessage,
        @Nullable String traceId,
        @Nullable NetworkCaptureData networkCaptureData);

    /**
     * {@see Embrace#logNetworkRequest}
     */
    void logNetworkRequest(@NonNull String url,
                           String httpMethod,
                           long startTime,
                           long endTime,
                           int bytesSent,
                           int bytesReceived,
                           int statusCode,
                           @Nullable String error,
                           @Nullable String traceId);
}
