package io.embrace.android.embracesdk;

import android.util.Log;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;

/**
 * Wrapper for the Android {@link Log} utility. This class is part of the Embrace Public API.
 */
public class EmbraceLogger {

    /**
     * The severity of the log message.
     */
    public enum Severity {
        DEVELOPER,
        DEBUG,
        INFO,
        WARNING,
        ERROR,
        NONE
    }

    private EmbraceLogger() {
        // Restricted constructor
    }

    /**
     * Sets the logging threshold. Anything below the threshold will not be logged.
     *
     * @param threshold the threshold
     */
    public static void setThreshold(Severity threshold) {
        InternalStaticEmbraceLogger.setThreshold(threshold);
    }

    /**
     * Logs a debug message.
     *
     * @param msg the message to log
     */
    public static void logDebug(String msg) {
        log(Severity.DEBUG, msg, null, false);
    }

    /**
     * Logs a debug message and sends exception to Embrace.
     *
     * @param msg       the message to log
     * @param throwable the exception
     */
    public static void logDebug(String msg, Throwable throwable) {
        log(Severity.DEBUG, msg, throwable, false);
    }

    /**
     * Logs a debug message with a stack trace.
     *
     * @param msg           the message to log
     * @param throwable     the exception
     * @param logStacktrace include stacktrace in log
     */
    public static void logDebug(String msg, Throwable throwable, Boolean logStacktrace) {
        log(Severity.DEBUG, msg, throwable, logStacktrace);
    }

    /**
     * Logs an info message.
     *
     * @param msg the message to log
     */
    public static void logInfo(String msg) {
        log(Severity.INFO, msg, null, false);
    }

    /**
     * Logs an info message and sends exception to Embrace.
     *
     * @param msg       the message to log
     * @param throwable the exception
     */
    public static void logInfo(String msg, Throwable throwable) {
        log(Severity.INFO, msg, throwable, false);
    }

    /**
     * Logs an info message with a stack trace.
     *
     * @param msg       the message to log
     * @param throwable the exception
     */
    public static void logInfo(String msg, Throwable throwable, Boolean logStacktrace) {
        log(Severity.INFO, msg, throwable, logStacktrace);
    }

    /**
     * Logs a warning message.
     *
     * @param msg the message to log
     */
    public static void logWarning(String msg) {
        log(Severity.WARNING, msg, null, false);
    }

    /**
     * Logs a warning message and sends exception to Embrace.
     *
     * @param msg       the message to log
     * @param throwable the exception
     */
    public static void logWarning(String msg, Throwable throwable) {
        log(Severity.WARNING, msg, throwable, false);
    }

    /**
     * Logs a warning message with a stack trace.
     *
     * @param msg           the message to log
     * @param throwable     the exception
     * @param logStacktrace include stacktrace in log
     */
    public static void logWarning(String msg, Throwable throwable, Boolean logStacktrace) {
        log(Severity.WARNING, msg, throwable, logStacktrace);
    }

    /**
     * Logs an error message.
     *
     * @param msg the message to log
     */
    public static void logError(String msg) {
        log(Severity.ERROR, msg, null, false);
    }

    /**
     * Logs an error and sends exception to Embrace.
     *
     * @param msg       the message to log
     * @param throwable the exception
     */
    public static void logError(String msg, Throwable throwable) {
        log(Severity.ERROR, msg, throwable, false);
    }

    /**
     * Logs an error message with a stack trace.
     *
     * @param msg           the message to log
     * @param throwable     the exception
     * @param logStacktrace include stacktrace in log
     */
    public static void logError(String msg, Throwable throwable, Boolean logStacktrace) {
        log(Severity.ERROR, msg, throwable, logStacktrace);
    }

    /**
     * Logs a message with a stack trace at the specified severity. The 'data' attributes are
     * ignored.
     *
     * @param severity      the severity of the message
     * @param msg           the message to log
     * @param throwable     the exception
     * @param logStacktrace include stacktrace in log
     */
    public static void log(Severity severity, String msg, Throwable throwable, boolean logStacktrace) {
        if (msg == null) {
            msg = "";
        }

        String cleanedMsg = msg.replace('\n', '_').replace('\r', '_');
        InternalStaticEmbraceLogger.log(cleanedMsg, severity, throwable, logStacktrace);
    }
}
