package io.embrace.android.embracesdk;

import android.os.Parcelable;

import androidx.annotation.NonNull;

import java.io.Serializable;
import java.util.AbstractMap.SimpleEntry;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.utils.StreamUtilsKt;

/**
 * Utility to for sanitizing user-supplied properties.
 */
final class PropertyUtils {

    static final int MAX_PROPERTY_SIZE = 10;

    /**
     * This method will normalize the map by applying the following rules:
     * <p>
     * - Null key registries will be discarded.
     * - Null value registries will be renamed to null as a String.
     * - Cap the properties map to a maximum of {@link PropertyUtils#MAX_PROPERTY_SIZE} properties.
     *
     * @param properties properties to be normalized.
     * @return a normalized Map of the provided properties.
     */
    @NonNull
    static Map<String, Object> sanitizeProperties(Map<String, Object> properties) {
        if (properties == null) {
            return new HashMap<>();
        }

        if (properties.size() > MAX_PROPERTY_SIZE) {
            String msg = "The maximum number of properties is " + MAX_PROPERTY_SIZE + ", the rest will be ignored.";
            InternalStaticEmbraceLogger.logWarning(msg);
        }

        List<Entry<String, Object>> entries = StreamUtilsKt.filter(properties.entrySet(), entry -> entry.getKey() != null);
        int toIndex = Math.min(entries.size(), MAX_PROPERTY_SIZE);
        List<Entry<String, Object>> slice = entries.subList(0, toIndex);
        List<Entry<String, Object>> sanitizedEntries = StreamUtilsKt.map(slice, PropertyUtils::mapNullValue);

        Map<String, Object> map = new HashMap<>();

        for (Entry<String, Object> sanitizedEntry : sanitizedEntries) {
            map.put(sanitizedEntry.getKey(), sanitizedEntry.getValue());
        }
        return map;
    }

    private static Entry<String, Object> mapNullValue(Entry<String, Object> entry) {

        return new SimpleEntry<>(entry.getKey(), checkIfSerializable(entry.getKey(), entry.getValue()));
    }

    private static Object checkIfSerializable(String key, Object value) {

        if (value == null) {
            return "null";
        }

        if (!(value instanceof Parcelable || value instanceof Serializable)) {

            String msg = "The property with key " + key + " has an entry that cannot be serialized. It will be ignored.";
            InternalStaticEmbraceLogger.logWarning(msg);
            return "not serializable";
        }

        return value;
    }

    private PropertyUtils() {
        // Restricted constructor
    }
}
