package io.embrace.android.embracesdk

import android.content.Context
import io.embrace.android.embracesdk.Embrace.AppFramework
import io.embrace.android.embracesdk.capture.crash.CrashService
import io.embrace.android.embracesdk.logging.InternalEmbraceLogger
import io.embrace.android.embracesdk.payload.JsException

internal class ReactNativeInternalInterfaceImpl(
    private val embrace: EmbraceImpl,
    private val impl: EmbraceInternalInterface,
    private val framework: AppFramework,
    private val preferencesService: PreferencesService,
    private val crashService: CrashService,
    private val metadataService: MetadataService,
    private val logger: InternalEmbraceLogger
) : EmbraceInternalInterface by impl, ReactNativeInternalInterface {

    override fun logUnhandledJsException(
        name: String,
        message: String,
        type: String?,
        stacktrace: String?
    ) {
        if (embrace.isStarted) {
            val exception = JsException(name, message, type, stacktrace)
            logger.logDeveloper(
                "Embrace",
                "Log Unhandled JS exception: $name -- stacktrace: $stacktrace"
            )
            crashService.logUnhandledJsException(exception)
        } else {
            logger.logSDKNotInitialized("log JS exception")
        }
    }

    override fun setJavaScriptPatchNumber(number: String?) {
        if (embrace.isStarted) {
            if (number == null) {
                logger.logError("JavaScript patch number must not be null")
                return
            }
            if (number.isEmpty()) {
                logger.logError("JavaScript patch number must have non-zero length")
                return
            }
            preferencesService.javaScriptPatchNumber = number
        } else {
            logger.logSDKNotInitialized("set JavaScript patch number")
        }
    }

    override fun setReactNativeVersionNumber(version: String?) {
        if (embrace.isStarted) {
            if (version == null) {
                logger.logError("ReactNative version must not be null")
                return
            }
            if (version.isEmpty()) {
                logger.logError("ReactNative version must have non-zero length")
                return
            }
            preferencesService.reactNativeVersionNumber = version
        } else {
            logger.logSDKNotInitialized("set React Native version number")
        }
    }

    override fun setJavaScriptBundleUrl(context: Context, url: String) {
        if (embrace.isStarted) {
            if (framework != AppFramework.REACT_NATIVE) {
                logger.logError(
                    "Failed to set Java Script bundle ID URL. Current framework: " +
                        framework.name + " is not React Native."
                )
                return
            }
            metadataService.setReactNativeBundleId(context, url)
        } else {
            logger.logSDKNotInitialized("set JavaScript bundle URL")
        }
    }
}
