package io.embrace.android.embracesdk;

import androidx.annotation.NonNull;

import com.google.gson.annotations.SerializedName;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

/**
 * Breadcrumb that represents the dispatched actions from your state managment.
 */
@InternalApi
public class RnActionBreadcrumb implements Breadcrumb {

    enum RnOutputType {
        SUCCESS, FAIL, INCOMPLETE
    }

    /**
     * Method that return the valid outcomes
     */
    @NonNull
    public static String getValidRnBreadcrumbOutputName() {
        return Arrays.toString(RnOutputType.values());
    }

    /**
     * Method that validate the output is valid
     */
    public static boolean validateRnBreadcrumbOutputName(@NonNull String output) {
        for (RnOutputType rnOutput : RnOutputType.values()) {
            if (rnOutput.name().equalsIgnoreCase(output)) {
                return true;
            }
        }
        return false;
    }

    /**
     * The action name
     */
    @SerializedName("n")
    private final String name;
    /**
     * The output message SUCCESS | FAIL | INCOMPLETE
     */
    @SerializedName("o")
    private String output;

    /**
     * The timestamp at which the action started.
     */
    @SerializedName("st")
    private final long startTime;

    /**
     * The timestamp at which the action ended.
     */
    @SerializedName("en")
    private long endTime;

    /**
     * The timestamp at which the action ended.
     */
    @SerializedName("pz")
    int bytesSent;

    /**
     * This object is for extra properties / data that was not cover
     * with the already defined properties
     */
    @SerializedName("p")
    private Map<String, Object> properties;

    RnActionBreadcrumb(String name, long start, long end, Map<String, Object> properties, int bytesSent, String output) {
        this.name = name;
        this.startTime = start;
        this.endTime = end;
        this.properties = properties != null ? properties : new HashMap<String, Object>();
        this.output = output;
        this.bytesSent = bytesSent;
    }

    @NonNull
    public String getName() {
        return name;
    }

    @NonNull
    public String getOutput() {
        return output;
    }

    public long getEndTime() {
        return endTime;
    }

    @NonNull
    public Map<String, Object> getProperties() {
        return properties;
    }

    @Override
    public long getStartTime() {
        return startTime;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        RnActionBreadcrumb that = (RnActionBreadcrumb) o;

        if (startTime != that.startTime) {
            return false;
        }
        if (endTime != that.endTime) {
            return false;
        }
        if (bytesSent != that.bytesSent) {
            return false;
        }
        if (name != null ? !name.equals(that.name) : that.name != null) {
            return false;
        }
        if (output != null ? !output.equals(that.output) : that.output != null) {
            return false;
        }
        return properties != null ? properties.equals(that.properties) : that.properties == null;
    }

    @Override
    public int hashCode() {
        int result = name != null ? name.hashCode() : 0;
        result = 31 * result + (output != null ? output.hashCode() : 0);
        result = 31 * result + (int) (startTime ^ (startTime >>> 32));
        result = 31 * result + (int) (endTime ^ (endTime >>> 32));
        result = 31 * result + bytesSent;
        result = 31 * result + (properties != null ? properties.hashCode() : 0);
        return result;
    }
}
