package io.embrace.android.embracesdk.anr

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger.Companion.logger
import java.util.concurrent.TimeUnit
import java.util.concurrent.locks.ReentrantLock

private const val LOCK_TIMEOUT_SECS = 1L
private val lock = ReentrantLock()

/**
 * Runs an action after acquiring the 'ANR service lock'. This prevents data races
 * and other synchronization issues.
 *
 * The implementation effectively follows the stdlib implementation of [kotlin.concurrent.withLock],
 * but importantly adds a timeout of 1s in an attempt to avoid deadlock. An internal error
 * is logged if this is exceeded.
 */
internal fun <T> runWithAnrServiceLock(action: () -> T): T? {
    val success = lock.tryLock(LOCK_TIMEOUT_SECS, TimeUnit.SECONDS)
    if (success) {
        try {
            return action()
        } finally {
            lock.unlock()
        }
    } else {
        val msg = "Unexpected ANR lock timeout"
        logger.logError(msg, IllegalStateException(msg), true)
    }
    return null
}
