package io.embrace.android.embracesdk.anr

import io.embrace.android.embracesdk.anr.detection.AnrProcessErrorStateInfo
import io.embrace.android.embracesdk.config.ConfigService
import io.embrace.android.embracesdk.payload.AnrInterval
import java.io.Closeable

/**
 * Service which detects when the application is not responding.
 */
internal interface AnrService : Closeable {

    /**
     * Gets the intervals during which the application was not responding (ANR).
     *
     * @param startTime the time to search from
     * @param endTime   the time to search until
     * @return the list of ANR intervals
     */
    fun getAnrIntervals(startTime: Long, endTime: Long): List<AnrInterval>

    /**
     * Gets all available anr process error states obtained from the OS.
     *
     * @param startTime the time to search from
     * @return the list of ANR process errors
     */
    fun getAnrProcessErrors(startTime: Long): List<AnrProcessErrorStateInfo>

    /**
     * Forces ANR tracking stop by closing the monitoring thread when a crash is
     * handled by the [EmbraceCrashService].
     */
    fun forceAnrTrackingStopOnCrash()

    /**
     * Finishes initialization of the AnrService so that it can react appropriately to
     * lifecycle changes and capture the correct data according to the config. This is necessary
     * as the service can be initialized before the rest of the SDK.
     *
     * @param configService        the configService
     */
    fun finishInitialization(
        configService: ConfigService
    )

    /**
     * Adds a listener which is invoked when the thread becomes blocked/unblocked.
     */
    fun addBlockedThreadListener(listener: BlockedThreadListener)
}
