package io.embrace.android.embracesdk.anr

import androidx.annotation.VisibleForTesting
import io.embrace.android.embracesdk.MemoryCleanerListener
import io.embrace.android.embracesdk.anr.detection.ThreadMonitoringState
import io.embrace.android.embracesdk.clock.Clock
import io.embrace.android.embracesdk.config.ConfigService
import io.embrace.android.embracesdk.payload.AnrInterval
import io.embrace.android.embracesdk.payload.AnrSample
import io.embrace.android.embracesdk.payload.AnrSampleList
import java.util.NavigableMap
import java.util.concurrent.ConcurrentSkipListMap

/**
 * This class is responsible for tracking the state of JVM stacktraces sampled during an ANR.
 */
internal class AnrStacktraceSampler(
    private var configService: ConfigService,
    private val clock: Clock,
    targetThread: Thread
) : BlockedThreadListener, MemoryCleanerListener {

    @VisibleForTesting
    internal var anrIntervals: NavigableMap<Long, AnrInterval> = ConcurrentSkipListMap()
    private val samples = mutableListOf<AnrSample>()
    private var lastUnblockedMs: Long = 0
    private val threadInfoCollector = ThreadInfoCollector(targetThread)
    private val intervalsWithSamples
        get() = anrIntervals.filter {
            it.value.code != AnrSample.CODE_SAMPLE_LIMIT_REACHED
        }

    fun setConfigService(configService: ConfigService) {
        this.configService = configService
    }

    internal fun size() = samples.size

    override fun onThreadBlocked(thread: Thread, timestamp: Long) {
        threadInfoCollector.clearStacktraceCache()
        lastUnblockedMs = timestamp
    }

    override fun onThreadBlockedInterval(thread: Thread, timestamp: Long) {
        val limit = configService.anrBehavior.getMaxStacktracesPerInterval()
        val anrSample = if (size() >= limit) {
            AnrSample(timestamp, null, 0, AnrSample.CODE_SAMPLE_LIMIT_REACHED)
        } else {
            val start = clock.now()
            val threads = threadInfoCollector.captureSample(configService)
            val sampleOverheadMs = clock.now() - start
            AnrSample(timestamp, threads, sampleOverheadMs)
        }
        samples.add(anrSample)
    }

    override fun onThreadUnblocked(thread: Thread, timestamp: Long) {
        // Finalize AnrInterval
        val responseMs = lastUnblockedMs
        val anrInterval = AnrInterval(
            responseMs,
            null,
            timestamp,
            AnrInterval.Type.UI,
            AnrSampleList(samples.toList())
        )
        anrIntervals[timestamp] = anrInterval

        if (reachedAnrCaptureLimit()) {
            findLeastValuableInterval()?.let { entry ->
                anrIntervals[entry.key] = entry.value.clearSamples()
            }
        }

        // reset state
        samples.clear()
        lastUnblockedMs = timestamp
        threadInfoCollector.clearStacktraceCache()
    }

    /**
     * Finds the 'least valuable' ANR interval. This is used when the maximum number of ANR
     * intervals with samples has been reached & the SDK needs to discard samples. We attempt
     * to pick the least valuable interval in this case.
     */
    @VisibleForTesting
    internal fun findLeastValuableInterval() =
        intervalsWithSamples.minByOrNull { it.value.duration() } as Map.Entry<Long, AnrInterval>

    override fun cleanCollections() {
        // create additional collections (avoid any potential for mutating data before
        // it's serialized)
        anrIntervals = ConcurrentSkipListMap()
    }

    @VisibleForTesting
    internal fun reachedAnrCaptureLimit(): Boolean {
        val limit = configService.anrBehavior.getMaxAnrIntervalsPerSession()
        val count = intervalsWithSamples.size - 1 // account for off-by-one
        return count >= limit
    }

    /**
     * Retrieves ANR intervals that match the given start/time windows.
     */
    fun getAnrIntervals(
        startTime: Long,
        endTime: Long,
        state: ThreadMonitoringState,
        clock: Clock
    ): List<AnrInterval> {
        val results: MutableList<AnrInterval> = ArrayList()
        val safeStartTime = startTime + BACKGROUND_ANR_SAFE_INTERVAL_MS
        if (safeStartTime >= endTime) {
            return emptyList()
        }

        // return _all_ the intervals to allow capturing BG ANRs.
        val intervals: Collection<AnrInterval> = anrIntervals.values
        if (!configService.anrBehavior.isBgAnrCaptureEnabled()) {
            // Filter out ANRs that started before session start
            intervals.filterTo(results) { it.startTime >= safeStartTime }
        } else {
            results.addAll(intervals)
        }

        // add any in-progress ANRs
        if (state.anrInProgress) {
            val intervalEndTime = clock.now()
            val responseMs = state.lastTargetThreadResponseMs
            val anrInterval = AnrInterval(
                responseMs,
                intervalEndTime,
                null,
                AnrInterval.Type.UI,
                AnrSampleList(samples.toList())
            )
            results.add(anrInterval)
        }
        return results.map(AnrInterval::deepCopy)
    }

    companion object {

        /**
         * The number of milliseconds which the monitor thread is allowed to timeout before we
         * assume that the process has been put into the cached state.
         */
        private const val BACKGROUND_ANR_SAFE_INTERVAL_MS = 10L
    }
}
