package io.embrace.android.embracesdk.capture.aei

import android.app.ActivityManager
import android.app.ApplicationExitInfo
import android.os.Build
import androidx.annotation.RequiresApi
import androidx.annotation.VisibleForTesting
import io.embrace.android.embracesdk.PreferencesService
import io.embrace.android.embracesdk.config.ConfigListener
import io.embrace.android.embracesdk.config.ConfigService
import io.embrace.android.embracesdk.config.behavior.AppExitInfoBehavior
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger.Companion.logDebug
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger.Companion.logError
import io.embrace.android.embracesdk.payload.AppExitInfoData
import io.embrace.android.embracesdk.worker.BackgroundWorker
import java.io.IOException
import java.nio.charset.Charset
import java.util.concurrent.Callable
import java.util.concurrent.Future

@RequiresApi(Build.VERSION_CODES.R)
internal class EmbraceApplicationExitInfoService constructor(
    private val worker: BackgroundWorker,
    private val configService: ConfigService,
    private val am: ActivityManager?,
    private val preferencesService: PreferencesService
) : ApplicationExitInfoService, ConfigListener {

    @VisibleForTesting
    var appExitInfoList: Future<List<AppExitInfoData>>? = null

    init {
        configService.addListener(this)
        if (configService.isAppExitInfoCaptureEnabled()) {
            appExitInfoList = startService()
        }
    }

    @VisibleForTesting
    fun trackAppExitInfo(): List<AppExitInfoData> {
        logDebug("App exit Info - trackAppExitInfo ")

        // A process ID that used to belong to this package but died later;
        // a value of 0 means to ignore this parameter and return all matching records.
        val pid = 0

        // number of results to be returned; a value of 0 means to ignore this parameter and return all matching records
        val maxNum = 5

        val applicationExitInfo: List<ApplicationExitInfo> =
            am?.getHistoricalProcessExitReasons(null, pid, maxNum) ?: emptyList()

        if (applicationExitInfo.isEmpty()) {
            return emptyList()
        }

        // Generates the set of current aei captured
        val allAeiHashCodes = applicationExitInfo.map(::generateUniqueHash).toSet()

        // Get hash codes that were previously delivered
        val deliveredHashCodes = preferencesService.applicationExitInfoHistory ?: emptySet()

        // Subtracts aei hashcodes of already sent information to get new entries
        val unsentHashCodes = allAeiHashCodes.subtract(deliveredHashCodes)

        // Updates preferences with the new set of hashcodes
        preferencesService.applicationExitInfoHistory = allAeiHashCodes

        // Get AEI objects that were not sent
        val unsentAeiObjects = applicationExitInfo.filter {
            unsentHashCodes.contains(generateUniqueHash(it))
        }

        // map them onto our model class & return them
        return unsentAeiObjects.map(::buildAppExitInfoData)
    }

    @VisibleForTesting
    fun collectExitInfoTraces(appExitInfo: ApplicationExitInfo): AppExitInfoBehavior.CollectTracesResult {
        try {
            appExitInfo.traceInputStream?.let { traces ->
                val bytes = traces.readBytes()
                val size = bytes.size
                return if (size <= configService.appExitInfoBehavior.getTracesMaxLimit()) {
                    AppExitInfoBehavior.CollectTracesResult.Success(
                        String(
                            bytes,
                            Charset.forName("UTF-8")
                        )
                    )
                } else {
                    AppExitInfoBehavior.CollectTracesResult.TooLarge("too_large: size= $size")
                }
            }
        } catch (e: IOException) {
            logError("AEI - IOException: " + e.message, e)
            return AppExitInfoBehavior.CollectTracesResult.TraceException(
                ("ioexception: " + e.message)
            )
        } catch (e: OutOfMemoryError) {
            logError("AEI - Out of Memory: " + e.message, e)
            return AppExitInfoBehavior.CollectTracesResult.TraceException(
                ("oom: " + e.message)
            )
        }

        logError("AEI - Not info traces collected")
        return AppExitInfoBehavior.CollectTracesResult.TracesNull("Not info traces collected")
    }

    @VisibleForTesting
    fun buildAppExitInfoData(appExitInfo: ApplicationExitInfo): AppExitInfoData {
        val tracesResult = collectExitInfoTraces(appExitInfo)

        val traces = when (tracesResult) {
            is AppExitInfoBehavior.CollectTracesResult.Success -> tracesResult.result
            else -> null
        }

        val traceException = when (tracesResult) {
            is AppExitInfoBehavior.CollectTracesResult.Success -> null
            else -> tracesResult.result
        }

        return AppExitInfoData(
            sessionId = String(appExitInfo.processStateSummary ?: ByteArray(0)),
            importance = appExitInfo.importance,
            pss = appExitInfo.pss,
            reason = appExitInfo.reason,
            rss = appExitInfo.rss,
            status = appExitInfo.status,
            timestamp = appExitInfo.timestamp,
            trace = traces,
            description = appExitInfo.description,
            traceStatus = traceException
        )
    }

    private fun generateUniqueHash(appExitInfo: ApplicationExitInfo): String {
        return appExitInfo.timestamp.toString() + "_" + appExitInfo.pid
    }

    override fun cleanCollections() {
    }

    override fun getCapturedData(): List<AppExitInfoData> {
        return try {
            appExitInfoList?.get() ?: ArrayList()
        } catch (e: Exception) {
            logError("Failed to get App Exit info data", e)
            ArrayList()
        }
    }

    override fun onConfigChange(configService: ConfigService) {
        if (appExitInfoList == null && configService.isAppExitInfoCaptureEnabled()) {
            appExitInfoList = startService()
        } else if (!configService.isAppExitInfoCaptureEnabled()) {
            endService()
        }
    }

    private fun endService() {
        try {
            appExitInfoList?.cancel(true)
            appExitInfoList = null
        } catch (e: Exception) {
            logError("Failed to disable EmbraceApplicationExitInfoService work", e)
        }
    }

    private fun startService() = try {
        worker.submit(Callable { trackAppExitInfo() })
    } catch (e: Exception) {
        logError("AEI - trackAppExitInfo submit error", e)
        null
    }
}
