package io.embrace.android.embracesdk.capture.screenshot

import android.app.Activity
import android.graphics.Bitmap
import android.graphics.Canvas
import io.embrace.android.embracesdk.ActivityService
import io.embrace.android.embracesdk.Embrace
import io.embrace.android.embracesdk.clock.Clock
import io.embrace.android.embracesdk.comms.delivery.DeliveryService
import io.embrace.android.embracesdk.config.ConfigService
import io.embrace.android.embracesdk.logging.InternalEmbraceLogger
import java.io.ByteArrayOutputStream

/**
 * Takes screenshots and submits them to the Embrace API.
 */
internal class EmbraceScreenshotService(
    private val activityService: ActivityService,
    private val configService: ConfigService,
    private val deliveryService: DeliveryService,
    private val logger: InternalEmbraceLogger,
    private val clock: Clock
) : ScreenshotService {
    @Volatile
    private var lastScreenshot: Long = 0

    override fun takeScreenshotLogEvent(logId: String): Boolean = takeScreenshot(logId, null)

    override fun takeScreenshotMoment(eventId: String): Boolean = takeScreenshot(null, eventId)

    private fun takeScreenshot(logId: String?, eventId: String?): Boolean {
        synchronized(this) {
            val timestamp = clock.now()
            if (activityService.isInBackground ||
                !configService.autoDataCaptureBehavior.isScreenshotCaptureEnabled() || timestamp - lastScreenshot < SCREENSHOT_COOLDOWN
            ) {
                logger.logDebug("Screenshots are disabled, app is backgrounded, or cooling down")
                return false
            }
            val optionalActivity = activityService.foregroundActivity
            return if (optionalActivity != null) {
                val screenshot = screenshot(optionalActivity)
                lastScreenshot = timestamp
                if (screenshot == null) {
                    logger.logDebug("Could not take screenshot")
                    return false
                }
                if (eventId != null) {
                    logger.logDeveloper(
                        "EmbraceScreenshotService",
                        "Attempting to send moment screenshot"
                    )
                    deliveryService.sendMomentScreenshot(screenshot, eventId)
                    logger.logDeveloper(
                        "EmbraceScreenshotService",
                        "Send moment screenshot running on background"
                    )
                } else if (logId != null) {
                    logger.logDeveloper(
                        "EmbraceScreenshotService",
                        "Attempting to send log screenshot"
                    )
                    deliveryService.sendLogScreenshot(screenshot, logId)
                    logger.logDeveloper(
                        "EmbraceScreenshotService",
                        "Send Log screenshot running on background"
                    )
                } else {
                    logger.logDeveloper(
                        "EmbraceScreenshotService",
                        "Foreground activity not present"
                    )
                    return false
                }
                true
            } else {
                logger.logDebug("Screenshot cannot be taken as there is no active activity")
                false
            }
        }
    }

    private fun takeScreenshot(activity: Activity?): Bitmap? {
        if (activity != null) {
            val view = activity.window.decorView
            return if (view != null) {
                if (view.width > 0 && view.height > 0) {
                    try {
                        val screenshot =
                            Bitmap.createBitmap(view.width, view.height, Bitmap.Config.ARGB_8888)
                        logger.logDeveloper("EmbraceScreenshotService", "Bitmap created")
                        view.draw(Canvas(screenshot))
                        screenshot
                    } catch (e: OutOfMemoryError) {
                        Embrace.getInstance().logInternalError(
                            "Empty screenshot 01 - Out of memory drawing screenshot",
                            "Run out of memory while creating the bitmap and drawing the screenshot"
                        )
                        null
                    }
                } else {
                    Embrace.getInstance().logInternalError(
                        "Empty screenshot 02 - decor view is 0 or lower",
                        "Height: " + view.height + ". Width: " + view.width
                    )
                    null
                }
            } else {
                Embrace.getInstance().logInternalError(
                    "Empty screenshot 03 - decor view is null",
                    "activity.getWindow().getDecorView() returned null"
                )
                null
            }
        }
        Embrace.getInstance()
            .logInternalError("Empty screenshot 04 - activity is null", "Null activity")
        return null
    }

    private fun screenshot(activity: Activity): ByteArray? {
        val screenshot = takeScreenshot(activity)
        if (screenshot == null) {
            Embrace.getInstance()
                .logInternalError("Empty screenshot 05 - null bitmap", "Created bitmap was null")
            return null
        }
        logger.logDebug("Compressing screenshot")
        val stream = ByteArrayOutputStream()
        try {
            screenshot.compress(
                Bitmap.CompressFormat.JPEG,
                SCREENSHOT_JPEG_COMPRESSION_VALUE,
                stream
            )
            logger.logDeveloper("EmbraceScreenshotService", "Screenshot compressed")
        } catch (e: OutOfMemoryError) {
            logger.logDebug("Failed to compress screenshot due insufficient memory.", e)
            Embrace.getInstance().logInternalError(
                "Empty screenshot 06 - compress OOM",
                "Failed to compress screenshot. Cause: OOM"
            )
            return null
        }
        screenshot.recycle()
        logger.logDeveloper("EmbraceScreenshotService", "Screenshot recycled")
        return stream.toByteArray()
    }

    companion object {
        private const val SCREENSHOT_COOLDOWN: Long = 3000
        private const val SCREENSHOT_JPEG_COMPRESSION_VALUE = 70
    }
}
