package io.embrace.android.embracesdk.comms

import android.os.Build
import android.os.Debug
import io.embrace.android.embracesdk.MetadataService
import io.embrace.android.embracesdk.config.ConfigService
import io.embrace.android.embracesdk.config.behavior.SdkEndpointBehavior
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger

internal class ApiUrlBuilder(
    private val provider: () -> ConfigService,
    private val metadataService: MetadataService,
    private val enableIntegrationTesting: Boolean
) {

    companion object {
        private const val API_VERSION = 1
        private const val CONFIG_API_VERSION = 2
    }

    private val baseUrls: SdkEndpointBehavior
        get() = provider().sdkEndpointBehavior

    private fun getScreenshotUrl() = buildUrl(baseUrls.getImages(), API_VERSION, "screenshot")
    private fun getConfigBaseUrl() = buildUrl(baseUrls.getConfig(), CONFIG_API_VERSION, "config")
    private fun getOperatingSystemCode() = Build.VERSION.SDK_INT.toString() + ".0.0"

    private fun getCoreBaseUrl(): String = if (isDebugBuild()) {
        baseUrls.getDataDev()
    } else {
        baseUrls.getData()
    }

    private fun getAppVersion(): String = metadataService.getAppVersionName()

    private fun getAppId() = metadataService.getAppId()

    private fun isDebugBuild(): Boolean {
        return metadataService.isDebug() && enableIntegrationTesting &&
            (Debug.isDebuggerConnected() || Debug.waitingForDebugger())
    }

    private fun buildUrl(config: String, configApiVersion: Int, path: String): String {
        return "$config/v$configApiVersion/$path"
    }

    fun getConfigUrl(): String {
        return "${getConfigBaseUrl()}?appId=${getAppId()}&osVersion=${getOperatingSystemCode()}" +
            "&appVersion=${getAppVersion()}&deviceId=${metadataService.getDeviceId()}"
    }

    fun getLogScreenshotUrl(logId: String): String {
        InternalStaticEmbraceLogger.logDeveloper(
            "ApiUrlBuilder",
            "getLogScreenshotUrl - logId: $logId"
        )
        return "${getScreenshotUrl()}/${getAppId()}/logs/$logId.jpg"
    }

    fun getMomentScreenshotUrl(eventId: String): String {
        InternalStaticEmbraceLogger.logDeveloper(
            "ApiUrlBuilder",
            "getMomentScreenshotUrl - eventId: $eventId"
        )
        return "${getScreenshotUrl()}/${getAppId()}/moments/$eventId.jpg"
    }

    fun getEmbraceUrlWithSuffix(suffix: String): String {
        InternalStaticEmbraceLogger.logDeveloper(
            "ApiUrlBuilder",
            "getEmbraceUrlWithSuffix - suffix: $suffix"
        )
        return "${getCoreBaseUrl()}/v$API_VERSION/log/$suffix"
    }
}
