package io.embrace.android.embracesdk.injection

import android.os.Looper
import io.embrace.android.embracesdk.SharedObjectLoader
import io.embrace.android.embracesdk.anr.AnrService
import io.embrace.android.embracesdk.anr.EmbraceAnrService
import io.embrace.android.embracesdk.anr.NoOpAnrService
import io.embrace.android.embracesdk.anr.detection.AnrProcessErrorSampler
import io.embrace.android.embracesdk.anr.detection.BlockedThreadDetector
import io.embrace.android.embracesdk.anr.detection.LivenessCheckScheduler
import io.embrace.android.embracesdk.anr.detection.TargetThreadHandler
import io.embrace.android.embracesdk.anr.detection.ThreadMonitoringState
import io.embrace.android.embracesdk.anr.sigquit.FilesDelegate
import io.embrace.android.embracesdk.anr.sigquit.FindGoogleThread
import io.embrace.android.embracesdk.anr.sigquit.GetThreadCommand
import io.embrace.android.embracesdk.anr.sigquit.GetThreadsInCurrentProcess
import io.embrace.android.embracesdk.anr.sigquit.GoogleAnrHandlerNativeDelegate
import io.embrace.android.embracesdk.anr.sigquit.GoogleAnrTimestampRepository
import io.embrace.android.embracesdk.anr.sigquit.SigquitDetectionService
import io.embrace.android.embracesdk.internal.ApkToolsConfig
import java.util.concurrent.Executors
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.ThreadFactory

internal interface AnrModule {
    val googleAnrTimestampRepository: GoogleAnrTimestampRepository
    val anrService: AnrService
}

internal class AnrModuleImpl(
    coreModule: CoreModule,
    systemServiceModule: SystemServiceModule,
    essentialServiceModule: EssentialServiceModule
) : AnrModule {

    private val configService = essentialServiceModule.configService

    override val googleAnrTimestampRepository: GoogleAnrTimestampRepository by singleton {
        GoogleAnrTimestampRepository(coreModule.logger)
    }

    override val anrService: AnrService by singleton {
        if (configService.autoDataCaptureBehavior.isAnrServiceEnabled() && !ApkToolsConfig.IS_ANR_MONITORING_DISABLED) {
            // the customer didn't enable early ANR detection, so construct the service
            // as part of normal initialization.
            EmbraceAnrService(
                configService,
                looper,
                coreModule.logger,
                sigquitDetectionService,
                livenessCheckScheduler,
                anrExecutorService,
                state,
                anrProcessErrorSampler,
                coreModule.clock
            )
        } else {
            NoOpAnrService()
        }
    }

    private val looper by singleton { Looper.getMainLooper() }

    private val state by singleton { ThreadMonitoringState(coreModule.clock) }

    private val targetThreadHandler by singleton {
        TargetThreadHandler(
            looper,
            state,
            configService,
            clock = coreModule.clock
        )
    }

    private val blockedThreadDetector by singleton {
        BlockedThreadDetector(
            configService,
            coreModule.clock,
            null,
            state,
            looper.thread
        )
    }

    private val livenessCheckScheduler by singleton {
        LivenessCheckScheduler(
            configService,
            anrExecutorService,
            coreModule.clock,
            state,
            targetThreadHandler,
            blockedThreadDetector
        )
    }

    private val anrProcessErrorSampler by singleton {
        AnrProcessErrorSampler(
            systemServiceModule.activityManager,
            configService,
            anrExecutorService,
            coreModule.clock,
            coreModule.logger
        )
    }

    private val sigquitDetectionService: SigquitDetectionService by singleton {
        val filesDelegate = FilesDelegate()

        SigquitDetectionService(
            SharedObjectLoader(),
            FindGoogleThread(
                coreModule.logger,
                GetThreadsInCurrentProcess(filesDelegate),
                GetThreadCommand(filesDelegate)
            ),
            GoogleAnrHandlerNativeDelegate(googleAnrTimestampRepository, coreModule.logger),
            googleAnrTimestampRepository,
            configService,
            coreModule.logger
        )
    }
}

private val anrMonitorThreadFactory = ThreadFactory { runnable: Runnable ->
    Executors.defaultThreadFactory().newThread(runnable).apply {
        anrMonitorThread = this
        name = "Embrace ANR Healthcheck"
    }
}

internal val anrExecutorService: ScheduledExecutorService by lazy {
    // must only have one thread in executor pool - synchronization model relies on this fact.
    Executors.newSingleThreadScheduledExecutor(
        anrMonitorThreadFactory
    )
}

internal lateinit var anrMonitorThread: Thread
