package io.embrace.android.embracesdk.injection

import android.os.Build
import io.embrace.android.embracesdk.BreadcrumbService
import io.embrace.android.embracesdk.EmbraceBreadcrumbService
import io.embrace.android.embracesdk.PushNotificationCaptureService
import io.embrace.android.embracesdk.capture.aei.ApplicationExitInfoService
import io.embrace.android.embracesdk.capture.aei.EmbraceApplicationExitInfoService
import io.embrace.android.embracesdk.capture.aei.NoOpApplicationExitInfoService
import io.embrace.android.embracesdk.capture.connectivity.EmbraceNetworkConnectivityService
import io.embrace.android.embracesdk.capture.connectivity.NetworkConnectivityService
import io.embrace.android.embracesdk.capture.connectivity.NoOpNetworkConnectivityService
import io.embrace.android.embracesdk.capture.crumbs.activity.ActivityLifecycleBreadcrumbService
import io.embrace.android.embracesdk.capture.crumbs.activity.EmbraceActivityLifecycleBreadcrumbService
import io.embrace.android.embracesdk.capture.memory.EmbraceMemoryService
import io.embrace.android.embracesdk.capture.memory.MemoryService
import io.embrace.android.embracesdk.capture.memory.NoOpMemoryService
import io.embrace.android.embracesdk.capture.powersave.EmbracePowerSaveModeService
import io.embrace.android.embracesdk.capture.powersave.NoOpPowerSaveModeService
import io.embrace.android.embracesdk.capture.powersave.PowerSaveModeService
import io.embrace.android.embracesdk.capture.strictmode.EmbraceStrictModeService
import io.embrace.android.embracesdk.capture.strictmode.NoOpStrictModeService
import io.embrace.android.embracesdk.capture.strictmode.StrictModeService
import io.embrace.android.embracesdk.capture.thermalstate.EmbraceThermalStatusService
import io.embrace.android.embracesdk.capture.thermalstate.NoOpThermalStatusService
import io.embrace.android.embracesdk.capture.thermalstate.ThermalStatusService
import io.embrace.android.embracesdk.capture.webview.EmbraceWebViewService
import io.embrace.android.embracesdk.capture.webview.WebViewService
import io.embrace.android.embracesdk.utils.BuildVersionChecker
import io.embrace.android.embracesdk.utils.VersionChecker
import io.embrace.android.embracesdk.worker.WorkerName
import io.embrace.android.embracesdk.worker.WorkerThreadModule

/**
 * This modules provides services that capture data from within an application. It could be argued
 * that a lot of classes could fit in this module, so to keep it small (<15 properties) it's best
 * to only include services whose main responsibility is just capturing data. It would be well
 * worth reassessing the grouping once this module grows larger.
 */
internal interface DataCaptureServiceModule {

    /**
     * Captures breadcrumbs
     */
    val breadcrumbService: BreadcrumbService

    /**
     * Captures memory events
     */
    val memoryService: MemoryService

    /**
     * Captures intervals where power save mode was enabled
     */
    val powerSaveModeService: PowerSaveModeService

    /**
     * Captures intervals where the network was/wasn't connected
     */
    val networkConnectivityService: NetworkConnectivityService

    /**
     * Captures information from webviews
     */
    val webviewService: WebViewService

    /**
     * Captures push notifications
     */
    val pushNotificationService: PushNotificationCaptureService

    /**
     * Captures strict mode violations
     */
    val strictModeService: StrictModeService

    /**
     * Captures thermal state events
     */
    val thermalStatusService: ThermalStatusService

    /**
     * Captures breadcrumbs of the activity lifecycle
     */
    val activityLifecycleBreadcrumbService: ActivityLifecycleBreadcrumbService?

    /**
     * Captures app exit info
     */
    val appExitInfoService: ApplicationExitInfoService
}

internal class DataCaptureServiceModuleImpl @JvmOverloads constructor(
    coreModule: CoreModule,
    systemServiceModule: SystemServiceModule,
    essentialServiceModule: EssentialServiceModule,
    workerThreadModule: WorkerThreadModule,
    versionChecker: VersionChecker = BuildVersionChecker
) : DataCaptureServiceModule {

    private val backgroundWorker =
        workerThreadModule.backgroundWorker(WorkerName.BACKGROUND_REGISTRATION)
    private val scheduledExecutor =
        workerThreadModule.scheduledExecutor(WorkerName.SCHEDULED_REGISTRATION)
    private val configService = essentialServiceModule.configService

    override val memoryService: MemoryService by singleton {
        if (configService.autoDataCaptureBehavior.isMemoryServiceEnabled()) {
            EmbraceMemoryService(coreModule.clock)
        } else {
            NoOpMemoryService()
        }
    }

    override val powerSaveModeService: PowerSaveModeService by singleton {
        if (configService.autoDataCaptureBehavior.isPowerSaveModeServiceEnabled() && versionChecker.isAtLeast(
                Build.VERSION_CODES.LOLLIPOP
            )
        ) {
            EmbracePowerSaveModeService(
                coreModule.context,
                backgroundWorker,
                coreModule.clock,
                systemServiceModule.powerManager
            )
        } else {
            NoOpPowerSaveModeService()
        }
    }

    override val networkConnectivityService: NetworkConnectivityService by singleton {
        if (configService.autoDataCaptureBehavior.isNetworkConnectivityServiceEnabled()) {
            EmbraceNetworkConnectivityService(
                coreModule.context,
                coreModule.clock,
                backgroundWorker,
                coreModule.logger,
                systemServiceModule.connectivityManager
            )
        } else {
            NoOpNetworkConnectivityService()
        }
    }

    override val webviewService: WebViewService by singleton {
        EmbraceWebViewService(configService, coreModule.jsonSerializer)
    }

    override val breadcrumbService: BreadcrumbService by singleton {
        EmbraceBreadcrumbService(
            coreModule.clock,
            configService,
            coreModule.logger
        )
    }

    override val pushNotificationService: PushNotificationCaptureService by singleton {
        PushNotificationCaptureService(
            breadcrumbService, coreModule.logger
        )
    }

    override val strictModeService: StrictModeService by singleton {
        if (versionChecker.isAtLeast(Build.VERSION_CODES.P) && configService.anrBehavior.isStrictModeListenerEnabled()) {
            EmbraceStrictModeService(configService, scheduledExecutor, coreModule.clock)
        } else {
            NoOpStrictModeService()
        }
    }

    override val thermalStatusService: ThermalStatusService by singleton {
        if (configService.sdkModeBehavior.isBetaFeaturesEnabled() && versionChecker.isAtLeast(Build.VERSION_CODES.Q)) {
            EmbraceThermalStatusService(
                scheduledExecutor,
                coreModule.clock,
                coreModule.logger,
                systemServiceModule.powerManager
            )
        } else {
            NoOpThermalStatusService()
        }
    }

    override val activityLifecycleBreadcrumbService: EmbraceActivityLifecycleBreadcrumbService? by singleton {
        if (configService.sdkModeBehavior.isBetaFeaturesEnabled() && versionChecker.isAtLeast(Build.VERSION_CODES.Q)) {
            EmbraceActivityLifecycleBreadcrumbService(configService, coreModule.clock)
        } else {
            null
        }
    }

    override val appExitInfoService: ApplicationExitInfoService by singleton {
        if (versionChecker.isAtLeast(Build.VERSION_CODES.R)) {
            EmbraceApplicationExitInfoService(
                backgroundWorker,
                configService,
                systemServiceModule.activityManager,
                essentialServiceModule.preferencesService
            )
        } else {
            NoOpApplicationExitInfoService()
        }
    }
}
