package io.embrace.android.embracesdk.injection

import android.preference.PreferenceManager
import io.embrace.android.embracesdk.ActivityService
import io.embrace.android.embracesdk.BuildInfo
import io.embrace.android.embracesdk.EmbraceActivityService
import io.embrace.android.embracesdk.EmbraceCpuInfoDelegate
import io.embrace.android.embracesdk.EmbraceMemoryCleanerService
import io.embrace.android.embracesdk.EmbraceMetadataService
import io.embrace.android.embracesdk.EmbracePreferencesService
import io.embrace.android.embracesdk.MemoryCleanerService
import io.embrace.android.embracesdk.MetadataService
import io.embrace.android.embracesdk.PreferencesService
import io.embrace.android.embracesdk.SharedObjectLoader
import io.embrace.android.embracesdk.capture.orientation.NoOpOrientationService
import io.embrace.android.embracesdk.capture.orientation.OrientationService
import io.embrace.android.embracesdk.capture.user.EmbraceUserService
import io.embrace.android.embracesdk.capture.user.UserService
import io.embrace.android.embracesdk.comms.ApiClient
import io.embrace.android.embracesdk.comms.ApiRequest
import io.embrace.android.embracesdk.comms.ApiResponseCache
import io.embrace.android.embracesdk.comms.ApiUrlBuilder
import io.embrace.android.embracesdk.config.ConfigService
import io.embrace.android.embracesdk.config.EmbraceConfigService
import io.embrace.android.embracesdk.config.local.LocalConfig
import io.embrace.android.embracesdk.gating.EmbraceGatingService
import io.embrace.android.embracesdk.gating.GatingService
import io.embrace.android.embracesdk.worker.WorkerName
import io.embrace.android.embracesdk.worker.WorkerThreadModule
import java.io.File

/**
 * This module contains services that are essential for bootstrapping other functionality in
 * the SDK during initialization.
 */
internal interface EssentialServiceModule {
    val memoryCleanerService: MemoryCleanerService
    val orientationService: OrientationService
    val activityService: ActivityService
    val preferencesService: PreferencesService
    val metadataService: MetadataService
    val configService: ConfigService
    val gatingService: GatingService
    val userService: UserService
    val urlBuilder: ApiUrlBuilder
    val cache: ApiResponseCache
    val apiClient: ApiClient
    val sharedObjectLoader: SharedObjectLoader
    val embraceCpuInfoDelegate: EmbraceCpuInfoDelegate
}

internal class EssentialServiceModuleImpl(
    coreModule: CoreModule,
    systemServiceModule: SystemServiceModule,
    workerThreadModule: WorkerThreadModule,
    buildInfo: BuildInfo,
    customAppId: String?,
    enableIntegrationTesting: Boolean,
    private val configStopAction: () -> Unit
) : EssentialServiceModule {

    private val backgroundWorker =
        workerThreadModule.backgroundWorker(WorkerName.BACKGROUND_REGISTRATION)
    private val backgroundExecutorService =
        workerThreadModule.backgroundExecutor(WorkerName.BACKGROUND_REGISTRATION)

    // bootstrap initialization. ConfigService not created yet...
    private val configProvider: () -> ConfigService = { configService }

    override val memoryCleanerService: MemoryCleanerService by singleton {
        EmbraceMemoryCleanerService()
    }

    override val orientationService: OrientationService by singleton {
        // Embrace is not processing orientation changes on this moment, so return no-op service.
        NoOpOrientationService()
    }

    override val activityService: ActivityService by singleton {
        EmbraceActivityService(coreModule.application, orientationService, coreModule.clock)
    }

    override val preferencesService: PreferencesService by singleton {
        val lazyPrefs = lazy {
            PreferenceManager.getDefaultSharedPreferences(
                coreModule.context
            )
        }
        EmbracePreferencesService(
            backgroundWorker,
            lazyPrefs,
            coreModule.clock,
            coreModule.jsonSerializer
        )
    }

    override val sharedObjectLoader: SharedObjectLoader by singleton {
        SharedObjectLoader()
    }

    override val embraceCpuInfoDelegate: EmbraceCpuInfoDelegate by singleton {
        EmbraceCpuInfoDelegate(sharedObjectLoader, coreModule.logger)
    }

    override val metadataService: MetadataService by singleton {
        EmbraceMetadataService.ofContext(
            coreModule.context,
            buildInfo,
            configProvider,
            coreModule.appFramework,
            preferencesService,
            activityService,
            backgroundWorker,
            systemServiceModule.storageManager,
            systemServiceModule.windowManager,
            systemServiceModule.activityManager,
            coreModule.clock,
            embraceCpuInfoDelegate
        )
    }

    override val urlBuilder by singleton {
        ApiUrlBuilder(
            configProvider,
            metadataService,
            enableIntegrationTesting
        )
    }

    override val cache by singleton {
        ApiResponseCache(
            coreModule.jsonSerializer,
            { File(coreModule.context.cacheDir, "emb_config_cache") }
        )
    }

    override val apiClient by singleton {
        ApiClient(
            urlBuilder,
            coreModule.jsonSerializer,
            { url: String, request: ApiRequest -> cache.retrieveCachedConfig(url, request) },
            coreModule.logger
        )
    }

    override val configService: ConfigService by singleton {
        EmbraceConfigService(
            LocalConfig.fromResources(coreModule.resources, coreModule.context.packageName, customAppId, coreModule.jsonSerializer),
            apiClient,
            metadataService,
            preferencesService,
            coreModule.clock,
            coreModule.logger,
            backgroundExecutorService,
            configStopAction
        )
    }

    override val gatingService: GatingService by singleton {
        EmbraceGatingService(configService)
    }

    override val userService: UserService by singleton {
        EmbraceUserService(
            preferencesService,
            coreModule.logger
        )
    }
}
