package io.embrace.android.embracesdk.internal.spans

import io.embrace.android.embracesdk.InternalApi
import io.embrace.android.embracesdk.internal.spans.EmbraceAttributes.Attribute
import io.opentelemetry.api.trace.Span
import io.opentelemetry.api.trace.SpanBuilder
import io.opentelemetry.api.trace.StatusCode
import io.opentelemetry.api.trace.Tracer
import java.util.concurrent.TimeUnit

/**
 * Extension functions and constants to augment the core OpenTelemetry SDK and provide Embrace-specific customizations
 *
 * Note: there's no explicit tests for these extensions as their functionality will be validated as part of other tests.
 */

/**
 * Prefix added to [Span] names for all Spans recorded internally by the SDK
 */
private const val EMBRACE_SPAN_NAME_PREFIX = "emb-"

/**
 * Prefix added to all [Span] attribute keys for all attributes added by the SDK
 */
private const val EMBRACE_ATTRIBUTE_NAME_PREFIX = "emb."

/**
 * Attribute name for the monotonically increasing sequence ID given to completed [Span] that expected to sent to the server
 */
private const val SEQUENCE_ID_ATTRIBUTE_NAME = EMBRACE_ATTRIBUTE_NAME_PREFIX + "sequence_id"
private const val KEY_SPAN_ATTRIBUTE_NAME = EMBRACE_ATTRIBUTE_NAME_PREFIX + "key"

/**
 * Creates a new [SpanBuilder] with the correctly prefixed name, to be used for recording Spans in the SDK internally
 */
@InternalApi
internal fun Tracer.embraceSpanBuilder(name: String): SpanBuilder = spanBuilder(EMBRACE_SPAN_NAME_PREFIX + name)

/**
 * Sets and returns the [EmbraceAttributes.Type] attribute for the given [SpanBuilder]
 */
@InternalApi
internal fun SpanBuilder.setType(value: EmbraceAttributes.Type): SpanBuilder {
    setAttribute(value.keyName(), value.toString())
    return this
}

/**
 * Mark this [Span] as "key" so the backend will create aggregate metrics for it, and the UI will show it as a "top level" span
 */
@InternalApi
internal fun SpanBuilder.makeKey(): SpanBuilder {
    setAttribute(KEY_SPAN_ATTRIBUTE_NAME, true)
    return this
}

/**
 * Allow a [SpanBuilder] to take in a lambda around which a span will be created for its execution
 */
internal fun <T> SpanBuilder.record(code: () -> T): T {
    val returnValue: T
    var span: Span? = null

    try {
        span = startSpan()
        returnValue = code()
        span.endSpan()
    } catch (t: Throwable) {
        span?.endSpan(ErrorCode.FAILURE)
        throw t
    }

    return returnValue
}

/**
 * Monotonically increasing ID given to completed [Span] that expected to sent to the server. Can be used to track data loss on the server.
 */
@InternalApi
internal fun Span.setSequenceId(id: Long): Span {
    setAttribute(SEQUENCE_ID_ATTRIBUTE_NAME, id)
    return this
}

/**
 * Ends the given [Span], and setting the correct properties per the optional [EmbraceAttributes.ErrorCode] passed in. If [errorCode]
 * is not specified, it means the [Span] completed successfully, and no [EmbraceAttributes.ErrorCode] will be set.
 */
@InternalApi
internal fun Span.endSpan(errorCode: ErrorCode? = null, endTimeNanos: Long? = null): Span {
    if (errorCode == null) {
        setStatus(StatusCode.OK)
    } else {
        setStatus(StatusCode.ERROR)
        setAttribute(errorCode.keyName(), errorCode.toString())
    }

    if (endTimeNanos != null) {
        end(endTimeNanos, TimeUnit.NANOSECONDS)
    } else {
        end()
    }

    return this
}

/**
 * Contains the set of attributes (i.e. implementers of the [Attribute] interface) set on a [Span] by the SDK that has special meaning
 * in the Embrace world. Each enum defines the attribute name used in the [Span] and specifies the set of valid values it can be set to.
 */
@InternalApi
internal object EmbraceAttributes {

    /**
     * Attribute to categorize a [Span] and give it a distinct semantic meaning. Spans of each [Type] may be treated differently by the
     * backend and can be expected to contain a set of attributes to further flesh out the given semantic meanings.
     *
     * The [internal] attribute (defaults to false) denotes a [Type] that is only used from within the SDK code, basically a span that is
     * internal to the SDK and should not be logged .
     *
     * TODO: Formalize in code this mapping of span types to the potential set of custom attributes.
     */
    internal enum class Type(val internal: Boolean = false) : Attribute {

        /**
         * The root span that models the lifetime of the app process and serves as the root for all the [Type.SESSION] spans.
         */
        PROCESS(internal = true),

        /**
         * The children of the [Type.PROCESS] spans that model a session or background activity. It serves as the parent of all Key Spans
         * that are recorded.
         */
        SESSION(internal = true),

        /**
         * A [Span] that tracks the time it takes for the SDK to initialize
         */
        SDK_STARTUP(internal = true),

        /**
         * A [Span] that tracks an internal operation of the SDK
         */
        INTERNAL(internal = true),

        /**
         * A [Span] created by an SDK user to measure the performance of an operation
         */
        PERFORMANCE;

        override val canonicalName = "type"
    }

    /**
     * The reason for the termination of a process span
     */
    internal enum class AppTerminationCause : Attribute {
        CRASH,
        USER_TERMINATION,
        UNKNOWN;

        override val canonicalName: String = "termination_cause"
    }

    /**
     * Denotes an attribute added by the SDK with a restricted set of valid values
     */
    internal interface Attribute {

        /**
         * The name used to identify this [Attribute]
         */
        val canonicalName: String

        /**
         * The name used as the key for the [Attribute] in the attributes map
         */
        fun keyName(): String = EMBRACE_ATTRIBUTE_NAME_PREFIX + canonicalName
    }
}
