package io.embrace.android.embracesdk.internal.spans

import com.google.gson.annotations.SerializedName
import io.embrace.android.embracesdk.InternalApi
import io.opentelemetry.api.trace.Span
import io.opentelemetry.api.trace.StatusCode
import io.opentelemetry.sdk.trace.data.EventData
import io.opentelemetry.sdk.trace.data.SpanData

/**
 * Serializable representation of [Span]
 */
@InternalApi
internal data class EmbraceSpanData(
    @SerializedName("trace_id")
    val traceId: String,

    @SerializedName("span_id")
    val spanId: String,

    @SerializedName("parent_span_id")
    val parentSpanId: String?,

    @SerializedName("name")
    val name: String,

    @SerializedName("start_time_unix_nano")
    val startTimeNanos: Long,

    @SerializedName("end_time_unix_nano")
    val endTimeNanos: Long,

    @SerializedName("status")
    val status: StatusCode = StatusCode.UNSET,

    @SerializedName("events")
    val events: List<EmbraceSpanEvent> = emptyList(),

    @SerializedName("attributes")
    val attributes: Map<String, String> = emptyMap()
) {
    constructor(spanData: SpanData) : this(
        traceId = spanData.spanContext.traceId,
        spanId = spanData.spanContext.spanId,
        parentSpanId = spanData.parentSpanId,
        name = spanData.name,
        startTimeNanos = spanData.startEpochNanos,
        endTimeNanos = spanData.endEpochNanos,
        status = spanData.status.statusCode,
        events = spanData.events.map { EmbraceSpanEvent(eventData = it) },
        attributes = spanData.attributes.asMap().entries.associate { it.key.key to it.value.toString() }
    )
}

/**
 * Serializable representation of a [EventData]
 */
@InternalApi
internal data class EmbraceSpanEvent(
    @SerializedName("name")
    val name: String,

    @SerializedName("time_unix_nano")
    val timestamp: Long,

    @SerializedName("attributes")
    val attributes: Map<String, String> = emptyMap()
) {
    constructor(eventData: EventData) : this(
        name = eventData.name,
        timestamp = eventData.epochNanos,
        attributes = eventData.attributes.asMap().entries.associate { it.key.key to it.value.toString() }
    )
}
