package io.embrace.android.embracesdk.internal.spans

import io.embrace.android.embracesdk.InternalApi
import io.embrace.android.embracesdk.clock.Clock
import io.embrace.android.embracesdk.internal.Systrace
import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger
import io.opentelemetry.api.OpenTelemetry
import io.opentelemetry.api.common.Attributes
import io.opentelemetry.api.trace.Span
import io.opentelemetry.api.trace.SpanBuilder
import io.opentelemetry.api.trace.Tracer
import io.opentelemetry.context.Context
import io.opentelemetry.sdk.OpenTelemetrySdk
import io.opentelemetry.sdk.common.CompletableResultCode
import io.opentelemetry.sdk.trace.SdkTracerProvider
import io.opentelemetry.sdk.trace.data.EventData
import io.opentelemetry.sdk.trace.data.SpanData
import java.util.concurrent.TimeUnit

/**
 * Implementation of the core logic for [SpansService]
 */
@InternalApi
internal class SpansServiceImpl(
    sdkInitStartTimeNanos: Long,
    sdkInitEndTimeNanos: Long,
    private val clock: Clock
) : SpansService {
    private val sdkTracerProvider: SdkTracerProvider
        by lazy {
            Systrace.start("spans-service-init")
            Systrace.trace("init-sdk-tracer-provider") {
                SdkTracerProvider
                    .builder()
                    .addSpanProcessor(EmbraceSpanProcessor(EmbraceSpanExporter(this)))
                    .build()
            }
        }

    private val openTelemetry: OpenTelemetry
        by lazy {
            Systrace.trace("init-otel-sdk") {
                OpenTelemetrySdk.builder()
                    .setTracerProvider(sdkTracerProvider)
                    .build()
            }
        }

    private val tracer: Tracer
        by lazy {
            Systrace.trace("init-tracer") {
                openTelemetry.getTracer("embrace-sdk", "0.0.1")
            }
        }

    /**
     * The root span of the trace that models the lifetime of the process
     */
    private val processRootSpan: Span
        by lazy {
            // Note that the start time used here isn't really the start time for the process. Right now, this span isn't being exposed
            // to the public, so the start time really doesn't matter. If we change that, we should think more deeply about what this time
            // should be and implemented it properly that way.
            Systrace.trace("first-span-start") {
                createEmbraceSpanBuilder(name = "process-root-span", type = EmbraceAttributes.Type.PROCESS)
                    .setNoParent()
                    .setStartTimestamp(sdkInitStartTimeNanos, TimeUnit.NANOSECONDS)
                    .startSpan()
            }
        }

    /**
     * The currently open child of [processRootSpan] that models the lifetime of the current session or background activity
     */
    private var currentSessionSpan: Span = createSessionSpan(sdkInitStartTimeNanos)

    /**
     * Spans that have finished, successfully or not, that will be sent with the next session or background activity payload. These
     * should be cached along with the other data in the payload.
     */
    private val completedSpans: MutableList<EmbraceSpanData> = mutableListOf()

    init {
        Systrace.trace("log-sdk-init") {
            recordCompletedSpan(
                name = "sdk-init",
                startTimeNanos = sdkInitStartTimeNanos,
                endTimeNanos = sdkInitEndTimeNanos,
                type = EmbraceAttributes.Type.SDK_STARTUP,
                events = listOf<EventData>(
                    EventData.create(sdkInitStartTimeNanos, "start-time", Attributes.empty())
                )
            )
        }
        Systrace.end()
    }

    override fun createSpan(name: String, type: EmbraceAttributes.Type): EmbraceSpan? {
        return if (currentSessionSpan.isRecording) {
            EmbraceSpanImpl(createKeySpan(name = name, type = type))
        } else {
            null
        }
    }

    override fun <T> recordSpan(
        name: String,
        type: EmbraceAttributes.Type,
        code: () -> T
    ): T {
        return if (currentSessionSpan.isRecording) {
            InternalStaticEmbraceLogger.logDebug("Logging span '$name'")
            Systrace.start("log-span-$name")
            try {
                createKeySpan(name = name, type = type).record(code)
            } finally {
                Systrace.end()
            }
        } else {
            InternalStaticEmbraceLogger.logWarning(
                "Logging span '$name' failed: service not in a state to log. Lambda will still run."
            )
            code()
        }
    }

    override fun recordCompletedSpan(
        name: String,
        startTimeNanos: Long,
        endTimeNanos: Long,
        type: EmbraceAttributes.Type,
        attributes: Map<String, String>,
        events: List<EventData>,
        errorCode: ErrorCode?
    ): Boolean {
        if (startTimeNanos > endTimeNanos) {
            InternalStaticEmbraceLogger.logWarning(
                "Logging completed span '$name' failed: start time is after end time"
            )
            return false
        }

        return if (currentSessionSpan.isRecording) {
            InternalStaticEmbraceLogger.logDebug("Logging completed span '$name'")
            Systrace.trace("log-completed-span-$name") {
                val span = createKeySpan(name = name, type = type)
                    .setStartTimestamp(startTimeNanos, TimeUnit.NANOSECONDS)
                    .startSpan()

                attributes.forEach {
                    span.setAttribute(it.key, it.value)
                }

                events.forEach {
                    span.addEvent(it.name, it.attributes, it.epochNanos, TimeUnit.NANOSECONDS)
                }

                span.endSpan(errorCode, endTimeNanos)
            }
            true
        } else {
            InternalStaticEmbraceLogger.logWarning(
                "Logging completed span '$name' failed: service not in a state to log"
            )
            false
        }
    }

    override fun storeCompletedSpans(spans: List<SpanData>): CompletableResultCode {
        try {
            synchronized(completedSpans) {
                completedSpans += spans.map { EmbraceSpanData(spanData = it) }
            }
        } catch (t: Throwable) {
            return CompletableResultCode.ofFailure()
        }

        return CompletableResultCode.ofSuccess()
    }

    override fun completedSpans(): List<EmbraceSpanData> {
        synchronized(completedSpans) {
            return completedSpans.toList()
        }
    }

    override fun flushSpans(appTerminationCause: EmbraceAttributes.AppTerminationCause?): List<EmbraceSpanData> {
        synchronized(completedSpans) {
            currentSessionSpan.endSpan()
            if (appTerminationCause == null) {
                currentSessionSpan = createSessionSpan(TimeUnit.MILLISECONDS.toNanos(clock.now()))
            } else {
                processRootSpan.setAttribute(appTerminationCause.keyName(), appTerminationCause.name)
                processRootSpan.endSpan()
            }

            val flushedSpans = completedSpans.toList()
            completedSpans.clear()
            return flushedSpans
        }
    }

    private fun createSessionSpan(startTimeNanos: Long): Span =
        createEmbraceSpanBuilder(name = "session-span", type = EmbraceAttributes.Type.SESSION)
            .setParent(Context.current().with(processRootSpan))
            .setStartTimestamp(startTimeNanos, TimeUnit.NANOSECONDS)
            .startSpan()

    private fun createKeySpan(name: String, type: EmbraceAttributes.Type): SpanBuilder =
        createEmbraceSpanBuilder(name = name, type = type)
            .setParent(Context.current().with(currentSessionSpan))
            .makeKey()

    private fun createEmbraceSpanBuilder(name: String, type: EmbraceAttributes.Type): SpanBuilder {
        val spanBuilder = if (type.internal) tracer.embraceSpanBuilder(name) else tracer.spanBuilder(name)
        return spanBuilder.setType(type)
    }
}
