package io.embrace.android.embracesdk.network;

import android.net.Uri;
import android.webkit.URLUtil;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.EnumSet;
import java.util.Locale;

import io.embrace.android.embracesdk.logging.InternalStaticEmbraceLogger;
import io.embrace.android.embracesdk.network.http.HttpMethod;

/**
 * This class is used to create manually-recorded network requests.
 */
public class EmbraceNetworkRequest {

    private static final EnumSet<HttpMethod> allowedMethods = EnumSet.of(
        HttpMethod.GET,
        HttpMethod.PUT,
        HttpMethod.POST,
        HttpMethod.DELETE,
        HttpMethod.PATCH);

    /**
     * The request's URL. Must start with http:// or https://
     */
    private final String url;
    /**
     * The request's method. Must be one of the following: GET, PUT, POST, DELETE, PATCH.
     */
    private final HttpMethod httpMethod;
    /**
     * The time the request started.
     */
    private final Long startTime;
    /**
     * The time the request ended. Must be greater than the startTime.
     */
    private final Long endTime;
    /**
     * The number of bytes received.
     */
    private final Long bytesReceived;
    /**
     * The number of bytes sent.
     */
    private final Long bytesSent;
    /**
     * The response status of the request. Must be in the range 100 to 599.
     */
    private final Integer responseCode;
    /**
     * Error that describes a non-HTTP error, e.g. a connection error.
     */
    private final Throwable error;
    /**
     * Optional trace ID that can be used to trace a particular request. Max length is 64 characters.
     */
    private final String traceId;

    EmbraceNetworkRequest(Builder builder) {
        this.url = builder.urlString;
        this.httpMethod = builder.httpMethod;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.bytesReceived = builder.bytesReceived;
        this.bytesSent = builder.bytesSent;
        this.responseCode = builder.responseCode;
        this.error = builder.error;
        this.traceId = builder.traceId;
    }

    static boolean validateMethod(HttpMethod method) {

        if (method == null) {
            InternalStaticEmbraceLogger.logError("Method cannot be null");
            return false;
        }

        if (!allowedMethods.contains(method)) {
            InternalStaticEmbraceLogger.logError("Not a valid method: " + method.name());
            return false;
        }

        return true;
    }

    @NonNull
    public static Builder newBuilder() {
        return new Builder();
    }

    @SuppressWarnings("MissingJavadocMethodCheck")
    public boolean canSend() {
        if (this.url == null) {
            InternalStaticEmbraceLogger.logError("Request must contain URL");
            return false;
        }
        if (this.httpMethod == null) {
            InternalStaticEmbraceLogger.logError("Request must contain method");
            return false;
        }
        if (this.startTime == null) {
            InternalStaticEmbraceLogger.logError("Request must contain startTime");
            return false;
        }
        if (this.endTime == null) {
            InternalStaticEmbraceLogger.logError("Request must contain endTime");
            return false;
        }
        if ((this.responseCode == null || this.responseCode == -1) && this.error == null) {
            InternalStaticEmbraceLogger.logError("Request must either have responseCode or error set");
            return false;
        }

        return true;
    }

    @NonNull
    public String getUrl() {
        return url;
    }

    @NonNull
    public String getHttpMethod() {
        return httpMethod != null ? httpMethod.name().toUpperCase() : null;
    }

    @NonNull
    public Long getStartTime() {
        return startTime;
    }

    @Nullable
    public Long getEndTime() {
        return endTime;
    }

    @NonNull
    public Long getBytesIn() {
        return bytesReceived == null ? 0 : bytesReceived;
    }

    @NonNull
    public Long getBytesOut() {
        return bytesSent == null ? 0 : bytesSent;
    }

    @Nullable
    public Integer getResponseCode() {
        return responseCode;
    }

    @Nullable
    public Throwable getError() {
        return error;
    }

    @Nullable
    public String getTraceId() {
        return traceId;
    }

    @NonNull
    @SuppressWarnings("MissingJavadocMethodCheck")
    public String description() {
        return "<" + this + ": " + this +
            " URL = " + this.url +
            " Method = " + this.httpMethod +
            " Start = " + this.startTime + ">";
    }

    @SuppressWarnings("MissingJavadocTypeCheck")
    public static final class Builder {
        String urlString;
        HttpMethod httpMethod;
        Long startTime;
        Long endTime;
        Long bytesReceived;
        Long bytesSent;
        Integer responseCode;
        Throwable error;
        String traceId;

        Builder() {

        }

        public void withUrl(@NonNull String urlString) {
            this.urlString = urlString;
        }

        public void withHttpMethod(@NonNull HttpMethod httpMethod) {
            this.httpMethod = httpMethod;
        }

        public void withStartTime(@NonNull Long startTime) {
            this.startTime = startTime;
        }

        @SuppressWarnings("MissingJavadocMethodCheck")
        public void withEndTime(@NonNull Long date) {
            if (date == null) {
                InternalStaticEmbraceLogger.logError("End time cannot be null");
                return;
            }

            if (date <= this.startTime) {
                InternalStaticEmbraceLogger.logError("End time cannot be before the start time");
                return;
            }

            this.endTime = date;
        }

        @Deprecated
        public void withByteIn(@NonNull Long byteReceived) {
            withBytesIn(byteReceived);
        }

        @SuppressWarnings("MissingJavadocMethodCheck")
        public void withBytesIn(@NonNull Long bytesReceived) {

            if (bytesReceived < 0) {
                InternalStaticEmbraceLogger.logError("bytesReceived must be a positive long");
                return;
            }

            this.bytesReceived = bytesReceived;
        }

        @Deprecated
        public void withByteOut(@NonNull Long bytesSent) {

            withBytesOut(bytesSent);
        }

        @SuppressWarnings("MissingJavadocMethodCheck")
        public void withBytesOut(@NonNull Long bytesSent) {

            if (bytesSent < 0) {
                InternalStaticEmbraceLogger.logError("BytesOut must be a positive long");
                return;
            }

            this.bytesSent = bytesSent;
        }

        @SuppressWarnings("MissingJavadocMethodCheck")
        public void withResponseCode(@NonNull Integer responseCode) {

            if (responseCode < 100 || responseCode > 599) {
                InternalStaticEmbraceLogger.logError(String.format(Locale.getDefault(), "Invalid responseCode: %d", responseCode));
                return;
            }

            this.responseCode = responseCode;
        }

        @SuppressWarnings("MissingJavadocMethodCheck")
        public void withError(@NonNull Throwable error) {

            if (error == null) {
                InternalStaticEmbraceLogger.logError("Ignoring null error");
                return;
            }

            this.error = error;
        }

        public void withTraceId(@NonNull String traceId) {
            this.traceId = traceId;
        }

        @NonNull
        @SuppressWarnings("MissingJavadocMethodCheck")
        public EmbraceNetworkRequest build() {

            Uri url;
            try {
                url = Uri.parse(urlString);
            } catch (NullPointerException ex) {
                InternalStaticEmbraceLogger.logError("Invalid URL: " + urlString);
                return null;
            }

            if (url == null || url.getScheme() == null || url.getHost() == null) {
                InternalStaticEmbraceLogger.logError("Invalid URL: " + urlString);
                return null;
            }

            if (!URLUtil.isHttpsUrl(urlString) && !URLUtil.isHttpUrl(urlString)) {
                InternalStaticEmbraceLogger.logError("Only http and https schemes are supported: " + urlString);
                return null;
            }

            if (!validateMethod(httpMethod)) {
                return null;
            }

            if (startTime == null) {
                InternalStaticEmbraceLogger.logError("Start time cannot be null");
                return null;
            }

            return new EmbraceNetworkRequest(this);
        }
    }
}
