package io.embrace.android.embracesdk.worker

import java.io.Closeable
import java.util.concurrent.Callable
import java.util.concurrent.ExecutorService
import java.util.concurrent.Future
import java.util.concurrent.RejectedExecutionException

/**
 * Background worker which executes work on one or more threads.
 */
internal class BackgroundWorker(private val executorService: ExecutorService) : Closeable {

    /**
     * Submits a task to be executed on another thread.
     *
     * @param task the task to be executed
     * @param <T>  the return type of the task
     * @return a future wrapping the result of the task
     </T> */
    fun <T> submit(task: Callable<T>?): Future<T> = executorService.submit(task)

    /**
     * Eagerly loads the value returned by a Lazy property on a background thread.
     *
     * When the Lazy property is accessed for the first time, the value is returned if it is already loaded, or the
     * main thread is blocked until it finishes loading.
     */
    fun <T> eagerLazyLoad(task: Callable<T>): Lazy<T> {
        val future = submitSafe(task)
        return lazy {
            if (future != null) {
                try {
                    return@lazy future.get()
                } catch (exc: Exception) {
                    return@lazy getCallableValue<T>(task)
                }
            }
            getCallableValue(task)
        }
    }

    fun <T> submitSafe(task: Callable<T>): Future<T>? {
        return try {
            executorService.submit(task)
        } catch (exc: RejectedExecutionException) {
            null
        }
    }

    private fun <T> getCallableValue(task: Callable<T>): T {
        return try {
            task.call()
        } catch (e: Exception) {
            throw IllegalStateException("Failed to load property.", e)
        }
    }

    /**
     * This should only be called if the worker is never expected to be used again. If a [BackgroundWorker] is not exclusively
     * being used or owned by a particular component (like the ones in [WorkerThreadModule]), it should not be shutdown when said component
     * is being shut down.
     */
    override fun close() = executorService.shutdown()
}
