package io.embrace.android.embracesdk.worker

import java.io.Closeable
import java.util.concurrent.ExecutorService
import java.util.concurrent.ScheduledExecutorService

/**
 * A set of shared executors and background workers to be used throughout the SDK
 */
internal interface WorkerThreadModule : Closeable {

    /**
     * Return the [ExecutorService] given the [WorkerName]
     */
    fun backgroundExecutor(workerName: WorkerName): ExecutorService

    /**
     * Return the [ScheduledExecutorService] given the [WorkerName]
     */
    fun scheduledExecutor(workerName: WorkerName): ScheduledExecutorService

    /**
     * Return the [BackgroundWorker] given the [WorkerName]
     */
    fun backgroundWorker(workerName: WorkerName): BackgroundWorker

    /**
     * This should only be invoked when the SDK is shutting down. Closing all the worker threads in production means the
     * SDK will not be functional afterwards.
     */
    override fun close()
}

/**
 * The key used to reference a specific shared [ExecutorService] or the [BackgroundWorker] that uses it
 */
internal enum class WorkerName(internal val threadName: String) {
    BACKGROUND_REGISTRATION("Background Registration Worker"),
    SCHEDULED_REGISTRATION("Scheduled Registration Worker"),
    CACHED_SESSIONS("Cached sessions"),
    SEND_SESSIONS("Send sessions"),
    DELIVERY_CACHE("Delivery cache sessions"),
    API_RETRY("API Retry Worker"),
    NATIVE_CRASH_CLEANER("Native Crash Cleaner"),
    NATIVE_STARTUP("Native Startup"),
    SESSION_CACHE_WORKER("Session cache worker"),
    REMOTE_LOGGING("Remote logging"),
    SESSION("Session"),
    SESSION_CLOSER("Session closer"),
    SESSION_CACHING("Session caching"),
}
