package eventcenter.remote.dubbo.publisher;

import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.URL;
import com.alibaba.dubbo.common.utils.ConfigUtils;
import com.alibaba.dubbo.common.utils.NetUtils;
import com.alibaba.dubbo.config.RegistryConfig;
import com.alibaba.dubbo.registry.NotifyListener;
import com.alibaba.dubbo.registry.RegistryService;
import eventcenter.remote.EventPublisher;
import eventcenter.remote.EventSubscriber;
import eventcenter.remote.publisher.PublisherGroup;
import eventcenter.remote.saf.EventForward;
import eventcenter.remote.saf.PublishGroupChangeable;
import eventcenter.remote.saf.StoreAndForwardPolicy;
import eventcenter.remote.utils.StringHelper;
import org.apache.log4j.Logger;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 使用zookeeper获取事件订阅者，这个需要依靠dubbo的{@link RegistryService}获取
 * @author JackyLIU
 *
 */
public class DubboRegistryEventPublisher implements EventPublisher, PublishGroupChangeable, ApplicationContextAware {
	
    public static final String REGISTRY_ADDRESS = "dubbo.registry.address";
    
    public static final String APPLICATION_NAME = "dubbo.application.name";
    
	/**
	 * 需要监听zk的dubbo的group名称，默认为所有的分组
	 */
	private String dubboGroup;
	
	@Autowired(required=false)
    private RegistryService registryService;
	
	private ApplicationContext applicationContext;

	SubcriberHolder subcriberHolder = new SubcriberHolder();
	
	protected List<PublisherGroup> localPublisherGroups = null;
	
	protected Map<String, List<URL>> serviceProviders = new ConcurrentHashMap<String, List<URL>>();

	private final Logger logger = Logger.getLogger(this.getClass());
	
	RegistryConfig registryConfig;

	/**
	 * dubbo使用的registryUrl，默认会使用dubbo.properties中定义的dubbo.registry.address
	 */
	private String registryUrl;

	public RegistryService getRegistryService() {
		return registryService;
	}

	public void setRegistryService(RegistryService registryService) {
		this.registryService = registryService;
		this.subcriberHolder.setRegistryConfig(this.registryConfig);
	}

	public boolean isCopySendUnderSameVersion() {
		return subcriberHolder.isCopySendUnderSameVersion();
	}

	public void setCopySendUnderSameVersion(boolean copySendUnderSameVersion) {
		this.subcriberHolder.setCopySendUnderSameVersion(copySendUnderSameVersion);
	}

	public boolean isDevMode() {
		return subcriberHolder.isDevMode();
	}

	public void setDevMode(boolean devMode) {
		subcriberHolder.setDevMode(devMode);
	}

	public String getDubboGroup() {
		return dubboGroup;
	}

	public void setDubboGroup(String dubboGroup) {
		this.dubboGroup = dubboGroup;
	}

	@Override
	public void startup() {		
		if(null == registryService && applicationContext != null){
			registryService = RegistryServiceFactory.buildDefault(applicationContext);
		}

		// 加载applicationConfig和registryConfig
		if(applicationContext != null) {
			if(null == registryConfig){
				try {
					registryConfig = applicationContext.getBean(RegistryConfig.class);
				} catch (NoSuchBeanDefinitionException e) {
				}
			}
		}

		if(subcriberHolder.getLocalSubscriberId() == null && applicationContext != null){
			try{
				EventSubscriber localEventSubscriber = applicationContext.getBean(EventSubscriber.class);
				subcriberHolder.setLocalSubscriberId(localEventSubscriber.getId());
			}catch(Exception e){
				
			}
		}
		
		if(dubboGroup == null || dubboGroup.length() == 0) {
			throw new IllegalArgumentException("Please set dubboGroup property");
		}else{
			subcriberHolder.setDubboGroup(dubboGroup);
		}

		initRegistryService();
	}

	/**
	 * init registry service and subscriber notify
	 */
	void initRegistryService(){
		if(registryConfig == null) {
			String url = StringHelper.isNotEmpty(registryUrl) ? registryUrl : ConfigUtils.getProperty(REGISTRY_ADDRESS);
			if (url == null || url.length() == 0) {
				throw new IllegalArgumentException("Please set java start argument: -D" + REGISTRY_ADDRESS + "=zookeeper://127.0.0.1:2181");
			}

			registryConfig = new RegistryConfig();
			registryConfig.setAddress(url);
			registryConfig.setRegister(false);
		}

		subcriberHolder.setRegistryConfig(registryConfig);

		URL eventSubscriberUrl = new URL(Constants.ADMIN_PROTOCOL, NetUtils.getLocalHost(), 0, "",
				Constants.INTERFACE_KEY, EventSubscriber.class.getName(),
				Constants.GROUP_KEY, dubboGroup,
				Constants.VERSION_KEY, Constants.ANY_VALUE,
				Constants.CLASSIFIER_KEY, Constants.ANY_VALUE,
				Constants.CATEGORY_KEY, Constants.PROVIDERS_CATEGORY,
				Constants.CHECK_KEY, String.valueOf(false));
		registryService.subscribe(eventSubscriberUrl, createNotifyListener());
		logger.info("startup dubbo registry success, group:" + dubboGroup);
	}
	
	NotifyListener createNotifyListener(){
		return new NotifyListener() {
        	@Override
        	public void notify(List<URL> urls) {
        		_notify(urls);
        	}
        };
	}
	
	protected void _notify(List<URL> urls){
		if (urls == null || urls.size() == 0) {
            return;
        }        		

        Map<String, List<URL>> providerMap = new HashMap<String, List<URL>>();
        for (URL url : urls) {
        	String category = url.getParameter(Constants.CATEGORY_KEY, Constants.DEFAULT_CATEGORY);
        	if (!Constants.PROVIDERS_CATEGORY.equals(category)) {
				continue;
			}
        	
        	String service = url.getServiceInterface();
        	if(!EventSubscriber.class.getName().equals(service)) {
				continue;
			}
        	
        	if (Constants.EMPTY_PROTOCOL.equals(url.getProtocol())) {
        		destroyAllSubscribers(service);
        		return ;
        	}
        	
        	List<URL> list = providerMap.get(service);
            if (list == null) {
                list = new ArrayList<URL>();
                providerMap.put(service, list);
            }
            list.add(url);
        }
        
        List<URL> offlineProviders = findOfflineProviders(providerMap.get(EventSubscriber.class.getName()));
        serviceProviders.putAll(providerMap);
        if(offlineProviders != null && offlineProviders.size() > 0){
        	// 销毁离线的服务端
        	subcriberHolder.destroySubscribers(offlineProviders);
        }
        loadSubscriber(providerMap);
	}
	
	/**
	 * 找出离线的接口提供者
	 * @param urls
	 * @return
	 */
	protected List<URL> findOfflineProviders(List<URL> urls){
		if(!serviceProviders.containsKey(EventSubscriber.class.getName())){
			return null;
		}
		
		List<URL> localUrls = serviceProviders.get(EventSubscriber.class.getName());
		Map<String, URL> remoteMap = toMap(urls);
		List<URL> offlines = new ArrayList<URL>();
		for(URL localUrl : localUrls){
			if(!remoteMap.containsKey(subcriberHolder.createAddress(localUrl))){
				offlines.add(localUrl);
			}
		}
		return offlines;
	}
	
	private Map<String, URL> toMap(List<URL> urls){
		Map<String, URL> map = new HashMap<String, URL>(urls.size());
		for(URL url : urls){
			map.put(subcriberHolder.createAddress(url), url);
		}
		return map;
	}
	
	@Override
	public void shutdown() {
		subcriberHolder.shutdown();
	}
	
	/**
	 * 加载远程订阅者
	 * @param providerMap
	 */
	protected void loadSubscriber(Map<String, List<URL>> providerMap){
		if(providerMap.containsKey(EventSubscriber.class.getName())){
			loadEventSubscriber(providerMap.get(EventSubscriber.class.getName()), null);
		}
	}
	
	protected void loadEventSubscriber(List<URL> urls, List<String> remoteEventsList){
		if(null == urls || urls.size() == 0) {
			return ;
		}

		for(int i = 0;i < urls.size();i++){
			URL url = urls.get(i);
			/*String offlineEvents = null;
			if(null != remoteEventsList && remoteEventsList.size() > 0){
				offlineEvents = remoteEventsList.get(i);
			}*/

			try{
				subcriberHolder.loadEventSubscriber(url);
			}catch(Exception e){
				logger.error(e.getMessage(), e);
			}
		}
	}
	
	protected void destroyAllSubscribers(String service){
		if(serviceProviders == null || serviceProviders.size() == 0) {
			return ;
		}
		
		List<URL> urls = serviceProviders.get(service);
		subcriberHolder.destroySubscribers(urls);
	}

	@Override
	public void setApplicationContext(ApplicationContext applicationContext)
			throws BeansException {
		this.applicationContext = applicationContext;
	}

	/**
	 * 这个主要用于 PublishEventCenter 设置setPublisherGroups，这个方法只接受 LocalPublisherGroup
	 */
	@Override
	public void publish(List<PublisherGroup> groups) {
		if(null == localPublisherGroups){
			localPublisherGroups = new ArrayList<PublisherGroup>();
		}
		
		localPublisherGroups.addAll(groups);
	}

	@Override
	public List<PublisherGroup> getPublisherGroups() {
		List<PublisherGroup> groups = new ArrayList<PublisherGroup>(subcriberHolder.getPublisherGroups().values());
		if(localPublisherGroups != null && localPublisherGroups.size() > 0){
			groups.addAll(localPublisherGroups);
		}
		
		return groups;
	}

	public void setExpiryOffline(int expiryOffline) {
		this.subcriberHolder.setExpiryOffline(expiryOffline);
	}

	@Override
	public int getExpiryOffline() {
		return this.subcriberHolder.getExpiryOffline();
	}
	
	@Override
	public void setForwardAndStorePolicy(EventForward eventForward,
			StoreAndForwardPolicy policy) {
		this.subcriberHolder.setEventForward(eventForward);
		this.subcriberHolder.setStoreAndForwardPolicy(policy);
	}

	public String getRegistryUrl() {
		return registryUrl;
	}

	public void setRegistryUrl(String registryUrl) {
		this.registryUrl = registryUrl;
	}

	/**
	 * 将参数中的registryConfig拷贝到类中的registryConfig中
	 * @param config
	 */
	public void copy2RegistryConfig(RegistryConfig config){
		if(null == registryConfig){
			registryConfig = new RegistryConfig();
		}
		if(StringHelper.isNotEmpty(config.getAddress())){
			registryConfig.setAddress(config.getAddress());
		}
		if(StringHelper.isNotEmpty(config.getClient())){
			registryConfig.setClient(config.getClient());
		}
		if(StringHelper.isNotEmpty(config.getCluster())){
			registryConfig.setCluster(config.getCluster());
		}
		if(StringHelper.isNotEmpty(config.getFile())){
			registryConfig.setFile(config.getFile());
		}
		if(StringHelper.isNotEmpty(config.getGroup())){
			registryConfig.setGroup(config.getGroup());
		}
		if(StringHelper.isNotEmpty(config.getPassword())){
			registryConfig.setPassword(config.getPassword());
		}
		if(StringHelper.isNotEmpty(config.getProtocol())){
			registryConfig.setProtocol(config.getProtocol());
		}
		if(StringHelper.isNotEmpty(config.getServer())){
			registryConfig.setServer(config.getServer());
		}
		if(StringHelper.isNotEmpty(config.getTransporter())){
			registryConfig.setTransporter(config.getTransporter());
		}
		if(StringHelper.isNotEmpty(config.getUsername())){
			registryConfig.setUsername(config.getUsername());
		}
		if(StringHelper.isNotEmpty(config.getVersion())){
			registryConfig.setVersion(config.getVersion());
		}
		if(StringHelper.isNotEmpty(config.getId())){
			registryConfig.setId(config.getId());
		}
		if(null != config.getParameters()){
			registryConfig.setParameters(config.getParameters());
		}
		if(null != config.getPort()){
			registryConfig.setPort(config.getPort());
		}
		if(null != config.getSession()){
			registryConfig.setSession(config.getSession());
		}
		if(null != config.getTimeout()){
			registryConfig.setTimeout(config.getTimeout());
		}
	}

	public String getLocalSubscriberId() {
		return subcriberHolder.getLocalSubscriberId();
	}

	public void setLocalSubscriberId(String localSubscriberId) {
		subcriberHolder.setLocalSubscriberId(localSubscriberId);
	}
}
