package eventcenter.remote.dubbo.publisher;

import com.alibaba.dubbo.common.URL;
import com.alibaba.dubbo.common.utils.NetUtils;
import com.alibaba.dubbo.common.utils.StringUtils;
import com.alibaba.dubbo.config.RegistryConfig;
import eventcenter.remote.EventSubscriber;
import eventcenter.remote.SubscriberGroup;
import eventcenter.remote.dubbo.subscriber.watchdog.SubscriberAware;
import eventcenter.remote.dubbo.subscriber.watchdog.SubscriberWatchDog;
import eventcenter.remote.publisher.PublisherGroup;
import eventcenter.remote.saf.EventForward;
import eventcenter.remote.saf.StoreAndForwardPolicy;
import eventcenter.remote.utils.ExpiryMap;
import eventcenter.remote.utils.StringHelper;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 事件订阅端holder，控制发布端管理订阅端的dubbo ref的控制器
 * @author liumingjian
 * @date 2018/9/3
 **/
class SubcriberHolder {

    static final String LOCAL_IP1 = "127.0.0.1";

    static final String LOCAL_IP2 = "localhost";

    EventForward eventForward;

    StoreAndForwardPolicy storeAndForwardPolicy;

    /**
     * 远程注册的事件，如果在这个subscribEvents找不到，那么需要从serviceProviders获取，key为host:port,
     */
    // TODO 这里的缓存应该有个时间控制，比如半个小时失效一次，然后再从remoteSubscriberFactories中获取
    Map<String, String> subscribEvents = new ConcurrentHashMap<String, String>();

    /**
     * 远程订阅工厂，key为host:port
     */
    Map<String, RemoteSubscriberFactory> remoteSubscriberFactories = new ConcurrentHashMap<String, RemoteSubscriberFactory>();

    Map<String, EventSubscriber> eventSubscribers = new ConcurrentHashMap<String, EventSubscriber>();

    Map<String, PublisherGroupFactory> publisherGroupFactories = new ConcurrentHashMap<String, PublisherGroupFactory>();

    Map<String, Date> monitorMap = new ConcurrentHashMap<String, Date>();

    /**
     * 当订阅端离线时，给定一个有效期，如果有效期过了，则删除掉离线队列
     */
    ExpiryMap<String, URL> garbagePool;

    Map<String, PublisherGroup> publisherGroups = new ConcurrentHashMap<String, PublisherGroup>();

    /**
     * 如果远程dubbo的版本号和分组信息都是一致时，并且存在多个点时，事件将会远程发布到每一个点钟。如果为false，那么默认使用
     * dubbo自己的负载方式进行访问
     */
    private boolean copySendUnderSameVersion = false;

    /**
     * 默认有效期为1天
     */
    private int expiryOffline = 24*3600;

    private Object locker = new Object();

    /**
     * 本地订阅者的序列号
     */
    private String localSubscriberId;

    /**
     * 本地的IP+端口
     */
    private String localAddress;

    /**
     * 是否开启开发模式，当开启后，事件将只发布到当前机器的订阅者
     */
    private boolean devMode = false;

    private SubscriberWatchDog subscriberWatchDog;

    /**
     * 设置subscriberWatchDog中的expiry过期时间，单位毫秒
     */
    private Long subscriberWatchDogExpiry;

    /**
     * 设置subscriberWatchDog中的watchInterval的监控间隔时间，单位毫秒
     */
    private Long subscriberWatchInterval;

    private String dubboGroup;

    private RegistryConfig registryConfig;

    private final Logger logger = Logger.getLogger(this.getClass());

    public void shutdown(){
        if(needUseGarbagePool() && garbagePool != null){
            garbagePool.shutdown();
        }
        if(this.subscriberWatchDog != null){
            this.subscriberWatchDog.shutdown();
        }
    }

    public Map<String, PublisherGroup> getPublisherGroups() {
        return publisherGroups;
    }

    public RegistryConfig getRegistryConfig() {
        return registryConfig;
    }

    public void setRegistryConfig(RegistryConfig registryConfig) {
        this.registryConfig = registryConfig;
    }

    public String getDubboGroup() {
        return dubboGroup;
    }

    public void setDubboGroup(String dubboGroup) {
        this.dubboGroup = dubboGroup;
    }

    public boolean isDevMode() {
        return devMode;
    }

    public void setDevMode(boolean devMode) {
        this.devMode = devMode;
    }

    public String getLocalSubscriberId() {
        return localSubscriberId;
    }

    public void setLocalSubscriberId(String localSubscriberId) {
        this.localSubscriberId = localSubscriberId;
    }

    public void setExpiryOffline(int expiryOffline) {
        this.expiryOffline = expiryOffline;
    }

    public int getExpiryOffline() {
        return expiryOffline;
    }

    public void setEventForward(EventForward eventForward) {
        this.eventForward = eventForward;
    }

    public void setStoreAndForwardPolicy(StoreAndForwardPolicy storeAndForwardPolicy) {
        this.storeAndForwardPolicy = storeAndForwardPolicy;
    }

    public Map<String, String> getSubscribEvents() {
        return subscribEvents;
    }

    public Map<String, RemoteSubscriberFactory> getRemoteSubscriberFactories() {
        return remoteSubscriberFactories;
    }

    public Map<String, EventSubscriber> getEventSubscribers() {
        return eventSubscribers;
    }

    public Map<String, PublisherGroupFactory> getPublisherGroupFactories() {
        return publisherGroupFactories;
    }

    public Map<String, Date> getMonitorMap() {
        return monitorMap;
    }

    public boolean isCopySendUnderSameVersion() {
        return copySendUnderSameVersion;
    }

    public void setCopySendUnderSameVersion(boolean copySendUnderSameVersion) {
        this.copySendUnderSameVersion = copySendUnderSameVersion;
    }

    ExpiryMap<String, URL> getGarbagePool(){
        if(null != garbagePool){
            return garbagePool;
        }
        synchronized (locker){
            if(null != garbagePool){
                return garbagePool;
            }
            garbagePool = createGarbagePool();
            garbagePool.startup();
            garbagePool.setExpiriedCallback(new ExpiryMap.ExpiriedCallback<String, URL>() {
                @Override
                public void onExpiried(String key, URL value) {

                    removeSubscriber(key, value);
                }
            });
            return garbagePool;
        }
    }

    ExpiryMap<String, URL> createGarbagePool(){
        return new ExpiryMap<String, URL>();
    }

    /**
     * 判断是否需要使用回收池
     * @return
     */
    boolean needUseGarbagePool(){
        return null != eventForward;
    }

    /**
     * 是否可以忽略初始化订阅者实例
     * @return
     */
    boolean canIgnoreInstanceSubscriber(String address){
        return needUseGarbagePool() && getGarbagePool().containKey(address);
    }

    EventSubscriber createEventSubscriber(String address, URL url, RegistryConfig registryConfig){
        RemoteSubscriberFactory factory = getRemoteSubscriberFactories().get(address);
        if (null == factory) {
            factory = createRemoteSubscriberFactory(url, address, registryConfig);
            getRemoteSubscriberFactories().put(address, factory);
        }

        if(canIgnoreInstanceSubscriber(address)){
            return null;
        }

        return factory.createEventSubscriber();
    }

    void loadEventSubscriber(URL url){
        String address = createAddress(url);

        if(StringHelper.equals(address, localAddress)){
            return ;
        }

        if(!filterWithDevMode(url)){
            return ;
        }

        EventSubscriber subscriber = createEventSubscriber(address, url, registryConfig);

        if(null == subscriber){
            return ;
        }

        SubscriberInitStatus subscriberInitStatus = initEventSubscirber(dubboGroup, address, url, subscriber);
        if(SubscriberInitStatus.SUCCESS != subscriberInitStatus){
            if(SubscriberInitStatus.NO_NEED == subscriberInitStatus && this.getRemoteSubscriberFactories().containsKey(address)){
                this.getRemoteSubscriberFactories().get(address).destroy();
                this.getRemoteSubscriberFactories().remove(address);
            }
            return ;
        }

        loadAndCreatePublisherGroup(url, address, registryConfig);
    }

    protected RemoteSubscriberFactory createRemoteSubscriberFactory(URL url, String address, RegistryConfig registryConfig){
        RemoteSubscriberFactory factory = RemoteSubscriberFactory.buildWith(url, registryConfig);
        if(devMode || isCopySendUnderSameVersion()){
            if(isCopySendUnderSameVersion()){
                factory.getReferenceConfig().setUrl(buildDubboUrl(address));
            }
            else{
                factory.getReferenceConfig().setUrl(buildDubboUrl(new StringBuilder(url.getHost()).append(":").append(url.getPort()).toString()));
            }
        }
        return factory;
    }

    String buildDubboUrl(String address){
        return new StringBuilder("dubbo://").append(address).toString();
    }

    /**
     * 初始化eventSubscriber
     * @param dubboGroup
     * @param address
     * @param url
     * @param subscriber
     * @return 如果返回true，则后续的步骤可以继续执行，反之则不做后面的操作
     */
    protected SubscriberInitStatus initEventSubscirber(String dubboGroup, String address, URL url, EventSubscriber subscriber){

        try {
            // 过滤本地的订阅器
            if (null != localSubscriberId && localSubscriberId.equals(subscriber.getId())) {
                // 这个表示当前加载的订阅器是本地启动的，这个将会被忽略
                localAddress = address;
                return SubscriberInitStatus.NO_NEED;
            }

        }catch(Exception e){
            logger.error(e.getMessage(), e);
        }

        try {
            eventSubscribers.put(address, subscriber);
            SubscriberGroup group = subscriber.getSubscriberGroup(dubboGroup);
            if (null == group) {
                logger.warn(new StringBuilder(url.toString()).append(" can't find group events!"));
                return SubscriberInitStatus.NO_NEED;
            }
            final String remoteEvents = group.getRemoteEvents();
            if(StringUtils.isEmpty(remoteEvents)){
                logger.warn(new StringBuilder(url.toFullString()).append(" remote events is empty, ignore subscriber"));
                return SubscriberInitStatus.NO_NEED;
            }

            subscribEvents.put(address, remoteEvents);

            removeFromGarbagePool(address);
            return SubscriberInitStatus.SUCCESS;
        }catch(Exception e){
            logger.error("load subscriber group failure:" + e.getMessage(), e);
            if(!subscribEvents.containsKey(address)){
                SubscriberWatchDog subscriberWatchDog = getSubscriberWatchDog();
                if(!subscriberWatchDog.containsSubscriber(subscriber)){
                    subscriberWatchDog.addSubscriber(subscriber, url);
                }
            }
            return SubscriberInitStatus.FAILURE;
        }
    }

    /**
     * 加载eventSubscriber同时，创建出PublosherGroup
     * @param url
     * @param address
     * @param registryConfig
     */
    protected PublisherGroup loadAndCreatePublisherGroup(URL url, String address, RegistryConfig registryConfig) {

        PublisherGroup publisherGroup = loadPublisherGroup(url, registryConfig);

        if(needUseGarbagePool() && publisherGroup != null){
            try {
                addMonitor(address, publisherGroup);
            } catch (IOException e) {
                logger.error("add event forward monitor failure:" + e.getMessage(), e);
            }
        }

        if(null != publisherGroup) {
            publisherGroups.put(address, publisherGroup);
            if(logger.isDebugEnabled()){
                logger.debug(new StringBuilder("load url:").append(url).append(" success. remote events:").append(publisherGroup.getRemoteEvents()));
            }
        }

        return publisherGroup;
    }

    PublisherGroup loadPublisherGroup(URL url, RegistryConfig registryConfig){
        String address = createAddress(url);

        PublisherGroupFactory factory = getPublisherGroupFactories().get(address);
        if (null == factory) {
            factory = createPublisherGroupFactory(url, address, registryConfig);
            getPublisherGroupFactories().put(address, factory);
        } else {
            if (needUpdatePublisherGroupFactory(factory, address)) {
                updatePublisherGroupFactory(factory, address);
            }
        }

        try{
            PublisherGroup group = factory.createPublisherGroup();
            group.setRemoteUrl(url.getAddress());
            group.setGroupName(buildGroupName(url));
            return group;
        }catch(Exception e){
            logger.error(e.getMessage(), e);
        }

        return null;
    }

    /**
     * 构建publishGroup的groupName，格式为group^application
     * @param url
     * @return
     */
    String buildGroupName(URL url){
        return new StringBuilder(url.getParameter("group")).append("_").append(url.getParameter("application")).toString();
    }

    void destroySubscribers(List<URL> urls){
        for(URL url : urls){
            destroySubscriber(url);
        }
    }

    /**
     * 销毁远程订阅者
     * @param url
     */
    void destroySubscriber(URL url){
        String address = createAddress(url);

        // 放入到回收池
        if(needUseGarbagePool()){
            if(!getGarbagePool().containKey(address)){
                if(!filterWithDevMode(url)){
                    return ;
                }
                if(null != localAddress && localAddress.equals(address)){
                    return ;
                }
                getGarbagePool().put(address, expiryOffline * 1000, url);
                if(logger.isTraceEnabled()){
                    logger.trace("发现消费点离线了，url:" + new StringBuilder(url.getHost()).append(":").append(url.getPort()).append(",version:").append(url.getParameter("version,离线队列失效时间")).append(expiryOffline).append(" s."));
                }
            }

            return ;
        }

        removeSubscriber(address, url);
    }

    boolean filterWithDevMode(URL url){
        if(!devMode){
            return true;
        }
        final String remoteAddress = new StringBuilder(url.getHost()).append(":").append(url.getPort()).toString();

        if(remoteAddress.contains(LOCAL_IP1) || remoteAddress.contains(LOCAL_IP2)) {
            return true;
        }
        return remoteAddress.contains(NetUtils.getLocalHost());
    }

    public Long getSubscriberWatchDogExpiry() {
        return subscriberWatchDogExpiry;
    }

    public void setSubscriberWatchDogExpiry(Long subscriberWatchDogExpiry) {
        this.subscriberWatchDogExpiry = subscriberWatchDogExpiry;
    }

    public Long getSubscriberWatchInterval() {
        return subscriberWatchInterval;
    }

    public void setSubscriberWatchInterval(Long subscriberWatchInterval) {
        this.subscriberWatchInterval = subscriberWatchInterval;
    }

    void updatePublisherGroupFactory(PublisherGroupFactory factory, String address) {
        factory.setRemoteEvents(getSubscribEvents().get(address));
    }

    boolean needUpdatePublisherGroupFactory(PublisherGroupFactory factory, String address) {
        return !StringUtils.isEquals(factory.getRemoteEvents(), getSubscribEvents().get(address));
    }

    String createAddress(URL url){
        if(copySendUnderSameVersion) {
            return new StringBuilder(url.getHost()).append(":").append(url.getPort()).toString();
        }

        return url.getParameter("version");
    }

    protected PublisherGroupFactory createPublisherGroupFactory(URL url, String address, RegistryConfig registryConfig){
        if(copySendUnderSameVersion){
            return PublisherGroupFactory.buildWith(url, subscribEvents.get(address));
        }else{
            return PublisherGroupFactory.buildWith(url, subscribEvents.get(address), registryConfig);
        }
    }

    void addMonitor(String address, PublisherGroup publisherGroup) throws IOException {
        if(monitorMap.containsKey(address)){
            if(logger.isTraceEnabled()) {
                logger.trace(new StringBuilder(address).append(" had been add monitor, can't add again!"));
            }
            return ;
        }
        eventForward.addMonitor(publisherGroup, storeAndForwardPolicy.createEventQueue(address));
        monitorMap.put(address, new Date());
        if(logger.isTraceEnabled()){
            logger.trace(new StringBuilder("add monitor for event forward:").append(address));
        }
    }

    void removeSubscriber(String address, URL url){

        PublisherGroup group = publisherGroups.get(address);

        if(needUseGarbagePool() && group != null){
            boolean health = false;
            try{
                // 先检测下是否健康
                health = group.getEventTransmission().checkHealth();
            }catch(Exception e){
                logger.error(new StringBuilder("check health failure before remove subscriber, reason:").append(e.getMessage()).append(", url:").append(url));
            }
            if(health){
                logger.info(new StringBuilder("recovery connection, subscriber won't be remove:").append(url));
                return ;
            }
            try {
                monitorMap.remove(address);
                eventForward.removeMonitor(group);
                if(logger.isTraceEnabled()){
                    logger.trace(new StringBuilder("remove monitor for event forward:").append(address));
                }
            } catch (IOException e) {
                logger.error(e.getMessage(), e);
            }
        }

        removeEventSubscriber(address);

        removePublisherGroup(address, url);
    }

    /**
     * 移除掉Holder引用的EventSubscriber相关的内部对象，并销毁掉
     * @param address
     */
    void removeEventSubscriber(String address){
        if(eventSubscribers.containsKey(address)){
            EventSubscriber eventSubscriber = eventSubscribers.remove(address);
            if(null != subscriberWatchDog && getSubscriberWatchDog().containsSubscriber(eventSubscriber)){
                getSubscriberWatchDog().removeSubscriber(eventSubscriber);
            }
        }
        if(subscribEvents.containsKey(address)){
            subscribEvents.remove(address);
        }
        if(remoteSubscriberFactories.containsKey(address)){
            try{
                remoteSubscriberFactories.get(address).destroy();
                remoteSubscriberFactories.remove(address);
            }catch(Exception e){
                logger.error(e.getMessage(), e);
            }
        }
    }

    void removePublisherGroup(String address, URL url){
        if(publisherGroups.containsKey(address)){
            publisherGroups.remove(address);
        }
        if(publisherGroupFactories.containsKey(address)){
            try{
                publisherGroupFactories.get(address).destroy();
                publisherGroupFactories.remove(address);
            }catch(Exception e){
                logger.error(e.getMessage(), e);
            }
            if(logger.isDebugEnabled()){
                logger.debug(new StringBuilder("destroy url:").append(url).append(" success"));
            }
        }
    }

    void removeFromGarbagePool(String address){
        if(needUseGarbagePool() && getGarbagePool().containKey(address)){
            getGarbagePool().remove(address);
            if(logger.isDebugEnabled()){
                logger.debug(new StringBuilder("remove garbagePool:").append(address));
            }
        }
    }

    SubscriberWatchDog getSubscriberWatchDog(){
        if(null != subscriberWatchDog){
            return initSubscriberWatchDog();
        }
        synchronized (locker){
            if(subscriberWatchDog != null){
                return subscriberWatchDog;
            }
            subscriberWatchDog = new SubscriberWatchDog(new InnerSubscriberAware(), dubboGroup);
            if(null != subscriberWatchDogExpiry){
                subscriberWatchDog.setExpiry(subscriberWatchDogExpiry);
            }
            if(null != subscriberWatchInterval){
                subscriberWatchDog.setWatchInterval(subscriberWatchInterval);
            }
            return initSubscriberWatchDog();
        }
    }

    SubscriberWatchDog initSubscriberWatchDog(){
        if(subscriberWatchDog.isOpen()){
            return subscriberWatchDog;
        }
        synchronized (locker){
            if(subscriberWatchDog.isOpen()){
                return subscriberWatchDog;
            }
            subscriberWatchDog.start();
        }
        return subscriberWatchDog;
    }

    class InnerSubscriberAware implements SubscriberAware {

        @Override
        public void checkRemoteEventsSuccess(EventSubscriber eventSubscriber, URL url, SubscriberGroup subscriberGroup) {
            final String address = findAddress(eventSubscriber);
            if(subscribEvents.containsKey(address)){
                return ;
            }

            final String remoteEvents = subscriberGroup.getRemoteEvents();
            if(StringUtils.isEmpty(remoteEvents)){
                return ;
            }
            subscribEvents.put(address, remoteEvents);

            loadAndCreatePublisherGroup(url, address, registryConfig);
        }

        @Override
        public void expiried(EventSubscriber eventSubscriber, URL url) {
            final String address = findAddress(eventSubscriber);
            // 如果holder已经加载了，则忽略
            if(subscribEvents.containsKey(address)){
                return ;
            }

            removeEventSubscriber(address);
        }

        String findAddress(EventSubscriber eventSubscriber){
            Set<Map.Entry<String, EventSubscriber>> entries = eventSubscribers.entrySet();
            for(Map.Entry<String, EventSubscriber> entry : entries){
                if(entry.getValue() != null && entry.getValue() == eventSubscriber){
                    return entry.getKey();
                }
            }
            return null;
        }
    }

    /**
     * 初始化订阅者的状态
     */
    enum SubscriberInitStatus {

        /**
         * 创建成功
         */
        SUCCESS,

        /**
         * 创建失败，例如连接订阅端时报错了
         */
        FAILURE,

        /**
         * 不需要创建，例如本地订阅端订阅了本地事件时
         */
        NO_NEED
    }

}
