package eventcenter.remote.dubbo.subscriber.watchdog;

import com.alibaba.dubbo.common.URL;
import eventcenter.remote.EventSubscriber;
import eventcenter.remote.SubscriberGroup;
import eventcenter.remote.dubbo.publisher.DubboRegistryEventPublisher;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.log4j.Logger;

import java.util.*;

/**
 * 当使用{@link DubboRegistryEventPublisher}初始化动态订阅的EventSubscriber时，如果调用getRemoteEvent失败，则需要塞入到WatchDog中，定时检查是否可以成功调用EventSubscriber，如果一直都不行，则抛弃掉
 * @author liumingjian
 * @date 2018/9/17
 **/
public class SubscriberWatchDog extends Thread{

    /**
     * 默认有效期为1小时
     */
    private long expiry = 3600*1000;

    /**
     * 检测事件订阅端的健康状态，默认为10秒
     */
    private long watchInterval = 10 * 1000;

    private final SubscriberAware subscriberAware;

    private final String group;

    private Object lock = new Object();

    private volatile boolean open = false;

    private final Logger logger = Logger.getLogger(this.getClass());

    private Map<EventSubscriber, SubscriberInfo> subscribers = Collections.synchronizedMap(new HashMap<EventSubscriber, SubscriberInfo>());

    public SubscriberWatchDog(SubscriberAware subscriberAware, String group){
        super("ec-subscriber-watch-dog");
        this.subscriberAware = subscriberAware;
        this.group = group;
    }

    public boolean isOpen(){
        return open;
    }

    public long getExpiry() {
        return expiry;
    }

    public void setExpiry(long expiry) {
        this.expiry = expiry;
    }

    public long getWatchInterval() {
        return watchInterval;
    }

    public void setWatchInterval(long watchInterval) {
        this.watchInterval = watchInterval;
    }

    public boolean containsSubscriber(EventSubscriber eventSubscriber){
        return subscribers.containsKey(eventSubscriber);
    }

    public boolean removeSubscriber(EventSubscriber eventSubscriber){
        return subscribers.remove(eventSubscriber) != null;
    }

    public int sizeOfSubscribers(){
        return subscribers.size();
    }

    public void addSubscriber(EventSubscriber eventSubscriber, URL url){
        subscribers.put(eventSubscriber, new SubscriberInfo(eventSubscriber, url));
    }

    @Override
    public void start(){
        if(open){
            return ;
        }
        open = true;
        super.start();
    }

    @Override
    public void run() {
        try {
            logger.info("open subscriber watch dog success");
            while (open) {
                if (sizeOfSubscribers() == 0) {
                    sleep();
                    continue;
                }

                iterateAndCheck();
                sleep();
            }
            logger.info("shutdown subscriber watch dog success");
        }catch (InterruptedException e){
            logger.error(e.getMessage(), e);
        }
    }

    void iterateAndCheck(){
        if(logger.isTraceEnabled()){
            logger.trace("start to check, size of subscribers:" + subscribers.size());
        }
        final Collection<SubscriberInfo> infos = subscribers.values();
        final int listSize = infos.size() > 16?infos.size():infos.size();
        final List<SubscriberInfo> deleteInfos = new ArrayList<SubscriberInfo>(listSize);
        for(SubscriberInfo info : infos){
            SubscriberGroup subscriberGroup = checkSubscriberInfo(info);
            if(null != subscriberGroup){
                try{
                    subscriberAware.checkRemoteEventsSuccess(info.getEventSubscriber(), info.getUrl(), subscriberGroup);
                    if(logger.isDebugEnabled()){
                        logger.debug("check subscriber success[" + subscriberGroup + "]");
                    }
                }catch(Exception e){
                    logger.error(e.getMessage(), e);
                }
                deleteInfos.add(info);
                continue;
            }
            if(info.isExpiry()){
                deleteInfos.add(info);
                if(logger.isDebugEnabled()){
                    logger.debug("check subscriber expiry, it would remove it!");
                }
            }
        }
        if(deleteInfos.size() == 0){
            return ;
        }

        // 删除deleteInfos中的SubscriberInfo
        for(SubscriberInfo deletedInfo : deleteInfos){
            subscribers.remove(deletedInfo.getEventSubscriber());
            subscriberAware.expiried(deletedInfo.getEventSubscriber(), deletedInfo.getUrl());
        }
    }

    SubscriberGroup checkSubscriberInfo(SubscriberInfo info){
        try{
            return info.getSubscriberGroup(group);
        }catch(Exception e){
            logger.error(e.getMessage(), e);
            return null;
        }
    }

    void sleep() throws InterruptedException {
        synchronized (lock){
            lock.wait(watchInterval);
        }
    }

    /**
     * notify all
     */
    void wakeUp(){
        synchronized (lock){
            lock.notifyAll();
        }
    }

    public void shutdown(){
        if(!open){
            return ;
        }
        open = false;
        subscribers.clear();
        wakeUp();
    }

    class SubscriberInfo {
        final EventSubscriber eventSubscriber;

        final URL url;

        final Date created = new Date();

        final Date expiryDate = DateUtils.addMilliseconds(created, (int) expiry);

        SubscriberInfo(EventSubscriber eventSubscriber, URL url){
            this.eventSubscriber = eventSubscriber;
            this.url = url;
        }

        /**
         * 判断队列是否过期
         * @return
         */
        boolean isExpiry(){
            return expiryDate.before(now());
        }

        SubscriberGroup getSubscriberGroup(String group){
            return this.eventSubscriber.getSubscriberGroup(group);
        }

        public EventSubscriber getEventSubscriber() {
            return eventSubscriber;
        }

        public URL getUrl() {
            return url;
        }

        public Date getCreated() {
            return created;
        }

        Date now(){
            return new Date();
        }
    }
}
