/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.maven.docker.service;

import io.fabric8.maven.docker.access.DockerAccess;
import io.fabric8.maven.docker.access.DockerAccessException;
import io.fabric8.maven.docker.config.ImageConfiguration;
import io.fabric8.maven.docker.config.RunImageConfiguration;
import io.fabric8.maven.docker.config.WaitConfiguration;
import io.fabric8.maven.docker.log.DefaultLogCallback;
import io.fabric8.maven.docker.log.LogDispatcher;
import io.fabric8.maven.docker.log.LogOutputSpec;
import io.fabric8.maven.docker.model.Container;
import io.fabric8.maven.docker.service.QueryService;
import io.fabric8.maven.docker.util.Logger;
import io.fabric8.maven.docker.wait.ExitCodeChecker;
import io.fabric8.maven.docker.wait.HealthCheckChecker;
import io.fabric8.maven.docker.wait.HttpPingChecker;
import io.fabric8.maven.docker.wait.LogWaitChecker;
import io.fabric8.maven.docker.wait.PreconditionFailedException;
import io.fabric8.maven.docker.wait.TcpPortChecker;
import io.fabric8.maven.docker.wait.WaitChecker;
import io.fabric8.maven.docker.wait.WaitTimeoutException;
import io.fabric8.maven.docker.wait.WaitUtil;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import org.apache.commons.text.StrSubstitutor;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.shared.utils.StringUtils;

public class WaitService {
    private final QueryService queryService;
    private DockerAccess dockerAccess;
    private Logger log;

    public WaitService(DockerAccess dockerAccess, QueryService queryService, Logger log) {
        this.dockerAccess = dockerAccess;
        this.log = log;
        this.queryService = queryService;
    }

    public void wait(ImageConfiguration imageConfig, Properties projectProperties, String containerId) throws MojoExecutionException {
        List<WaitChecker> checkers = this.prepareWaitCheckers(imageConfig, projectProperties, containerId);
        int timeout = this.getTimeOut(imageConfig);
        if (checkers.isEmpty()) {
            if (timeout > 0) {
                this.log.info("%s: Pausing for %d ms", imageConfig.getDescription(), timeout);
                WaitUtil.sleep(timeout);
            }
            return;
        }
        String logLine = this.extractCheckerLog(checkers);
        ContainerRunningPrecondition precondition = new ContainerRunningPrecondition(this.dockerAccess, containerId);
        try {
            long waited = WaitUtil.wait((WaitUtil.Precondition)precondition, timeout, checkers);
            this.log.info("%s: Waited %s %d ms", imageConfig.getDescription(), logLine, waited);
        }
        catch (WaitTimeoutException exp) {
            String desc = String.format("%s: Timeout after %d ms while waiting %s", imageConfig.getDescription(), exp.getWaited(), logLine);
            this.log.error(desc, new Object[0]);
            throw new MojoExecutionException(desc);
        }
        catch (PreconditionFailedException exp) {
            String desc = String.format("%s: Container stopped with exit code %d unexpectedly after %d ms while waiting %s", imageConfig.getDescription(), precondition.getExitCode(), exp.getWaited(), logLine);
            this.log.error(desc, new Object[0]);
            throw new MojoExecutionException(desc);
        }
    }

    private int getTimeOut(ImageConfiguration imageConfig) {
        WaitConfiguration wait = this.getWaitConfiguration(imageConfig);
        return wait != null ? wait.getTime() : 0;
    }

    private String extractCheckerLog(List<WaitChecker> checkers) {
        ArrayList<String> logOut = new ArrayList<String>();
        for (WaitChecker checker : checkers) {
            logOut.add(checker.getLogLabel());
        }
        return StringUtils.join((Object[])logOut.toArray(), (String)" and ");
    }

    private List<WaitChecker> prepareWaitCheckers(ImageConfiguration imageConfig, Properties projectProperties, String containerId) throws MojoExecutionException {
        WaitConfiguration wait = this.getWaitConfiguration(imageConfig);
        if (wait == null) {
            return Collections.emptyList();
        }
        ArrayList<WaitChecker> checkers = new ArrayList<WaitChecker>();
        if (wait.getUrl() != null) {
            checkers.add(this.getUrlWaitChecker(imageConfig.getDescription(), projectProperties, wait));
        }
        if (wait.getLog() != null) {
            this.log.debug("LogWaitChecker: Waiting on %s", wait.getLog());
            checkers.add(new LogWaitChecker(wait.getLog(), this.dockerAccess, containerId, this.log));
        }
        if (wait.getTcp() != null) {
            try {
                Container container = this.queryService.getMandatoryContainer(containerId);
                checkers.add(this.getTcpWaitChecker(container, imageConfig.getDescription(), projectProperties, wait.getTcp()));
            }
            catch (DockerAccessException e) {
                throw new MojoExecutionException("Unable to access container.", (Exception)e);
            }
        }
        if (wait.getHealthy()) {
            checkers.add(new HealthCheckChecker(this.dockerAccess, containerId, imageConfig.getDescription(), this.log));
        }
        if (wait.getExit() != null) {
            checkers.add(new ExitCodeChecker(wait.getExit(), this.queryService, containerId));
        }
        return checkers;
    }

    private WaitConfiguration getWaitConfiguration(ImageConfiguration imageConfig) {
        RunImageConfiguration runConfig = imageConfig.getRunConfiguration();
        return runConfig.getWaitConfiguration();
    }

    private WaitChecker getUrlWaitChecker(String imageConfigDesc, Properties projectProperties, WaitConfiguration wait) {
        HttpPingChecker checker;
        String waitUrl = StrSubstitutor.replace((Object)wait.getUrl(), (Properties)projectProperties);
        WaitConfiguration.HttpConfiguration httpConfig = wait.getHttp();
        if (httpConfig != null) {
            checker = new HttpPingChecker(waitUrl, httpConfig.getMethod(), httpConfig.getStatus(), httpConfig.isAllowAllHosts());
            this.log.info("%s: Waiting on url %s with method %s for status %s.", imageConfigDesc, waitUrl, httpConfig.getMethod(), httpConfig.getStatus());
        } else {
            checker = new HttpPingChecker(waitUrl);
            this.log.info("%s: Waiting on url %s.", imageConfigDesc, waitUrl);
        }
        return checker;
    }

    private WaitChecker getTcpWaitChecker(Container container, String imageConfigDesc, Properties projectProperties, WaitConfiguration.TcpConfiguration tcpConfig) throws MojoExecutionException {
        ArrayList<Integer> ports = new ArrayList();
        List<Integer> portsConfigured = this.getTcpPorts(tcpConfig);
        String host = this.getTcpHost(tcpConfig, projectProperties);
        WaitConfiguration.TcpConfigMode mode = this.getTcpMode(tcpConfig, host);
        if (mode == WaitConfiguration.TcpConfigMode.mapped) {
            for (int port : portsConfigured) {
                Container.PortBinding binding = container.getPortBindings().get(port + "/tcp");
                if (binding == null) {
                    throw new MojoExecutionException(String.format("Cannot watch on port %d, since there is no network binding", port));
                }
                ports.add(binding.getHostPort());
            }
            this.log.info("%s: Waiting for mapped ports %s on host %s", imageConfigDesc, ports, host);
        } else {
            host = container.getIPAddress();
            ports = portsConfigured;
            this.log.info("%s: Waiting for ports %s directly on container with IP (%s).", imageConfigDesc, ports, host);
        }
        return new TcpPortChecker(host, ports);
    }

    private List<Integer> getTcpPorts(WaitConfiguration.TcpConfiguration tcpConfig) throws MojoExecutionException {
        List<Integer> portsConfigured = tcpConfig.getPorts();
        if (portsConfigured == null || portsConfigured.isEmpty()) {
            throw new MojoExecutionException("TCP wait config given but no ports to wait on");
        }
        return portsConfigured;
    }

    private WaitConfiguration.TcpConfigMode getTcpMode(WaitConfiguration.TcpConfiguration tcpConfig, String host) {
        WaitConfiguration.TcpConfigMode mode = tcpConfig.getMode();
        if (mode == null) {
            return "localhost".equals(host) ? WaitConfiguration.TcpConfigMode.direct : WaitConfiguration.TcpConfigMode.mapped;
        }
        return mode;
    }

    private String getTcpHost(WaitConfiguration.TcpConfiguration tcpConfig, Properties projectProperties) {
        String host = tcpConfig.getHost();
        if (host == null) {
            host = projectProperties.getProperty("docker.host.address");
        }
        return host;
    }

    private class ContainerRunningPrecondition
    implements WaitUtil.Precondition {
        private final String containerId;
        private final DockerAccess dockerAccess;
        private Integer exitCode;

        ContainerRunningPrecondition(DockerAccess dockerAccess, String containerId) {
            this.dockerAccess = dockerAccess;
            this.containerId = containerId;
        }

        @Override
        public boolean isOk() {
            try {
                this.exitCode = this.dockerAccess.getContainer(this.containerId).getExitCode();
                return this.exitCode == null;
            }
            catch (DockerAccessException e) {
                return false;
            }
        }

        @Override
        public void cleanup() {
            if (this.exitCode != null && WaitService.this.log.isVerboseEnabled()) {
                new LogDispatcher(this.dockerAccess).fetchContainerLog(this.containerId, LogOutputSpec.DEFAULT);
                this.dockerAccess.getLogSync(this.containerId, new DefaultLogCallback(new LogOutputSpec.Builder().color("black", true).prefix(this.containerId.substring(0, 6)).useColor(true).logStdout(true).build()));
            }
        }

        Integer getExitCode() {
            return this.exitCode;
        }
    }
}

