
/**
 * Copyright (C) 2015 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

                  

package io.fabric8.kubernetes.client.handlers.core.v1;

import java.util.function.Predicate;

import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.ResourceHandler;
import io.fabric8.kubernetes.client.Watch;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.dsl.internal.core.v1.BindingOperationsImpl;

import io.fabric8.kubernetes.client.dsl.base.OperationContext;
import okhttp3.OkHttpClient;

import io.fabric8.kubernetes.api.model.DeletionPropagation;
import io.fabric8.kubernetes.api.model.Binding;
import io.fabric8.kubernetes.api.model.BindingBuilder;

import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

public class BindingHandler implements ResourceHandler<Binding, BindingBuilder> {

  @Override
  public String getKind() {
    return Binding.class.getSimpleName();
  }

  @Override
  public String getApiVersion() {
        return "v1";
      }

  @Override
  public Binding create(OkHttpClient client, Config config, String namespace, Binding item) {
    return new BindingOperationsImpl(client, config).withItem(item).inNamespace(namespace).create();
  }

  @Override
  public Binding replace(OkHttpClient client, Config config, String namespace, Binding item) {
    return new BindingOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).replace(item);
  }

  @Override
  public Binding reload(OkHttpClient client, Config config, String namespace, Binding item) {
    return new BindingOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).fromServer().get();
  }

  @Override
  public BindingBuilder edit(Binding item) {
    return new BindingBuilder(item);
  }

  @Override
  public Boolean delete(OkHttpClient client, Config config, String namespace, DeletionPropagation propagationPolicy, Binding item) {
    return new BindingOperationsImpl(client, config, namespace).withItem(item).withPropagationPolicy(propagationPolicy).delete();
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, Binding item, Watcher<Binding> watcher) {
    return new BindingOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(watcher);
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, Binding item, String resourceVersion, Watcher<Binding> watcher) {
    return new BindingOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(resourceVersion, watcher);
  }

  @Override
  public Binding waitUntilReady(OkHttpClient client, Config config, String namespace, Binding item, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new BindingOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilReady(amount, timeUnit);
  }

  @Override
  public Binding waitUntilCondition(OkHttpClient client, Config config, String namespace, Binding item, Predicate<Binding> condition, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new BindingOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilCondition(condition, amount, timeUnit);
  }
}
