
/**
 * Copyright (C) 2015 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

                                                

package io.fabric8.kubernetes.client.handlers.batch.v1;

import java.util.function.Predicate;

import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.ResourceHandler;
import io.fabric8.kubernetes.client.Watch;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.dsl.internal.batch.v1.JobOperationsImpl;

import io.fabric8.kubernetes.client.dsl.base.OperationContext;
import okhttp3.OkHttpClient;

import io.fabric8.kubernetes.api.model.DeletionPropagation;
import io.fabric8.kubernetes.api.model.batch.Job;
import io.fabric8.kubernetes.api.model.batch.JobBuilder;

import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

public class JobHandler implements ResourceHandler<Job, JobBuilder> {

  @Override
  public String getKind() {
    return Job.class.getSimpleName();
  }

  @Override
  public String getApiVersion() {
        return "batch/v1";
      }

  @Override
  public Job create(OkHttpClient client, Config config, String namespace, Job item) {
    return new JobOperationsImpl(client, config).withItem(item).inNamespace(namespace).create();
  }

  @Override
  public Job replace(OkHttpClient client, Config config, String namespace, Job item) {
    return new JobOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).replace(item);
  }

  @Override
  public Job reload(OkHttpClient client, Config config, String namespace, Job item) {
    return new JobOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).fromServer().get();
  }

  @Override
  public JobBuilder edit(Job item) {
    return new JobBuilder(item);
  }

  @Override
  public Boolean delete(OkHttpClient client, Config config, String namespace, DeletionPropagation propagationPolicy, Job item) {
    return new JobOperationsImpl(client, config, namespace).withItem(item).withPropagationPolicy(propagationPolicy).delete();
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, Job item, Watcher<Job> watcher) {
    return new JobOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(watcher);
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, Job item, String resourceVersion, Watcher<Job> watcher) {
    return new JobOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(resourceVersion, watcher);
  }

  @Override
  public Job waitUntilReady(OkHttpClient client, Config config, String namespace, Job item, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new JobOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilReady(amount, timeUnit);
  }

  @Override
  public Job waitUntilCondition(OkHttpClient client, Config config, String namespace, Job item, Predicate<Job> condition, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new JobOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilCondition(condition, amount, timeUnit);
  }
}
