/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.client.utils;

import com.fasterxml.jackson.core.JsonProcessingException;
import io.fabric8.kubernetes.api.model.NamedAuthInfo;
import io.fabric8.kubernetes.api.model.NamedContext;
import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.internal.KubeConfigUtils;
import io.fabric8.kubernetes.client.internal.SSLUtils;
import io.fabric8.kubernetes.client.utils.Serialization;
import io.fabric8.kubernetes.client.utils.URLUtils;
import io.fabric8.kubernetes.client.utils.Utils;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.spec.InvalidKeySpecException;
import java.util.Base64;
import java.util.Collections;
import java.util.Map;
import javax.net.ssl.KeyManager;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import okhttp3.FormBody;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OpenIDConnectionUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(OpenIDConnectionUtils.class);
    public static final String ID_TOKEN_KUBECONFIG = "id-token";
    public static final String ISSUER_KUBECONFIG = "idp-issuer-url";
    public static final String REFRESH_TOKEN_KUBECONFIG = "refresh-token";
    public static final String REFRESH_TOKEN_PARAM = "refresh_token";
    public static final String GRANT_TYPE_PARAM = "grant_type";
    public static final String CLIENT_ID_PARAM = "client_id";
    public static final String CLIENT_SECRET_PARAM = "client_secret";
    public static final String ID_TOKEN_PARAM = "id_token";
    public static final String ACCESS_TOKEN_PARAM = "access_token";
    public static final String CLIENT_ID_KUBECONFIG = "client-id";
    public static final String CLIENT_SECRET_KUBECONFIG = "client-secret";
    public static final String IDP_CERT_DATA = "idp-certificate-authority-data";
    public static final String TOKEN_ENDPOINT_PARAM = "token_endpoint";
    public static final String WELL_KNOWN_OPENID_CONFIGURATION = ".well-known/openid-configuration";
    public static final String GRANT_TYPE_REFRESH_TOKEN = "refresh_token";

    private OpenIDConnectionUtils() {
    }

    public static String resolveOIDCTokenFromAuthConfig(Map<String, String> currentAuthProviderConfig) {
        String accessToken = currentAuthProviderConfig.get(ID_TOKEN_KUBECONFIG);
        String issuer = currentAuthProviderConfig.get(ISSUER_KUBECONFIG);
        String clientId = currentAuthProviderConfig.get(CLIENT_ID_KUBECONFIG);
        String refreshToken = currentAuthProviderConfig.get(REFRESH_TOKEN_KUBECONFIG);
        String clientSecret = currentAuthProviderConfig.getOrDefault(CLIENT_SECRET_KUBECONFIG, "");
        String idpCert = currentAuthProviderConfig.get(IDP_CERT_DATA);
        if (OpenIDConnectionUtils.isTokenRefreshSupported(currentAuthProviderConfig)) {
            return OpenIDConnectionUtils.getOIDCProviderTokenEndpointAndRefreshToken(issuer, clientId, refreshToken, clientSecret, accessToken, idpCert);
        }
        return accessToken;
    }

    static String getOIDCProviderTokenEndpointAndRefreshToken(OkHttpClient client, Map<String, Object> wellKnownOpenIdConfiguration, String clientId, String refreshToken, String clientSecret, String accessToken, boolean shouldPersistUpdatedTokenInKubeConfig) {
        String oidcTokenEndpoint = OpenIDConnectionUtils.getParametersFromDiscoveryResponse(wellKnownOpenIdConfiguration, TOKEN_ENDPOINT_PARAM);
        try {
            String freshAccessToken = OpenIDConnectionUtils.refreshToken(client, oidcTokenEndpoint, clientId, refreshToken, clientSecret, shouldPersistUpdatedTokenInKubeConfig);
            if (freshAccessToken != null) {
                accessToken = freshAccessToken;
            }
        }
        catch (Exception e) {
            LOGGER.warn("Could not refresh OIDC token: {}", (Object)e.getMessage());
        }
        return accessToken;
    }

    static boolean isTokenRefreshSupported(Map<String, String> currentAuthProviderConfig) {
        return Utils.isNotNull(currentAuthProviderConfig.get(REFRESH_TOKEN_KUBECONFIG));
    }

    static String refreshToken(OkHttpClient client, String oidcTokenEndpoint, String clientId, String refreshToken, String clientSecret, boolean shouldPersistUpdatedTokenInKubeConfig) {
        try {
            Map<String, Object> response = OpenIDConnectionUtils.refreshOidcToken(client, clientId, refreshToken, clientSecret, oidcTokenEndpoint);
            if (!response.containsKey(ID_TOKEN_PARAM)) {
                LOGGER.warn("token response did not contain an id_token, either the scope \\\"openid\\\" wasn't requested upon login, or the provider doesn't support id_tokens as part of the refresh response.");
                return null;
            }
            if (shouldPersistUpdatedTokenInKubeConfig && !OpenIDConnectionUtils.persistKubeConfigWithUpdatedToken(response)) {
                LOGGER.warn("oidc: failure while persisting new tokens into KUBECONFIG");
            }
            return String.valueOf(response.get(ID_TOKEN_PARAM));
        }
        catch (IOException e) {
            LOGGER.warn("Failure in fetching refresh token: ", (Throwable)e);
            return null;
        }
    }

    static Map<String, Object> refreshOidcToken(OkHttpClient client, String clientId, String refreshToken, String clientSecret, String tokenURL) throws IOException {
        try (Response response = client.newCall(OpenIDConnectionUtils.getTokenRefreshHttpRequest(tokenURL, clientId, refreshToken, clientSecret)).execute();){
            if (response.body() != null) {
                String responseBody = response.body().string();
                if (response.isSuccessful()) {
                    Map<String, Object> map = OpenIDConnectionUtils.convertJsonStringToMap(responseBody);
                    return map;
                }
                LOGGER.warn("Response: {}", (Object)responseBody);
            }
        }
        return Collections.emptyMap();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    static Map<String, Object> getOIDCDiscoveryDocumentAsMap(OkHttpClient client, String issuer) {
        try (Response response = client.newCall(OpenIDConnectionUtils.getDiscoveryDocumentHttpRequest(issuer)).execute();){
            if (response.isSuccessful() && response.body() != null) {
                Map<String, Object> map = OpenIDConnectionUtils.convertJsonStringToMap(response.body().string());
                return map;
            }
            String responseBody = response.body() != null ? response.body().string() : null;
            LOGGER.warn("oidc: failed to query metadata endpoint: {} {}", (Object)response.code(), (Object)responseBody);
            return Collections.emptyMap();
        }
        catch (IOException e) {
            LOGGER.warn("Could not refresh OIDC token, failure in getting refresh URL", (Throwable)e);
        }
        return Collections.emptyMap();
    }

    static String getWellKnownUrlForOpenIDIssuer(String issuer) {
        return URLUtils.join(issuer, "/", WELL_KNOWN_OPENID_CONFIGURATION);
    }

    static String getParametersFromDiscoveryResponse(Map<String, Object> responseAsJson, String key) {
        if (responseAsJson.containsKey(key)) {
            return String.valueOf(responseAsJson.get(key));
        }
        LOGGER.warn("oidc: oidc: discovery object doesn't contain a {}", (Object)key);
        return "";
    }

    static boolean persistKubeConfigWithUpdatedToken(String kubeConfigPath, Map<String, Object> updatedAuthProviderConfig) throws IOException {
        io.fabric8.kubernetes.api.model.Config config = KubeConfigUtils.parseConfig(new File(kubeConfigPath));
        NamedContext currentNamedContext = KubeConfigUtils.getCurrentContext(config);
        if (currentNamedContext != null) {
            int currentUserIndex = KubeConfigUtils.getNamedUserIndexFromConfig(config, currentNamedContext.getContext().getUser());
            Map authProviderConfig = ((NamedAuthInfo)config.getUsers().get(currentUserIndex)).getUser().getAuthProvider().getConfig();
            authProviderConfig.put(ID_TOKEN_KUBECONFIG, String.valueOf(updatedAuthProviderConfig.get(ID_TOKEN_PARAM)));
            authProviderConfig.put(REFRESH_TOKEN_KUBECONFIG, String.valueOf(updatedAuthProviderConfig.get(ACCESS_TOKEN_PARAM)));
            ((NamedAuthInfo)config.getUsers().get(currentUserIndex)).getUser().getAuthProvider().setConfig(authProviderConfig);
            try {
                KubeConfigUtils.persistKubeConfigIntoFile(config, kubeConfigPath);
                return true;
            }
            catch (IOException exception) {
                LOGGER.warn("failed to write file {}", (Object)kubeConfigPath, (Object)exception);
            }
        }
        return false;
    }

    private static Map<String, Object> convertJsonStringToMap(String jsonString) throws JsonProcessingException {
        return (Map)Serialization.jsonMapper().readValue(jsonString, Map.class);
    }

    private static SSLContext getSSLContext(String idpCert) {
        SSLContext sslContext = null;
        if (idpCert != null) {
            String pemCert = new String(Base64.getDecoder().decode(idpCert));
            try {
                TrustManager[] trustManagers = SSLUtils.trustManagers(pemCert, null, false, null, null);
                KeyManager[] keyManagers = SSLUtils.keyManagers(pemCert, null, null, null, null, null, null, null);
                sslContext = SSLUtils.sslContext(keyManagers, trustManagers);
            }
            catch (IOException | KeyManagementException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException | InvalidKeySpecException e) {
                throw new RuntimeException("Could not import idp certificate", e);
            }
        }
        return sslContext;
    }

    private static OkHttpClient getOkHttpClient(SSLContext sslContext, String pemCert) {
        OkHttpClient.Builder clientBuilder = new OkHttpClient.Builder();
        if (sslContext != null) {
            clientBuilder.sslSocketFactory(sslContext.getSocketFactory(), OpenIDConnectionUtils.getTrustManagerForAllCerts(pemCert));
        }
        return clientBuilder.build();
    }

    private static X509TrustManager getTrustManagerForAllCerts(String pemCert) {
        X509TrustManager trustManager = null;
        try {
            TrustManager[] trustManagers = SSLUtils.trustManagers(pemCert, null, false, null, null);
            if (trustManagers != null && trustManagers.length == 1) {
                trustManager = (X509TrustManager)trustManagers[0];
            }
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            LOGGER.warn("Could not get trust manager");
        }
        return trustManager;
    }

    private static Request getTokenRefreshHttpRequest(String tokenEndpointUrl, String clientId, String refreshToken, String clientSecret) throws JsonProcessingException {
        HttpUrl.Builder httpUrlBuilder = HttpUrl.get((String)tokenEndpointUrl).newBuilder();
        RequestBody requestBody = OpenIDConnectionUtils.getRequestBodyContentForRefresh(clientId, refreshToken, clientSecret);
        Request.Builder requestBuilder = new Request.Builder().post(requestBody).url(httpUrlBuilder.build());
        String credentials = Base64.getEncoder().encodeToString((clientId + ':' + clientSecret).getBytes(StandardCharsets.UTF_8));
        requestBuilder.addHeader("Authorization", "Basic " + credentials);
        requestBuilder.addHeader("Content-Type", "application/x-www-form-urlencoded");
        return requestBuilder.build();
    }

    private static Request getDiscoveryDocumentHttpRequest(String issuer) throws MalformedURLException {
        URL wellKnown = new URL(OpenIDConnectionUtils.getWellKnownUrlForOpenIDIssuer(issuer));
        return new Request.Builder().url(wellKnown).build();
    }

    private static RequestBody getRequestBodyContentForRefresh(String clientId, String refreshToken, String clientSecret) {
        return new FormBody.Builder().add("refresh_token", refreshToken).add(GRANT_TYPE_PARAM, "refresh_token").add(CLIENT_ID_PARAM, clientId).add(CLIENT_SECRET_PARAM, clientSecret).build();
    }

    private static String getOIDCProviderTokenEndpointAndRefreshToken(String issuer, String clientId, String refreshToken, String clientSecret, String accessToken, String idpCert) {
        OkHttpClient okHttpClient = OpenIDConnectionUtils.getOkHttpClient(OpenIDConnectionUtils.getSSLContext(idpCert), idpCert);
        Map<String, Object> wellKnownOpenIdConfiguration = OpenIDConnectionUtils.getOIDCDiscoveryDocumentAsMap(okHttpClient, issuer);
        return OpenIDConnectionUtils.getOIDCProviderTokenEndpointAndRefreshToken(okHttpClient, wellKnownOpenIdConfiguration, clientId, refreshToken, clientSecret, accessToken, true);
    }

    private static boolean persistKubeConfigWithUpdatedToken(Map<String, Object> updatedAuthProviderConfig) throws IOException {
        return OpenIDConnectionUtils.persistKubeConfigWithUpdatedToken(Config.getKubeconfigFilename(), updatedAuthProviderConfig);
    }
}

