
/**
 * Copyright (C) 2015 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

                        

package io.fabric8.kubernetes.client.handlers;

import java.util.function.Predicate;

import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.ResourceHandler;
import io.fabric8.kubernetes.client.Watch;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.dsl.internal.SecretOperationsImpl;

import io.fabric8.kubernetes.client.dsl.base.OperationContext;
import okhttp3.OkHttpClient;

import io.fabric8.kubernetes.api.model.Secret;
import io.fabric8.kubernetes.api.model.SecretBuilder;

import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

public class SecretHandler implements ResourceHandler<Secret, SecretBuilder> {

  @Override
  public String getKind() {
    return Secret.class.getSimpleName();
  }

  @Override
  public String getApiVersion() {
        return "v1";
      }

  @Override
  public Secret create(OkHttpClient client, Config config, String namespace, Secret item) {
    return new SecretOperationsImpl(client, config).withItem(item).inNamespace(namespace).create();
  }

  @Override
  public Secret replace(OkHttpClient client, Config config, String namespace, Secret item) {
    return new SecretOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).replace(item);
  }

  @Override
  public Secret reload(OkHttpClient client, Config config, String namespace, Secret item) {
    return new SecretOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).fromServer().get();
  }

  @Override
  public SecretBuilder edit(Secret item) {
    return new SecretBuilder(item);
  }

  @Override
  public Boolean delete(OkHttpClient client, Config config, String namespace, Boolean cascading, Secret item) {
    return new SecretOperationsImpl(client, config, namespace).withItem(item).cascading(cascading).delete();
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, Secret item, Watcher<Secret> watcher) {
    return new SecretOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(watcher);
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, Secret item, String resourceVersion, Watcher<Secret> watcher) {
    return new SecretOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(resourceVersion, watcher);
  }

  @Override
  public Secret waitUntilReady(OkHttpClient client, Config config, String namespace, Secret item, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new SecretOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilReady(amount, timeUnit);
  }

  @Override
  public Secret waitUntilCondition(OkHttpClient client, Config config, String namespace, Secret item, Predicate<Secret> condition, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new SecretOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilCondition(condition, amount, timeUnit);
  }
}
