
/**
 * Copyright (C) 2015 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

                        

package io.fabric8.kubernetes.client.handlers;

import java.util.function.Predicate;

import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.ResourceHandler;
import io.fabric8.kubernetes.client.Watch;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.dsl.internal.NetworkPolicyOperationsImpl;

import io.fabric8.kubernetes.client.dsl.base.OperationContext;
import okhttp3.OkHttpClient;

import io.fabric8.kubernetes.api.model.networking.NetworkPolicy;
import io.fabric8.kubernetes.api.model.networking.NetworkPolicyBuilder;

import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

public class NetworkPolicyHandler implements ResourceHandler<NetworkPolicy, NetworkPolicyBuilder> {

  @Override
  public String getKind() {
    return NetworkPolicy.class.getSimpleName();
  }

  @Override
  public String getApiVersion() {
        return "networking.k8s.io/v1";
      }

  @Override
  public NetworkPolicy create(OkHttpClient client, Config config, String namespace, NetworkPolicy item) {
    return new NetworkPolicyOperationsImpl(client, config).withItem(item).inNamespace(namespace).create();
  }

  @Override
  public NetworkPolicy replace(OkHttpClient client, Config config, String namespace, NetworkPolicy item) {
    return new NetworkPolicyOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).replace(item);
  }

  @Override
  public NetworkPolicy reload(OkHttpClient client, Config config, String namespace, NetworkPolicy item) {
    return new NetworkPolicyOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).fromServer().get();
  }

  @Override
  public NetworkPolicyBuilder edit(NetworkPolicy item) {
    return new NetworkPolicyBuilder(item);
  }

  @Override
  public Boolean delete(OkHttpClient client, Config config, String namespace, Boolean cascading, NetworkPolicy item) {
    return new NetworkPolicyOperationsImpl(client, config, namespace).withItem(item).cascading(cascading).delete();
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, NetworkPolicy item, Watcher<NetworkPolicy> watcher) {
    return new NetworkPolicyOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(watcher);
  }

  @Override
  public Watch watch(OkHttpClient client, Config config, String namespace, NetworkPolicy item, String resourceVersion, Watcher<NetworkPolicy> watcher) {
    return new NetworkPolicyOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).watch(resourceVersion, watcher);
  }

  @Override
  public NetworkPolicy waitUntilReady(OkHttpClient client, Config config, String namespace, NetworkPolicy item, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new NetworkPolicyOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilReady(amount, timeUnit);
  }

  @Override
  public NetworkPolicy waitUntilCondition(OkHttpClient client, Config config, String namespace, NetworkPolicy item, Predicate<NetworkPolicy> condition, long amount, TimeUnit timeUnit) throws InterruptedException {
    return new NetworkPolicyOperationsImpl(client, config).withItem(item).inNamespace(namespace).withName(item.getMetadata().getName()).waitUntilCondition(condition, amount, timeUnit);
  }
}
