/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package io.floodplain.streams.remotejoin;

import io.floodplain.immutable.api.ImmutableMessage;
import io.floodplain.immutable.factory.ImmutableFactory;
import io.floodplain.replication.api.ReplicationMessage;
import io.floodplain.replication.api.ReplicationMessage.Operation;
import org.apache.kafka.streams.processor.api.Processor;
import org.apache.kafka.streams.processor.api.ProcessorContext;
import org.apache.kafka.streams.processor.api.Record;
import org.apache.kafka.streams.state.KeyValueStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;

public class ReduceReadProcessor implements Processor<String, ReplicationMessage,String, ReplicationMessage> {

    private static final Logger logger = LoggerFactory.getLogger(ReduceReadProcessor.class);

    private final String accumulatorStoreName;
    private final String inputStoreName;
    private final Function<ImmutableMessage, ImmutableMessage> initial;
    private final Optional<BiFunction<ImmutableMessage, ImmutableMessage, String>> keyExtractor;
    private KeyValueStore<String, ImmutableMessage> accumulatorStore;
    private KeyValueStore<String, ReplicationMessage> inputStore;
    private ProcessorContext<String, ReplicationMessage> context;

    public ReduceReadProcessor(String inputStoreName, String accumulatorStoreName, Function<ImmutableMessage, ImmutableMessage> initial, Optional<BiFunction<ImmutableMessage, ImmutableMessage, String>> keyExtractor) {
        this.accumulatorStoreName = accumulatorStoreName;
        this.inputStoreName = inputStoreName;
        this.initial = initial;
        this.keyExtractor = keyExtractor;
    }


    @SuppressWarnings("unchecked")
    @Override
    public void init(ProcessorContext context) {
        this.context = context;
        accumulatorStore = (KeyValueStore<String, ImmutableMessage>) context.getStateStore(accumulatorStoreName);
        inputStore = (KeyValueStore<String, ReplicationMessage>) context.getStateStore(inputStoreName);
    }


    @Override
    public void process(Record<String, ReplicationMessage> record) {
        ReplicationMessage inputValue = record.value();
        String key = record.key();
        ReplicationMessage stored = inputStore.get(key);
        String extracted;
        if (stored == null) {
            // no stored value, so must be upsert.
            if (inputValue == null || inputValue.operation() == Operation.DELETE) {
                logger.warn("Issue: Deleting (?) a message that isn't there. Is this bad? Dropping message. key: {} inputvalue: {}", key, inputValue);
                return;
            }
            extracted = keyExtractor.orElse((m, s) -> StoreStateProcessor.COMMONKEY)
                    .apply(inputValue.message(), inputValue.paramMessage().orElse(ImmutableFactory.empty()));
        } else {
            extracted = keyExtractor.orElse((m, s) -> StoreStateProcessor.COMMONKEY)
                    .apply(stored.message(), stored.paramMessage().orElse(ImmutableFactory.empty()));
        }
        ImmutableMessage storedAccumulator = this.accumulatorStore.get(extracted);
        ReplicationMessage value = inputValue;
        inputStore.put(key, inputValue);
        if (inputValue == null || inputValue.operation() == Operation.DELETE) {
            if (stored == null) {
                throw new RuntimeException("Issue: Deleting a message that isn't there. Is this bad?");
            }
            // delete
            ImmutableMessage param = storedAccumulator == null ? initial.apply(stored.message()) : storedAccumulator;
            value = stored.withOperation(Operation.DELETE).withParamMessage(param);
            inputStore.delete(key);
        } else {
            if (storedAccumulator == null) {
                storedAccumulator = initial.apply(inputValue.message());
            }
            value = value.withParamMessage(storedAccumulator);
            if (stored != null) {
                // already present, propagate old value first as delete
                context.forward(new Record<>(extracted,stored.withOperation(Operation.DELETE).withParamMessage(storedAccumulator != null ? storedAccumulator : initial.apply(inputValue.message())),record.timestamp()));
                storedAccumulator = this.accumulatorStore.get(extracted);
            }
            value = value.withParamMessage(storedAccumulator);
        }
        context.forward(new Record<>(extracted,value,record.timestamp()));

    }
}
