/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package io.floodplain.streams.remotejoin;

import io.floodplain.immutable.api.ImmutableMessage;
import io.floodplain.replication.api.ReplicationMessage;
import io.floodplain.replication.factory.ReplicationFactory;
import org.apache.kafka.streams.processor.api.Processor;
import org.apache.kafka.streams.processor.api.ProcessorContext;
import org.apache.kafka.streams.processor.api.Record;
import org.apache.kafka.streams.state.KeyValueStore;

import java.util.Optional;

public class StoreStateProcessor implements Processor<String, ReplicationMessage,String, ReplicationMessage> {


    private final String lookupStoreName;
    private KeyValueStore<String, ImmutableMessage> lookupStore;
    public static final String COMMONKEY = "singlerestore";
    private ProcessorContext<String, ReplicationMessage> context;

    public StoreStateProcessor(String lookupStoreName) {
        this.lookupStoreName = lookupStoreName;
    }

    @SuppressWarnings("unchecked")
    @Override
    public void init(ProcessorContext<String, ReplicationMessage> context) {
        this.context = context;
        this.lookupStore = (KeyValueStore<String, ImmutableMessage>) context.getStateStore(lookupStoreName);
    }

    @Override
    public void process(Record<String, ReplicationMessage> record) {
        String key = record.key();
        ReplicationMessage inputValue = record.value();
        Optional<ImmutableMessage> paramMessage = inputValue.paramMessage();

        // Use .get() here, I prefer to fail when that is missing//.get();
        if (paramMessage.isEmpty()) {
            throw new RuntimeException("In store state there should definitely be a secondary message");
        }
        ImmutableMessage storeMessage = paramMessage.get();
        lookupStore.put(key, storeMessage);
        context.forward(new Record<>(key, ReplicationFactory.standardMessage(storeMessage),record.timestamp()));
    }
}
