/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package io.floodplain.streams.remotejoin.ranged;

import io.floodplain.replication.api.ReplicationMessage;
import io.floodplain.replication.api.ReplicationMessage.Operation;
import io.floodplain.streams.api.CoreOperators;
import io.floodplain.streams.remotejoin.PreJoinProcessor;
import org.apache.kafka.streams.KeyValue;
import org.apache.kafka.streams.processor.api.Processor;
import org.apache.kafka.streams.processor.api.ProcessorContext;
import org.apache.kafka.streams.processor.api.Record;
import org.apache.kafka.streams.state.KeyValueIterator;
import org.apache.kafka.streams.state.KeyValueStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.function.BiFunction;

public class OneToManyGroupedProcessor implements Processor<String, ReplicationMessage,String, ReplicationMessage> {
    private static final Logger logger = LoggerFactory.getLogger(OneToManyGroupedProcessor.class);
    private final boolean debug;

    private final String storeName;
    private final String groupedStoreName;

    private final boolean optional;

    private KeyValueStore<String, ReplicationMessage> groupedLookupStore;
    private KeyValueStore<String, ReplicationMessage> lookupStore;
    private final BiFunction<ReplicationMessage, List<ReplicationMessage>, ReplicationMessage> joinFunction;
    private ProcessorContext<String, ReplicationMessage> context;

    public OneToManyGroupedProcessor(String storeName, String groupedStoreName, boolean optional, boolean debug) {
        this.storeName = storeName;
        this.groupedStoreName = groupedStoreName;
        this.optional = optional;
        this.joinFunction = CoreOperators.getListJoinFunctionToParam(false);
        this.debug = debug;
    }

    @Override
    public void init(ProcessorContext<String, ReplicationMessage> context) {
        this.context = context;
        this.lookupStore = context.getStateStore(storeName);
        this.groupedLookupStore = context.getStateStore(groupedStoreName);
    }


    @Override
    public void process(Record<String, ReplicationMessage> record) {
        String key = record.key();
        ReplicationMessage msg = record.value();
        boolean reverse = false;
        if (key.endsWith(PreJoinProcessor.REVERSE_IDENTIFIER)) {
            reverse = true;
            key = key.substring(0, key.length() - PreJoinProcessor.REVERSE_IDENTIFIER.length());
        }

        if (reverse) {
            reverseJoin(key, msg,record.timestamp());
        } else {
            if (msg == null) {
                logger.debug("O2M Emitting null message with key: {}", key);
                context.forward(new Record<>(key,null,record.timestamp()));
                return;
            }
            forwardJoin(key, msg,record.timestamp());
        }

    }

    private void forwardJoin(String key, ReplicationMessage msg, long timestamp) {
        List<ReplicationMessage> msgs = new ArrayList<>();
        try(KeyValueIterator<String, ReplicationMessage> it = groupedLookupStore.range(key + "|", key + "}")) {
            while (it.hasNext()) {
                KeyValue<String, ReplicationMessage> keyValue = it.next();
                msgs.add(keyValue.value);
            }
        }
        ReplicationMessage joined = msg;
        if (msgs.size() > 0 || optional) {
            joined = joinFunction.apply(msg, msgs);
        }
        if (optional || msgs.size() > 0) {
            forwardMessage(key, joined,timestamp);
        } else {
            // We are not optional, and have not joined with any messages. Forward a delete
            // -> TODO Improve this. It does not necesarily need a delete. If it is a new key, it can simply be ignored
            forwardMessage(key, joined.withOperation(Operation.DELETE),timestamp);
        }

    }

    private void reverseJoin(String key, ReplicationMessage msg, long timestamp) {

        String actualKey = CoreOperators.ungroupKeyReverse(key);
        ReplicationMessage one = lookupStore.get(actualKey);
        if (debug) {
            long storeSize = lookupStore.approximateNumEntries();
            logger.info("# of elements in reverse store: {}", storeSize);
        }
        if (one == null) {
            // We are doing a reverse join, but the original message isn't there.
            // Nothing to do for us here
            return;
        }
        // OneToMany, thus we need to find all the other messages that
        // we also have to join with us. Effectively the same as a
        // forward join.
        forwardJoin(actualKey, one,timestamp);
    }

    private void forwardMessage(String key, ReplicationMessage innerMessage, long timestamp) {
        context.forward(new Record<>(key, innerMessage,timestamp));
        // flush downstream stores with null:
        if (innerMessage.operation() == Operation.DELETE) {
            logger.debug("Delete forwarded, appending null forward with key: {}", key);
            context.forward(new Record<>(key, null,timestamp));
        }
    }


}
