/*
 * Copyright (c) 2016-2021 Flux Capacitor.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.fluxcapacitor.testserver.endpoints;

import io.fluxcapacitor.common.Awaitable;
import io.fluxcapacitor.common.Guarantee;
import io.fluxcapacitor.common.api.SerializedMessage;
import io.fluxcapacitor.common.api.VoidResult;
import io.fluxcapacitor.common.api.eventsourcing.AppendEvents;
import io.fluxcapacitor.common.api.eventsourcing.DeleteEvents;
import io.fluxcapacitor.common.api.eventsourcing.EventBatch;
import io.fluxcapacitor.common.api.eventsourcing.GetEvents;
import io.fluxcapacitor.common.api.eventsourcing.GetEventsResult;
import io.fluxcapacitor.common.api.modeling.GetAggregateIds;
import io.fluxcapacitor.common.api.modeling.GetAggregateIdsResult;
import io.fluxcapacitor.common.api.modeling.UpdateRelationships;
import io.fluxcapacitor.javaclient.persisting.eventsourcing.AggregateEventStream;
import io.fluxcapacitor.javaclient.persisting.eventsourcing.client.EventStoreClient;
import io.fluxcapacitor.testserver.Handle;
import io.fluxcapacitor.testserver.WebsocketEndpoint;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@AllArgsConstructor
public class EventSourcingEndpoint extends WebsocketEndpoint {

    private final EventStoreClient eventStore;

    @Handle
    public VoidResult handle(AppendEvents appendEvents) throws Exception {
        List<Awaitable> results = appendEvents.getEventBatches().stream().map(b -> eventStore
                .storeEvents(b.getAggregateId(), b.getEvents(),
                             b.isStoreOnly())).collect(
                Collectors.toList());
        for (Awaitable awaitable : results) {
            awaitable.await();
        }
        return new VoidResult(appendEvents.getRequestId());
    }

    @Handle
    public VoidResult handle(DeleteEvents deleteEvents) throws Exception {
        eventStore.deleteEvents(deleteEvents.getAggregateId()).awaitSilently();
        return new VoidResult(deleteEvents.getRequestId());
    }

    @Handle
    public GetEventsResult handle(GetEvents getEvents) throws Exception {
        AggregateEventStream<SerializedMessage> stream = eventStore
                .getEvents(getEvents.getAggregateId(), getEvents.getLastSequenceNumber());
        long lastSequenceNumber = stream.getLastSequenceNumber().orElse(-1L);
        return new GetEventsResult(getEvents.getRequestId(), new EventBatch(
                getEvents.getAggregateId(), stream.collect(Collectors.toList()), false), lastSequenceNumber);
    }

    @Handle
    public VoidResult handle(UpdateRelationships request) {
        Awaitable awaitable = eventStore.updateRelationships(request);
        if (request.getGuarantee().compareTo(Guarantee.STORED) >= 0) {
            awaitable.awaitSilently();
            return new VoidResult(request.getRequestId());
        }
        return null;
    }

    @Handle
    public GetAggregateIdsResult handle(GetAggregateIds request) {
        return new GetAggregateIdsResult(request.getRequestId(), eventStore.getAggregateIds(request));
    }
}
