package io.fullview.fullview_sdk.camera

import android.annotation.SuppressLint
import android.app.Application
import android.graphics.Bitmap
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import collector.models.CustomerEvent
import collector.models.EventType
import com.google.gson.Gson
import io.fullview.fullview_sdk.KoinApp
import io.fullview.fullview_sdk.R
import io.fullview.fullview_sdk.data.DisplayInfo
import io.fullview.fullview_sdk.helpers.ScreenshotHelpers
import io.reactivex.rxjava3.schedulers.Schedulers
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import network.models.RRWebMeta
import timber.log.Timber
import java.io.ByteArrayOutputStream
import java.util.Calendar
import java.util.TimeZone
import java.util.zip.Deflater

class CameraViewModel : ViewModel() {

    private val hubRepository = KoinApp.getHub()


    private val _flow: MutableStateFlow<Bitmap?> = MutableStateFlow(null)
    private val screenshotFlow: StateFlow<Bitmap?> = _flow.asStateFlow()
    lateinit var displayInfo: DisplayInfo

    private var snapshotTemplate : String? = null

    fun setTemplate(template: String) {
        snapshotTemplate = template
    }

    init {
        CoroutineScope(Dispatchers.Default).launch {
            screenshotFlow
                .filterNotNull()
                .distinctUntilChanged { old, new -> old.sameAs(new) }
                .collectLatest {
                    if (hubRepository.isCollectorConnected) {
                        sendMetaData {
                            sendSnapshot(ScreenshotHelpers.getEncodedScreenshot(it))
                        }
                    }
                }
        }
    }

    fun consumeScreenshot(bitmap: Bitmap) {
        if (hubRepository.isCollectorConnected) {
            _flow.update { bitmap }
        }
    }



    private fun getMetaData(height: Int, width: Int, href: String): String {
        val meta = RRWebMeta(
            4,
            RRWebMeta.RRWebData(
                href,
                width,
                height
            ),
            Calendar.getInstance(TimeZone.getTimeZone("UTC")).timeInMillis
        )
        val gson = Gson()
        return gson.toJson(meta)
    }

    private fun sendMetaData(callback: () -> Unit) {
        val metaData = getMetaData(
            displayInfo.correctedHeight.toInt(),
            displayInfo.correctedWidth.toInt(),
            ""
        )


        val deflater = Deflater(Deflater.BEST_COMPRESSION)
        deflater.setInput(metaData.toByteArray(Charsets.ISO_8859_1))
        deflater.finish()

        val outputStream = ByteArrayOutputStream()
        val buffer = ByteArray(1024)

        while (!deflater.finished()) {
            val compressedSize = deflater.deflate(buffer)
            outputStream.write(buffer, 0, compressedSize)
        }

        hubRepository.collectorConnection.invoke(
            "event", CustomerEvent(
                type = EventType.RRWebMeta,
                payload = outputStream.toByteArray().toString(Charsets.ISO_8859_1),
                timestamp = Calendar.getInstance(TimeZone.getTimeZone("UTC")).timeInMillis,
                content = null
            )
        )

        callback()
    }

    @SuppressLint("CheckResult")
    private fun sendSnapshot(imageData: String) {
        viewModelScope.launch(Dispatchers.IO) {

            if(snapshotTemplate == null) {
                return@launch
            }

            val timestamp = Calendar.getInstance(TimeZone.getTimeZone("UTC")).timeInMillis
            val payload = snapshotTemplate!!
                .replace("BASE64_JPG_REPLACE", imageData)
                .replace("TIMESTAMP_REPLACE", timestamp.toString())
                .replace(Regex("(\"(?=$timestamp)|(?<=$timestamp)\")"), "")
                .replace("\n", "")


            val deflater = Deflater(Deflater.BEST_COMPRESSION)
            deflater.setInput(payload.toByteArray(Charsets.ISO_8859_1))
            deflater.finish()

            val outputStream = ByteArrayOutputStream()
            val buffer = ByteArray(1024)

            while (!deflater.finished()) {
                val compressedSize = deflater.deflate(buffer)
                outputStream.write(buffer, 0, compressedSize)
            }

            if (hubRepository.isCollectorConnected) {
                hubRepository.collectorConnection.invoke(
                    "event", CustomerEvent(
                        type = EventType.RRWebFullSnapshot,
                        payload = outputStream.toByteArray().toString(Charsets.ISO_8859_1),
                        timestamp = Calendar.getInstance(TimeZone.getTimeZone("UTC")).timeInMillis,
                        content = null
                    )
                )
            }
        }
    }
}